package API::Authorization::User;

use Direct::Modern;

use List::Util qw/ any first max /;
use Mouse;
use Mouse::Util::TypeConstraints;

use API::ClientOptions;
use Primitives qw/query_all_api_special_options get_info_by_uid_passport/;
use User qw/get_user_data create_update_user/;
use Client qw/get_client_data/;

subtype 'ClientApiEnabled'
    => as 'Str'
    => where { API::ClientOptions::is_valid_api_enabled($_) };
=pod

=encoding utf8

    $Id$

=head1 NAME

    API::Authorization::User

=head1 SYNOPSIS

    my $user = API::Authorization::User->new( id => 123, login => 'hmepas' ); # формируем объект
    my $user = API::Authorization::User->new({ id => 123, login => 'hmepas' }); # формируем объект по ссылке на хэш
    my $user = API::Authorization::User->new(123); # будут автоматически получены данные пользователя с uid 123


=head1 DESCRIPTION

    Класс для работы с объектом пользователя в рамках авторизации

=head1 ассессоры

    uid
    login
    ClientID
    api_allowed_ips
    api_geo_allowed
    api_allow_old_versions
    api_units_daily
    api5_units_daily
    statusBlocked
    special_options
    rep_type

=head1 METHODS

=cut

sub _PARAMS {
    qw/
        uid
        login
        ClientID
        api_allowed_ips
        api_geo_allowed
        api_allow_old_versions
        api5_units_daily
        api_units_daily
        statusBlocked
        special_options
        rep_type
    /;
}

foreach my $p (_PARAMS()) {
    has $p => (
        is => 'ro',
        # required => 1
    );
}

=head2 units_limit

    Вычисляется от api_units_daily и api_units_daily

    api_units_daily может быть не заданым. Данное поле в БД испльзуется для ручного задания баллов, устанавливается через свойства API на клиента в интерфейса.

    api5_units_daily расчитывается по крону раз в сутки и для всех не новых пользовтаелей всегда задано. Также можно поменять в интерфейсе в настройках API на клиента. Будет перезаписано при следующем запуске считающем баллы для API5.

    В случае если api_units_daily задано, то, так как это ручная настройка, то она используется как для API4 так и для API5. В API5 выбирается большее значение между api5_units_daily и api_units_daily.

=cut

sub units_limit {
    my $self = shift;
    return max($self->api_units_daily || 0, $self->api5_units_daily || 0);
}

=head2 api_enabled

    Доступ в API: Yes | No | Default

=cut

has api_enabled => (is => 'ro', isa => 'ClientApiEnabled');

=head2 role

    Роль клиента (client, super, superreader, placer, manager, support, agency и т.д. )

=cut

has role => ( is => 'rw', isa => 'Str' );

=head2 is_client

    true если авторизованный пользотель - прямой клиент (не сотрудник яндекса и не агентство)

=cut

sub is_client { shift->role eq 'client' }


sub BUILDARGS {
    my $class = shift;

    # пришел сразу hashref параметров
    return $_[0] if ref $_[0] eq 'HASH';

    # пришел хэш параметров
    return { @_ } if $#_ > 0;

    # считаем что пришел uid
    my $uid = shift or die 'no uid specified';

    my $args = get_user_data( $uid, [ $class->_PARAMS ] );
    return {} unless $args && $args->{login};

    $args->{uid} = $uid;

    if ( any { $_ eq 'special_options' } $class->_PARAMS ) {
        $args->{special_options} = query_all_api_special_options($args->{ClientID});
    }

    my @api_opts_fields = qw/ api_enabled api5_units_daily /;
    my $api_opts = API::ClientOptions::get($args->{ClientID}, \@api_opts_fields);
    @$args{@api_opts_fields} = @$api_opts{@api_opts_fields};
    $args->{api_enabled} ||= 'Default';

    return $args;
}

=head2 id

    ID пользователя, тот что uid в БД

=cut

sub id { shift->uid }

=head2 client_id

    id клиента (ClientID в БД)

=cut

sub client_id { shift->ClientID }

=head2 allowed_concurrent_calls

    Максимально допустимое количество одновременных запросов в API

=cut

sub allowed_concurrent_calls { shift->special_options->{concurrent_calls} }

=head2 found

    false если объект пользователя -- пустой, по uid ничего не нашлось в БД

=cut

sub found { defined shift->{login} }

=head2 is_blocked

    забанен ли пользователь

=cut

sub is_blocked { (shift->{statusBlocked} // '') eq 'Yes' }

=head2 _client_data

    данные из таблиц clients, clients_options 

=cut

has _client_data => (
    is => 'ro',
    isa => 'Maybe[HashRef]',
    lazy => 1,
    default => sub {
        my $self = shift;
        my @fields = qw/is_ya_agency_client/;
        return get_client_data($self->ClientID, \@fields) // {};
    }
);

=head2 is_ya_agency_client

    признак того, что кампании клиента управляются яндекс-агенством

=cut

sub is_ya_agency_client { shift->_client_data->{is_ya_agency_client} }

__PACKAGE__->meta->make_immutable();

1;
