package API::Converter;

use strict;
use warnings;
use utf8;

=head2

    $Id$

=head1 NAME

    API::Converter

=head1 SYNOPSIS

=head1 DESCRIPTION

      Класс для преобразования структур из одного формата в другой
    (например: из формата который принимают ядровые функции или модели
    в формат ответа API )
    
      Использует карту соответствий между исходной и результирующей структурой
    Например такую:
    my $conversion = {
      login => 'Username',
      uid => 'UserId',
      campaigns => [ Campaigns => {
          cid => 'CampaignId',
          clicks => 'Clicks',
          strategy => \&convert_strategy
          groups => [
              AdGroups => {
                  group_name => 'AdGroupName',
                  pid => 'AdGroupId',
                  banners => [ Banners => {
                      bid => 'Id',
                      body => 'Text',
                      title => 'Title'
                      flags => [ Flags => \&convert_flags ]
                  }],
              }
          ]
      }]
    };
    Если нужно преобразовать только названия полей
        Правило:
            { bid => 'Id'}
        Исходные данные:
            { bid => ... }
        Результат:
            { Id => ... }
    Если нужно нужно преобразовать и значения полей, можно воспользоваться функцией.
        Правило:
            { flags => [ Flags => \&convert_flags ] }
        Исходные данные:
            { flags => 'tobaco' }
        Результат:
            { Flags => 'TOBACCO' }
        Преобразует поле flags во Flags,
        значение поля преобразуется в нужный формат с помощью функции convert_flags
    
    Если есть необходимость конвертировать при помощи функции не только значения но и нзвание поля. Например когда нет однозначного соответствия между названиями поля до и после конвертации (как в случае со стратегиями)
        Правило:
            { flags => \&convert_flags, }
        Исходные данные:
            {
                flags => 'age:18,tobacco'
            }
        Результат:
            {
                AdCategories => 'TOBACCO',
                AgeLabel => 18
            }
        Функция function_converter должна возвращать хеш
        {
            new_field_name => new_field_value
        }
        Таким образом, одному полю в исходной структуре может соответствовать несколько в результирующей
    
    Если нужно убрать вложенность можно указать мапинг только дловложенных полей 
        Правило:
        DailyBudget => {
            Amount => 'day_budget', 
            Mode => [day_budget_show_mode => \&convert_show_mode],
        },
        Исходные данные:
        DailyBudget => {
            Amount => ...,
            Mode => ...
        }
        Результат: {
            day_budget => ...,
            day_budget_show_mode => ...
        }
    Если наоборот нужно создать вложенность
        Правило:
        day_budget => [ DailyBudget => 'Amount'],
        или 
        day_budget_show_mode => [
                'DailyBudget',
                'Mode' => \&convert_show_mode_to_external
        ],
        Исходные данные:
        {
            day_budget => ...,
            day_budget_show_mode => ...
        }
        Результат:
        DailyBudget => {
            Amount => ...,
            Mode => ...
        }

=head1 METHODS

    new(map) - принимает карту соответствий между исходной и результирующей структурой,
               сохраняет в self
    convert(data) - принимает на вход данные которые нужно конвертировать
                    возвращает сконвертированные данные
=cut

use Yandex::HashUtils qw/hash_merge/;
use B qw(svref_2object);
use Carp;

use Mouse;

=head2 map
    Набор правил для приобразования полей
    из исходнного формата в результирующий
=cut

has map => (
    is => 'ro',
    isa => 'HashRef',
    default =>sub {{}},
    required => 1,
);

=head2 convert(data)

    Принимает структуру которую необходимо преобразовать
    В соответствии с картой соответствий
    преобразует ключи и значения структуры к выходному формату
    data может быть ссылкой на хеш или массив
    
    
=cut

sub convert {
    my $self = shift;
    my ($data, $map) = @_;
    $map ||= $self->map;
    foreach my $item (ref $data eq 'ARRAY' ? @$data : $data ) {
        foreach my $field (sort keys %$item) { # sort не обязателен, no just in case
            my $scheme = $map->{$field};
            my $value = exists $item->{$field} ? delete $item->{$field} : undef;
            if (!ref $scheme) {
            # для случая когда нужно конвертировать только название поля, значение остается не изменным
                if ($scheme) {
                    $item->{$scheme} = $value;
                } else {
                    next;
                }
            } elsif (ref $scheme eq 'ARRAY') {
            # для случая когда нужно конвертировать и значение
                my $last_index = $#{$scheme};
                my $sub_scheme = $scheme->[$last_index];
                my @link_names = @$scheme[0..$last_index-1];
                if (ref $sub_scheme eq 'CODE') {
                # для случая когда значение конвертируется какой либо функцией
                    my $result = $sub_scheme->($value, @link_names);
                    if (ref $result eq 'HASH') {
                         for my $key (keys %$result) {
                            make_nested_hash($item, (@link_names, $key), $result->{$key});
                         }
                    } else {
                        make_nested_hash($item, @link_names, $result);
                    }
                } elsif (ref $sub_scheme eq 'HASH') {
                # для случая когда значение представляет собой структуру которую тоже нужно конвертировать;
                    my $result = $self->convert($value, $sub_scheme);
                    make_nested_hash($item, @link_names, $result);
                } else {
                    push @link_names, $sub_scheme;
                    make_nested_hash($item, @link_names, $value);
                    #croak "converting rule of field \"$field\" has wrong format";
                }
            } elsif (ref $scheme eq 'HASH') {
            # для случая когда нужно убрать вложенность
                hash_merge $item, $self->convert($value, $scheme);
            } elsif (ref $scheme eq 'CODE') {
            # для случая когда из одного поля после конвертации нужно получить 2 или больше             
                my $result = $scheme->($value);
                if (ref $result eq 'HASH') {
                    hash_merge $item, $result;
                } else {
                    my $sub_name = _get_sub_name($scheme);
                    croak "converting function \"$sub_name\" must return hash";
                }
            } else {
                croak "converting rule of field \"$field\" has wrong format";
            }
        }
    }
    return $data;
}


sub _get_sub_name($){
    my $cv = svref_2object (shift);
    my $gv = $cv->GV;
    return $gv->NAME;
}

=head2 make_nested_hash
    Генерирует вложенную структуру
    Принимает: хеш в который нужно добавить структуру,
    массив полей в порядке в котором они должны быть вложены друг в друга,
    значение которое нужно присвоить самому "глубокому" элементу
    Возвращает: исходный хеш с добавленной структурой
=cut

sub make_nested_hash {
    my $ref = \shift;
    my $h = $$ref;
    my $value = pop;
    $ref = \$$ref->{ $_ } foreach @_;
    $$ref = $value;

    return $h;
}


__PACKAGE__->meta->make_immutable();

1;

__END__
