package API::Converter::ConvertSubs;

use Direct::Modern;

use POSIX qw/floor/;
use Currencies;

require Exporter;
our @ISA = qw/Exporter/;
our @EXPORT = qw/
        convert_to_money
        convert_to_long
        convert_to_float_2_decimal_places
        convert_string_comma_to_array
        convert_array_to_string_comma
        convert_string_space_to_array
        convert_array_to_string_space
        convert_nillable_array
        convert_items_element_to_array
        convert_from_yesno_to_bool
        CamelCase_to_SNAKE_CASE
        SNAKE_CASE_to_CamelCase
        uppercase
        /;
        
=head2 convert_to_money
    
    Преобразует целочисленную ставку (в API ставки присылают домноженными на 1_000_000) в деньги
    
=cut

sub convert_to_money {
    my $num = shift;
    if (!defined $num) {
        return;
    } else {
        return 0 + sprintf('%0.2f', $num / 1000_000); # округляем значение до второго знака после запятой
    }
}

=head2 convert_to_long
    
    Преобразует ставку в деньгах в целое число, что бы отдать пользователю
    
=cut

sub convert_to_long {
    my $num = shift;
    my $epsilon = $Currencies::EPSILON;
    if (!defined $num) {
        return;
    } else {
        return floor(($num * 1000_000) + $epsilon);
    }
}

=head2 convert_to_float_2_decimal_places
    
    Преобразует целочисленное число n*1000_000
    в n c двумя знаками после запятой
    
=cut

sub convert_to_float_2_decimal_places {
    my $num = shift;
    if (!defined $num) {
        return;
    } else {
        return int($num / 10000) / 100;
    }
}

=head2 convert_string_comma_to_array
    
    Преобразует строку с элементами через запятую
    в массив и оборачивает его в Items
    
=cut

sub convert_string_comma_to_array {
    my $value = shift;
    if ($value) {
        return { Items => [split(',' => $value)] };
    }
    return;
}

=head2 convert_array_to_string_comma
    
    Преобразует массив обернутый в Items в строку с элементами через запятую
    
=cut

sub convert_array_to_string_comma {
    my $value = shift;
    if ($value) {
        return join(',' => @{$value->{Items}});
    }
    return;
}

=head2 convert_string_space_to_array
    
    Преобразует строку с элементами через пробел
    в массив и оборачивает его в Items
    
=cut

sub convert_string_space_to_array {
    my $value = shift;
    if ($value) {
        return { Items => [split( /\s/, $value )] };
    }
    return;
}

=head2 convert_array_to_string_space
    
    Преобразует массив обернутый в Items в строку с элементами через пробел
    
=cut

sub convert_array_to_string_space {
    my $value = shift;
    if ($value) {
        return join(' ' => @{$value->{Items}});
    }
    return;
}

=head2 uppercase
    
    Приводит строку к верхнему регистру
    
=cut

sub uppercase {
    return uc shift;
}

=head2 convert_from_yesno_to_bool
    
    Преобразует значения 1\0 в YES\NO
    
=cut

sub convert_from_yesno_to_bool {
    my $value = uc shift;
    if ($value eq 'YES') {
        return 1;
    } elsif ($value eq 'NO') {
        return 0;
    } else {
        die 'unknown value'
    }
}

=head2 convert_nillable_array

    Оборчивает массив, который может быть nill, в структуру с Items

=cut

sub convert_nillable_array {
    my $array = shift;
    if (@$array) {
        return { Items => $array };
    }
    return;
}

=head2 convert_items_element_to_array

    Конвертирует элемент с Items в массив

=cut

sub convert_items_element_to_array {
    my $element = shift;
    return exists $element->{Items} ? $element->{Items} : [];
}

=head2 CamelCase_to_SNAKE_CASE
    
    Преобразует строку в формате CamelCase в формат SNAKE_CASE
    
=cut

sub CamelCase_to_SNAKE_CASE {
    my $value = shift;
    $value =~ s/([A-Z][a-z]+)/\U$1_/g;
    chop($value);
    return $value;
}

=head2 SNAKE_CASE_to_CamelCase
    
    Преобразует строку в формате SNAKE_CASE в формат CamelCase
    
=cut

sub SNAKE_CASE_to_CamelCase {
    my $value = shift;
    $value =~ s/_?([A-Z]+)/\u\L$1/g;
    return $value;
}

1;
