package API::PSGI::Base;

# $Id$

=pod

=head1 NAME

    API::PSGI;

=head1 DESCRIPTION

    Базовый класс для PSGI приложений большого API

=cut

use Direct::Modern;

use JSON;
use HTTP::Status qw/:constants/;
use List::Util qw/max/;
use Module::Load;
use Path::Tiny;
use Plack::Builder;
use Plack::NormalizeResponse;
use Plack::Response;
use Plack::UTF8Request;
use POSIX qw/strftime/;
use Scalar::Util qw/blessed/;
use Try::Tiny;

use Yandex::Clone qw/yclone/;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::Log::Messages;
use Yandex::Trace;
use Yandex::DBTools;

use Settings;

use APICommon ();
use Campaign::Types;
use Client qw//;
use Client::ClientFeatures ();
use EnvTools qw/is_beta/;
use LogTools qw//;
use SentryTools;
use MailNotification;
use Rbac;
use RBACDirect;

use API::App::CheckAccess;
use API::Authorization::TokenFormat;
use API::Authorization;
use API::Exception::IllegalResponse;
use API::RequestLog::Log;
use API::RequestLog::Logger;
use API::Services::Singleton;
use API::Units::Bucket;
use API::Units::Costs;
use API::Units::BucketManager;
use API::Units::FreeOfChargeRoles;
use API::UserConnectionLock;
use API::Version;
use API::Settings;
use API::Response::Headers;
use API::Request::Headers;

use Direct::Errors::Messages;

use HTTP::Status qw/HTTP_SEE_OTHER/;
use UnitsFactory;
use TvmChecker;

our %BLOCKED_METHODS = (
    #192854917 => { # ra-trinet-add DIRECT-57995
    #    changes => 1
    #},
    220883968 => { # hmepas-dev for tests
        changes => 1
    }
);

my %ERRORBOOSTER_SKIP_PACKAGES = (
    skip => [qw/main Try::Tiny API::PSGI::Base/], 
    skip_re => [qr|^Plack::|]        
);

=head2 %GENERIC_METHODS
    методы для которых не обязательно указание клиента (агентские мультиклиентые
    запросы или методы, которые не работают с данными клиента)

    Заданы в виде структуры (
        имя_сервиса => { (имя_метода | all) => 1, ..}
        ..
    )
=cut

our %GENERIC_METHODS = (
    dictionaries => { all => 1 },
    changes => { checkDictionaries => 1 }
);

=head2 new()

    Конструктор, параметров нет

=cut

sub new {
    my $class = shift;
    return bless {}, $class;
}

=head2 get_app

    Возвращает PSGI приложение

=cut

sub get_app {
    my $class = shift;
    my $self = $class->new();

    local $SIG{__WARN__} = sub {
        my $req_id = Yandex::Trace::current_span_id();
        warn strftime("%Y-%m-%d %H:%M:%S", localtime( time() ) )
            . ($req_id ? " [$req_id]: " : ': ')
            . join(' ', @_);
        return 1;
    };

    # основное приложение
    my $app = sub {
        my $env = shift;
        my $r = Plack::UTF8Request->new($env);

        my $response;
      
        no warnings 'once';
        local $Yandex::TVM2::APP_ID = $Settings::TVM2_APP_ID{api};
        local $Yandex::TVM2::SECRET_PATH = $Settings::TVM2_SECRET_PATH{api};
        local $Yandex::Blackbox::BLACKBOX_USE_TVM_CHECKER = \&TvmChecker::use_tvm;
      
        $self->log->request_id(Yandex::Trace::current_span_id());

        my $req_id = Yandex::Trace::current_span_id();
        $self->units_bucket_manager->req_id($req_id);
        local $SIG{__WARN__} = sub {
            warn strftime("%Y-%m-%d %H:%M:%S", localtime( time() ) )
                . ($req_id ? " [$req_id]: " : ': ')
                . join(' ', @_);
            return 1;
        };

        my $json_logger = JSON->new();
        $json_logger->utf8->allow_nonref->allow_blessed->allow_unknown->convert_blessed;

        try { # на случай если где-то не обработали исключение
            local $SIG{__DIE__} = SentryTools::prepare_die_handler();
            $response = $self->handle_plack_request($r, $req_id);
        } catch {
            my $error = $_;
            #Отправляем ошибку в errorbooster
            SentryTools::send_last_exception($error, %ERRORBOOSTER_SKIP_PACKAGES);
            my $log_message = "\"$error\"" . ( ref $error ? ': ' . $json_logger->encode($error) : '');

            if ( blessed $error ) {
                if ( $error->isa('API::Exception::IllegalRequest') ) {
                    $response = $self->client_fault( $error->reason );
                } elsif ( $error->isa('API::Exception::IllegalResponse') ) {
                    warn "Illegal service response: $log_message";
                    $response = $self->server_fault( error_OperationFailed );
                } else {
                    warn "Unexpected error: $log_message";
                }
            } else {
                warn "Unexpected error: $log_message";
            }
        } finally {
            if (my @u = @{$self->units_stats}) {
                $self->log->add_units_info(@u);
                $self->units_bucket_manager->units_stat(\@u);
            }

            if (my $units_bucket = $self->units_bucket_manager->get_bucket) {
                $self->log->units_spending_user_client_id($units_bucket->id);
            }
            $response //= $self->server_fault(error_ServerFailed);
            $env->{appcode} = $self->log->error_status; # пишем код серверной ошибки в X-Accel-Info
            API::RequestLog::Logger::log($self->log);

            # логируем данные для контроля списания баллов
            $self->log_units_data($self->units_bucket_manager->get_log_data());
        };

        my $normalized_response = $self->_get_finalized_plack_response($response);

        $self->after_request();
        return $normalized_response;
    };

    return $self->_wrap($app);
}

=head2 extra_response_headers($headers)

    Получение поля extra_response_headers, в котором хранятся дополнительные HTTP-заголовки ответа

=cut

sub extra_response_headers {
    my ($self) = @_;
    
    $self->{extra_response_headers} //= API::Response::Headers->new();
    return $self->{extra_response_headers};
}

=head2 flush_extra_response_headers()

    Удаляем дополнительные HTTP-заголовки ответа

=cut

sub flush_extra_response_headers {
    my ($self) = @_;
    $self->{extra_response_headers} = undef;
    return;
}

=head2 flush_request_headers_for_log()

    Удаляем HTTP-заголовки запроса, которые необходимо записать в логи

=cut

sub flush_request_headers_for_log {
    my ($self) = @_;
    $self->{http_request_headers_for_log} = undef;
    return;
}

sub _add_response_headers {
    my ($self, $headers) = @_;

    $self->extra_response_headers->RequestId($self->request->id) if $self->request;

    my $is_service_uses_units = exists $API::Settings::NO_UNITS_SERVICE->{$self->request->service_name} ? 0 : 1;

    if ( $is_service_uses_units && ( my @u = @{$self->units_stats} )) {
        $self->extra_response_headers->Units(join('/', @u));
        if (my $units_bucket = $self->units_bucket_manager->get_bucket) {
            $self->extra_response_headers->UnitsUsedLogin($units_bucket->owner_login);
        }
    }

    if ($self->extra_response_headers) {
        push @$headers, @{$self->extra_response_headers->to_array};
    }

    if (!$self->is_production()) {
        push @$headers, ( 'X-Perl-Implementation' => 'true' );
    }
    return;
}

sub _wrap {
    my $self = shift;
    my $app = shift;

    return builder {
        # uncomment for debug
        # enable 'Plack::Middleware::Dumper';
        enable 'MemLimit', limit => $Settings::SOAP_APACHE_MAX_PROC_SIZE;
        enable 'Trace', cmd_type => "direct.".(lc $self->cmd_type);
        enable 'SetAccelInfo';
        $app;
    };

}

=head2 cmd_type

    Тип операции, нужно для логирования, отличается для SOAP и JSON, должно
    быть определено в наследниках

=cut

sub cmd_type { # i.e. JSON-API
    die 'redefine'
}

=head2 init_service($service_name)

    Инициализирует объект сервиса оп имени

=cut

sub init_service {
    my $self = shift;
    my $service_name = shift or return;

    my $service;
    try {
        $service = $self->services->init($service_name);
    } catch {
        warn "service $service_name not loaded: $_";
    };
    return $service;
}

=head2 services

    Объект API::Services для текущей версии API

=cut

sub services {
    my $self = shift;
    return $self->{services} //= API::Services::Singleton::get_for_version($API::Version::number);
}

=head2 fill_request($plack_request)

    Заполняем $self->request по Plack::Request

=cut

sub fill_request {
    my ($self, $plack_request) = @_;
    return $self->{request} = $self->init_request($plack_request);
}

=head2 request

    Объект запроса API::Request

=cut

sub request { shift->{request} }

=head2 current_auth($auth)

    Ассесор для API::Authorization объекта с данными авторизации на текущий
    запрос, заполняется если $auth задано.

=cut

sub current_auth {
    my ($self, $auth) = @_;
    if($auth) {
        $self->{auth} = $auth;
    }
    $self->{auth};
}

=head2 after_request

    Метод для процедур которые нужно сделать после каждого запроса

    TODO: перевести стейт отсюда на $self->request_ctx объект создающийся на каждый запрос

=cut

sub after_request {
    my $self = shift;
    $self->flush_auth;
    $self->flush_log;
    $self->{request} = undef;
    $self->{_units_stats} = [];
    $self->{_units_bucket_manager} = undef;
    $self->flush_extra_response_headers;
    $self->flush_request_headers_for_log;
}

=head2 flush_auth

    Сбрасываем объект авторизации (в конце обработки запроса)

=cut

sub flush_auth {
    my $self = shift;
    $self->{auth} = undef;
    return;
}

=head2 log

    API::RequestLog::Log объект с данными текущего запроса, для заполнения в
    процессе обработки запроса, на выходе из хэндлера сохраняется в БД.

=cut

sub log { return shift->{log} //= API::RequestLog::Log->new() }

=head2 log_units_data($units_data_json)

    Выводим переданные данные $units_data_json в системный лог messages для последующей загрузки в ClickHouse

=cut

sub log_units_data {
    my ($self, $units_data_json) = @_;

    my $units_log = $self->{units_log} //= Yandex::Log::Messages->new();
    $units_log->bulk_out(units_data => [$units_data_json]);

    return;
}

=head2 http_request_headers_for_log($headers)

    Получение поля http_request_headers_for_log, в котором хранятся HTTP-заголовки запроса, которые необходимо записать в логи

=cut

sub http_request_headers_for_log {
    my ($self) = @_;

    $self->{http_request_headers_for_log} //= API::Request::Headers->new();
    return $self->{http_request_headers_for_log};
}

=head2 flush_log

    Сбрасываем объект с информацией для логирования в конце обработки запроса

=cut

sub flush_log {
    my $self = shift;
    $self->{log} = undef;
    return;
}

=head2 units_withdraw_for_request_error($bucket)

    Списываем с указанной корзины баллы за ошибочный запрос, учитываем роль оператора и текущее приложение

=cut

sub units_withdraw_for_request_error {
    my ($self, $bucket) = @_;

    my $auth = $self->current_auth;
    my $coef = API::Units::Costs::get_app_coef($auth->application_id);
    my $error_cost = API::Units::FreeOfChargeRoles::is_free_role($auth->role) ? 0 : $API::Units::Costs::COMMON_REQUEST_ERROR;

    my $units = UnitsFactory::create_adjusted(
        $bucket->id, $coef, $bucket->limit
    );
    $error_cost = $units->withdraw($error_cost);
    $self->{_units_stats} = [
        $error_cost,
        $units->balance,
        $units->limit
    ];

    return;
}

=head2 set_auth_by_token_and_log_on_success($request)

    Ауфентицируем пользователя по информации в запросе, в случае успеха логируем данные

    Метод закончивается выставлением $self->current_auth, где в случае фейковой
    авторизации будут уже данные фейкового пользователя.

    Все методе все что касается проверок токена и приложения. Дальнейшие
    проверки вынесены в API::Authorization::check_api_access, который в случае фейковой
    авторизации вызывается дважды -- для реала и фейка

=cut

sub set_auth_by_token_and_log_on_success {
    my ($self, $request) = @_;

    $request->token or return $self->client_fault(
        error_BadRequest(
            iget('OAuth-токен не указан')
        )
    );

    unless(API::Authorization::TokenFormat::is_valid_token($request->token)) {
        return $self->client_fault(
            error_BadRequest( iget('Неверный формат OAuth-токена') )
        );
    }

    my $auth = API::Authorization->new(
        token => $request->token,
        is_token_persistent => $request->is_token_persistent,
        remote_address => $request->remote_ip,
        request_id => $request->id
    );

    # Проверяем токен
    my $token_authorized;
    try {
        $token_authorized = $auth->is_token_authorized;
        return 1;
    } catch {
        warn "Authorization failed "  . to_json({authorize_user => $_});
        return;
    } or return $self->server_fault(error_AuthTmpUnavail);

    return $self->client_fault(
        error_AuthFail( iget('Недействительный OAuth-токен') )
    ) unless $token_authorized;

    return $self->client_fault( error_UnknownUser() ) unless $auth->operator_user->found;

    # Логируем пользователя только после проверки результато в валидации
    # иначе не поймать exception
    $self->log->add_auth_info($auth);

    if (!$request->is_token_persistent && !API::App::CheckAccess::application_has_api_access($auth->application_id)) {
        return $self->client_fault(error_IncompleteSignUp());
    }

    if(my $error = $auth->check_api_access) {
        return $self->client_fault($error);
    }

    # на дев. средах и ТС подменяем авторизацию если супером указан Fake-Login
    my $fake_auth;
    if (($auth->is_role_super || $auth->is_role_superreader) && $request->fake_login && !$self->is_production) {
        $fake_auth = $auth->fake_login_by_request($request) or return $self->client_fault(
            error_NotFound(iget("В HTTP-заголовке Fake-Login указан несуществующий логин"))
        );
        $fake_auth->application_id( $auth->application_id );
    }

    if($fake_auth) {
        if(my $error = $fake_auth->check_api_access) {
            return $self->client_fault($error);
        }
        $self->current_auth($fake_auth);
    } else {
        $self->current_auth($auth);
    }

    return; # auth ok no errors
}

=head2 handle_operator_failure($error)

    Списываем баллы с оператора/бренда и возвращаем ошибку $error, удобный хелпер для выхода из handler-а вида
    return $self->handle_operator_failure( error_NoRights(iget('Нет прав на указанного клиента')) )
    На этой стадии списываем баллы всегда с оператора/бренда

=cut

sub handle_operator_failure {
    my ($self, $error) = @_;

    $self->units_withdraw_for_request_error( $self->units_bucket_manager->create_bucket_for_operator_failure() );

    return $self->client_fault($error);
}

=head2 operator_claims_client($auth, $client_login)

    По $auth API::Authorization проверяет право оператора на работу с сабклиентом указанным в HTTP-заголовке Client-Login

    Параметры:
        $auth - объект авторизации клиента API::Authorization
        $client_login - строка из http-заголовка Client-Login (может быть undef или пустой)

    Ответ ($error, $subclient):
        $error ошибка Direct::Defect или undef
        $subclient это сабклиент (API::Authorization::User), если нет ошибки

=cut

sub operator_claims_client {
    my ($self, $auth, $client_login) = @_;

    if ($auth->is_role_client) {
        return $self->_client_claim_its_client_login($auth, $client_login);
    } else { # authenticated by agency or internal role
        return $self->_agency_or_internal_user_claims_client($auth, $client_login);
    }
}

=head2 _client_claim_its_client_login($auth, $client_login)

    см. operator_claims_client($auth, $client_login)

    Для прямого клиента необязательно указывать логин в заголовке,
    но если он указан, то должен сооветствовать либо логину оператора,
    либо логину его главного представителя

=cut

sub _client_claim_its_client_login {
    my ($self, $auth, $client_login) = @_;

    if ($client_login && $auth->check_operator_and_client_relation($client_login)){
        return (undef, $auth->subclient_by_login($client_login));
    }

    return error_NotFound_ClientLogin()
        if $client_login && !$auth->match_operator_or_chief_rep_login($client_login);

    return (undef, $auth->chief_rep_user);
}

=head2 _agency_or_internal_user_claims_client($auth, $client_login)

    см. operator_claims_client($auth, $client_login)

    В случае запроса от агентства или внутренней роли, если мы вызываем
    по-клиентскую операцию, то обязательно нужно указывать клиента с данными
    которого мы работает. В любом случае, если Client-Login указан, то проверяем
    на него права

=cut

sub _agency_or_internal_user_claims_client {
    my ($self, $auth, $client_login) = @_;

    my $subclient;
    if ($self->is_client_login_must_be_subclient) {
        if ($client_login) {
            $subclient = $auth->subclient_by_login($client_login) or return error_NotFound_ClientLogin();
            $subclient->role($auth->role_by_uid($subclient->uid));
            return error_BadRequest_ClientLogin() if !$subclient->is_client;
        } else {
            return error_BadRequest_ClientLogin();
        }
    } else {
        if ($client_login) {
            $subclient = $auth->subclient_by_login($client_login) or return error_NotFound_ClientLogin();
        } else {
            $subclient = $auth->chief_rep_user;
        }
    }

    return (undef, $subclient);
}

=head2 is_client_login_must_be_subclient

    true если для текущего запроса (сейчас это зависит от метода, который
    вызываем) Client-Login, указанный или вычисленный по оператору, должен быть
    обычным клиентом (не агентством и не менеджером)

=cut

sub is_client_login_must_be_subclient {
    my $self = shift;
    my $service = $self->request->service_name;
    return 1 unless exists $GENERIC_METHODS{$service};
    my $methods = $GENERIC_METHODS{$service};
    return ($methods->{all} || $methods->{$self->request->operation}) ? 0 : 1;
}

=head2 try_concurrency_lock_and_run(@_)

    Параметры аналогичны run_service_operation
    Метод запускает запускает run_service_operation(@_) и возвращает
    результат если не превышен лимит на количество одновременных соединений,
    если превышен, то возвращает ошибку

=cut

sub try_concurrency_lock_and_run {
    my ($self, $request, $service, $auth, $subclient) = @_;

    my $locker = $self->user_connection_lock( $subclient );

    return $self->handle_operator_failure(
        error_ConnectionsLimitExceeded( iget('Превышено максимальное количество одновременных запросов к API') )
    ) unless $locker->get;

    my $result = $self->run_service_operation($request, $service, $auth, $subclient);

    $locker->release;

    return $result;
}


=head2 handle_plack_request($r_)

    Обрабатываем запрос plack-запрос $r Plack::Request и отдаем результат в виде хэша

=cut

sub handle_plack_request {
    my ($self, $r, $req_id) = @_; # Plack::Request object, req_id just for traces

    Yandex::DBShards::clear_cache();
    Campaign::Types::clear_cache();
    Rbac::clear_cache();
    RBACDirect::clear_cache();
    Client::ClientFeatures::clear_cache();
    my $request = $self->fill_request($r);
    my $units_bucket_manager = $self->units_bucket_manager;
    $units_bucket_manager->client_login($request->login);
    $units_bucket_manager->use_operator_units($request->use_operator_units);

    $self->log->add_common_request_info($request);

    die 'something is really wrong, request from external network while development mode on'
        if $self->is_development && !$request->from_internal_network;

    my $service_name = $self->services->service_name_by_uri($request->service_name)
        or return {not_found => 1};

    my $service = $self->init_service($service_name)
        or return $self->server_fault(error_ServiceInitializationFailed);

    return $self->server_fault(error_ServiceInitializationFailed)
        if !$service->is_available_on_production && $self->is_production();

    if(my $client_fault = $self->set_auth_by_token_and_log_on_success($request)) {
        return $client_fault;
    }
    my $auth = $self->current_auth;
    $self->units_bucket_manager->auth($auth);

    # добавляем UID к комментариям sql-запросов
    my $old_vars = $Yandex::DBTools::TRACE_COMMENT_VARS;
    local $Yandex::DBTools::TRACE_COMMENT_VARS = sub {return {%{$old_vars->()}, operator => $auth->operator_uid}; };

    if ( !$service->access_allowed_to_application( $auth->application_id ) ) {
        return $self->server_fault(error_ServiceInitializationFailed);
    }

    my ($no_rights_to_subclient_error, $subclient) = $self->operator_claims_client($auth, $request->login);
    $self->units_bucket_manager->subclient($subclient);

    return $self->handle_operator_failure($no_rights_to_subclient_error) if $no_rights_to_subclient_error;

    if ($subclient->is_blocked && $request->operation ne 'get') {
        return $self->handle_operator_failure( error_AccessToApiDenied_AccountBlocked() );
    }

    $self->log->chief_rep_uid($subclient->uid);

    if (exists $BLOCKED_METHODS{$auth->operator_uid} && $BLOCKED_METHODS{$auth->operator_uid}{$request->service_name}) {
        return $self->client_fault( error_NoRights('Доступ к методу заблокирован'));
    }

    return $self->try_concurrency_lock_and_run($request, $service, $auth, $subclient);
}

=head2 run_service_operation($request, $service, $auth, $subclient)

    Параметры:
        * $request (API::Request) - разобранный запрос
        * $service (API::Service::Base) - уже проинициализированный объект сервиса
        * $auth - объекту авторизации API::Authorization
        * $subclient - API::Authorization::User объект главного представителя
            клиента указанного в Client-Login или, если заголовок не указан и
            авторизованный пользователь является представителем прямого клиента, то
            объект главного представителя оператора

    Запускает операцию из запроса и возвращает ответ

=cut

sub run_service_operation {
    my ($self, $request, $service, $auth, $subclient) = @_;

    if (my $error = $request->error) { # парсинг запроса завершился с ошибкой
        return $self->handle_operator_failure($error);
    }

    if ($request->is_use_operator_units_true_or_auto && !$auth->is_role_agency) {
        return $self->handle_operator_failure(error_BadRequest(iget("Использование HTTP-заголовка Use-Operator-Units доступно только агентствам")));
    }

    if (Client::is_client_converting_soon($subclient->client_id)) {
        # намеряно не списываем баллы, поэтому кидаем client_fault напрямую
        return $self->client_fault(
            error_AccessToApiDenied(APICommon::msg_converting_in_progress)
        );
    }

    if (Client::client_must_convert($subclient->client_id)) {
        # намеряно не списываем баллы, поэтому кидаем client_fault напрямую
        return $self->client_fault(
            error_NoRights(APICommon::msg_must_convert)
        );
    }

    # разбирает запрос, поэтому вызывать можно только в случае если 
    # запрос распарсился успешно
    $self->log->add_parsed_request_info($request);

    my $operation = $request->operation;

    if (!$service->can($operation)) {
        return $self->handle_operator_failure( error_OperationNotFound(iget('Операция не реализована')) );
    }

    # предпочитать перл-имплементацию методов, которые есть и в перл и java, регулируется
    # заголовком. Наличие заголовка не должно влиять на поведение прода
    my $prefer_perl = !$self->is_production && $request->prefer_perl_implementation ? 1 : 0;
    my $service_operation_name = $request->service_name . '.' . $operation;
    if ( !$prefer_perl && $self->dispatch_to_java_service($service_operation_name) ) {
        my $uri = $request->_plack_request->request_uri;
        $self->log->method( 'proxy' . $self->log->method );
        my $current_trace = Yandex::Trace::current_trace;
        $current_trace->method( 'proxy' . $service_operation_name );
        return { content_type => 'text/plain', reverse_proxy_redirect => '@try_java_api' };
    }

    # Настраиваем нотификатора, чтобы письма шли пользователю
    MailNotification::save_UID_host($auth->operator_uid);
    local $LogTools::context{UID} = $auth->operator_uid;

    my $result;
    my $operation_error;

    $request->trace->method($request->service_name.".$operation");
    $request->trace->annotations([ [version => 'v5'] ]);

    try {
        local $SIG{__DIE__} = SentryTools::prepare_die_handler();
        $service->flush;
        $service->set_authorization( $auth );
        $service->set_locale($request->locale);
        $service->set_subclient($subclient);
        $service->set_protocol($request->protocol);
        $service->set_current_operation($operation);
        $service->set_header_client_login($request->login);
        $service->set_units_multiplier( API::Units::Costs::get_app_coef($auth->application_id) );
        $service->set_units_bucket( $self->units_bucket_manager->create_bucket );
        $service->set_http_host($request->http_host);
        $service->set_http_request_headers($request->headers);
        $service->http_response_headers($self->extra_response_headers);
        $service->http_request_headers_for_log($self->http_request_headers_for_log);

        if ($self->units_bucket_manager->can_switch_to_operator_bucket && !$service->units_enough_for_operation) {
            $service->set_units_bucket($self->units_bucket_manager->switch_to_operator_bucket);
        }

        if($service->units_enough_for_operation) {
            $service->units_withdraw_for_operation;

            $result = $service->$operation(yclone($request->params));

            $self->log->cids($service->affected_cids);
            $self->log->response_headers($self->extra_response_headers->to_hash);
            
            # если операция не вернула ответ "запрос целиком ошибочный", пробуем извлечь из ответа ID объектов
            unless ($self->is_error($result)) {
                try {
                    my $result_copy = yclone($result);
                    if ( my $ids = $service->response_ids($result_copy) ) {
                        $self->log->add_response_ids($ids);
                    }
                    my ($errors_count, $warnings_count) = $service->count_error_warning_objects_in_reponse($result_copy);
                    $self->log->add_error_object_count($errors_count);
                    $self->log->add_warning_object_count($warnings_count);
                } catch {
                    warn "failed to add response_ids to log or log errors/warnings count: $_";
                };
            }

            # если операция вернула ошибку на весь запрос
            if($self->is_error($result)) {
                # то списываем стоимость ошибки в баллах
                $service->units_withdraw_for_request_error;
            }
        } else {
            $result = error_NotEnoughUnits( iget('Недостаточно баллов для выполнения операции') );
        }

        try {
            if ( $service->include_response_in_logs ) {
                $self->log->response( { %$result } );
            }
        } catch {
            warn "failed to add result to log: $_";
        };
	try {
	   if ( $service->include_request_headers_in_logs ) {
		$self->log->request_headers( $service->http_request_headers_for_log->to_hash );
	   }
	} catch {
	    warn "failed to add request headers to log: $_";
	};
    } catch { # если сервис упал, возвращаем серверную ошибку
        my $error = $_;
        #Отправляем ошибку в errorbooster
        SentryTools::send_last_exception($error, %ERRORBOOSTER_SKIP_PACKAGES);
        my $error_object;
        if($self->is_error($error)) {
            $error_object = $error;
        } else {
            warn "$error";
            $self->_log_message(operation_fault => [$error]);
            $error_object = error_OperationFailed(
                $self->is_development ? $error : ''
            );
        }
        $operation_error = $self->server_fault(
            $error_object
        );

        try {
            if ( $service->include_response_in_logs ) {
                $self->log->response( { server_fault => { %$error_object } } );
            }
        } catch {
            warn "failed to add server fault to log: $_";
        };
    };

    $self->{_units_stats} = [
        $service->units_spent,
        $service->units_balance,
        $service->units_limit,
    ];

    if ($operation_error) {
        # операция сфейлила, возвращаем сервер-фейл
        return $operation_error;
    }

    if ( $self->is_error($result) ) {
        # операция отработала, но вернула объект ошибки
        # возвращаем клиентскую ошибку
        return $self->client_fault($result);
    }

    # Успешный вызов, оборачиваем ответ
    my $result_to_return;
    try {
        $result_to_return = $self->operation_response(
            $request->service_name,
            $operation,
            $result
        );
    } catch {
        # ... что не всегда получается, например, если сервис из-за ошибки
        # программирования возвращает данные, которые не соответствуют XSD
        API::Exception::IllegalResponse->raise($_);
    };

    return $result_to_return;
}

=head2 dispatch_to_java_service($service_operation_name)

    $service_operation_name -- это, например, 'bids.get'

    Возвращает true/false: надо ли пробрасывать этот запрос в java

=cut

sub dispatch_to_java_service {
    my ( $self, $service_operation_name ) = @_;

    state $force_proxy_operation = undef;
    if ( is_beta() ) {
        # как поставить:
        # direct-mk api5-java-proxy-service-operations adgroups.add adgroups.get
        # как сбросить:
        # direct-mk api5-java-proxy-service-operations-reset
        unless ( defined $force_proxy_operation ) {
            $force_proxy_operation = {};
            if ( -f "$Settings::ROOT/beta-settings.json" ) {
                my $settings = JSON::from_json( path("$Settings::ROOT/beta-settings.json")->slurp );
                if ( my $operations = $settings->{"api5-java-proxy-service-operations"} ) {
                    $force_proxy_operation = { map { $_ => 1 } split /\s+/, $operations };
                }
            }
        }

        if ( $force_proxy_operation->{$service_operation_name} ) {
            return 1;
        }
    }

    state $property;
    $property //= Property->new($Settings::API5_JAVA_PROXY_SERVICE_OPERATIONS_PROP);

    state $last_parsed_value = "";

    # { operation => 1, ... }
    state $need_to_proxy_operation = {};

    my $property_value = $property->get($Settings::API5_JAVA_PROXY_SERVICE_OPERATIONS_PROP_CACHE_TTL) // '';
    if ( $property_value ne $last_parsed_value ) {
        my @operations = grep { $Settings::API5_JAVA_ALLOW_PROXYING_OPERATION{$_} } split / /, $property_value;
        $need_to_proxy_operation = { map { $_ => 1 } @operations };
        $last_parsed_value = $property_value;
    }

    return $need_to_proxy_operation->{$service_operation_name};
}

=head2 units_stats

    Баллы на последнюю операцию [
        потрачено # потрачено за вызов последней операции
        баланс, # осталось от суточного лимита
        лимит, # суточный лимит
    ]
=cut

sub units_stats { shift->{_units_stats} || [] };

=head2 is_error($defect_error)

    Проверяет что переданый объект -- ошибка типа Direct::Defect

=cut

sub is_error {
    my $self = shift;
    my $object = shift;
    return ( blessed($object) and $object->isa('Direct::Defect') ) ? 1 : 0;
}

=head2 user_connection_lock($user)

    API::UserConnectionLock объект для работы с пользовательскими локами
    (обеспечивают ограничение на кол-во одновременных соединений от
    пользователя)
    $user - API::Authorization::User объект с данными текущего авторизованного
    пользователя

=cut

sub user_connection_lock {
    my ($self, $user) = @_;
    my $max_connections = $user->allowed_concurrent_calls;
    $max_connections //= $Settings::API_PARALLEL_QUERIES;

    return API::UserConnectionLock->new($user->client_id, $max_connections);
}

=head2 init_request

    Инициализирует объект запроса, должен быть переопределен в наследниках,
    т.к. запрос зависит от протокола

=cut

sub init_request { die "please redefine" }

=head2 operation_response($service, $operation, $response_data)

    Преобразуем данные которые вернула операция сервиса в текст ответа сервера
    согласно текущему протоколу, данные валидируются по схеме
    Агрументы: имя сервиса, имя операции, данные

=cut

sub operation_response { die "please redefine" }

=head2 server_fault($error)

    Преобразуем ошибку в текст серверной ошибки согласно протоколу

=cut

sub server_fault { die "please redefine" }

=head2 client_fault($error)

    Преобразуем ошибку в текст клиентской ошибки (ошибки по вине клиента)

=cut

sub client_fault { die "please redefine" }

=head2 is_development

    Флаг, если взведен то текущая среда - девелоперская (не ТС и НЕ ПРОД)

=cut

sub is_development { $API::Version::is_development }

=head2 is_production

    Флаг, если взведен то текущая среда - боевая (не тс и не бета)

=cut

sub is_production { $API::Version::is_production }

=head2 error_name_to_http_code($error)

    $error - Direct::Defect объект
    по ошибке возвращает ее http код

=cut

sub error_name_to_http_code {
    my ($self, $error) = @_;

    state $ERROR_NAME_TO_HTTP_CODE_MAP = {
        NotFound => HTTP_NOT_FOUND,
        BadParams => HTTP_BAD_REQUEST,
        OperationFailed => HTTP_INTERNAL_SERVER_ERROR,
    };
    return exists $ERROR_NAME_TO_HTTP_CODE_MAP->{$error->name} ? $ERROR_NAME_TO_HTTP_CODE_MAP->{$error->name} : HTTP_BAD_REQUEST;
}

sub _get_finalized_plack_response {
    my ($self, $response_data) = @_;

    my $finalized_response;
    if ($response_data->{callback}) {
        my $headers = [ 'Content-Type' => $response_data->{content_type} ];
        $self->_add_response_headers($headers);
        $finalized_response = sub {
            my ($response) = @_;
            my $writer = $response->([200, $headers]);
            $response_data->{callback}->($writer);
            $writer->close();
        };
    } elsif ($response_data->{redirect}) {
        my $plack_response = Plack::Response->new();
        $plack_response->redirect($response_data->{redirect}, HTTP_SEE_OTHER);
        $finalized_response = $plack_response->finalize;
    } elsif ($response_data->{reverse_proxy_redirect}) {
        my $tvm_ticket = eval { Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{api}) } or die "Cannot get ticket for $Settings::TVM2_APP_ID{api}: $@";
        $finalized_response = Plack::Response->new( 200, [
            'X-Accel-Redirect' => $response_data->{reverse_proxy_redirect},
            'X-Ya-Service-Ticket' => $tvm_ticket,
            'Content-Type' => $response_data->{content_type},
        ] )->finalize;
    } elsif ($response_data->{not_found}) {
        $finalized_response = Plack::Response->new( 404, [] )->finalize;
    } else {
        $finalized_response = Plack::NormalizeResponse::normalize($response_data);
        $self->_add_response_headers($finalized_response->[1]);
    }

    return $finalized_response;
}

=head2 units_bucket_manager

    Объект менеджера балльных корзин, позволяет правильно формировать объект балльной корзины для запроса.

=cut

sub units_bucket_manager {
    my ($self) = @_;

    $self->{_units_bucket_manager} //= API::Units::BucketManager->new();

    return $self->{_units_bucket_manager};
}

sub _log_message {
    my ($self, @log_data) = @_;
    $self->{message_log} //= Yandex::Log::Messages->new();
    $self->{message_log}->bulk_out(@log_data);
}

=head2 remove_disallowed_letters($s)

    Возвращает строку в которой нет символов не входящих в Settings::ALLOW_BANNER_LETTERS_STR

=cut

sub remove_disallowed_letters {
    my ($self, $s) = @_;
    return '' unless $s;
    my $DISALLOW_BANNER_LETTER_RE = qr/[^\Q$Settings::ALLOW_BANNER_LETTERS_STR\E\r\t\n]/i;
    $s =~ s/$DISALLOW_BANNER_LETTER_RE//gs;
    return $s;
}

1;
