package API::PSGI::SOAP;

# $Id$

=head1 NAME

    API::PSGI::SOAP;

=head1 DESCRIPTION

    Базовое SOAP PSGI приложение большого API

=cut


use strict;
use warnings;

use utf8;

use Encode;

use Plack::Builder;

use Yandex::Trace;

use API::Request::SOAP;
use XML::LibXML;

use base qw/API::PSGI::Base/;

sub cmd_type { 'SOAP-API' }

use Settings;

sub _wrap {
    my $self = shift;
    my $app = shift;

    return builder {
        enable 'Headers', set => ['Content-Type' => 'text/xml; charset=utf-8'];
        $self->SUPER::_wrap($app);
    };
}

sub init_request {
    my ($self, $plack_request) = @_;
    my $profile = Yandex::Trace::new_profile('api:soap:init_request');
    return API::Request::SOAP->new($plack_request, $self->services);
}

sub operation_response {
    my ($self, $service, $operation, $response_data) = @_;
    my $profile = Yandex::Trace::new_profile('api:soap:make_response');
    return {
        text => $self->services->response_to_xml(
            $service, $operation, $response_data)
    };
}

sub client_fault {
    my ($self, $error) = @_;
    return $self->_soap_fault('client', $error);
}

sub server_fault {
    my ($self, $error) = @_;
    $self->units_bucket_manager->is_server_fault(1);
    return $self->_soap_fault('server', $error);
}

sub _soap_fault {
    my $self = shift;
    my $type = ucfirst( shift() );
    my $error = shift;

    return {
        text => $self->_soap_fault_text(
            "SOAP-ENV:$type",
            $error
        )
    }
}

sub _soap_fault_text {
    my ($self, $soap_fault_code, $error ) = @_;

    $self->log->add_error_info($error);
    my $detail = XML::LibXML::Element->new('detail');

    my $fault_response= XML::LibXML::Element->new('FaultResponse');
    $fault_response->setNamespace("$API::Version::ns_prefix/general", 'ns3');

    my $error_code = XML::LibXML::Element->new('errorCode');
    $error_code->appendText($error->code);
    my $error_detail = XML::LibXML::Element->new('errorDetail');

    my $request_id = XML::LibXML::Element->new('requestId');
    $request_id->appendText(Yandex::Trace::current_span_id());

    my $description = $self->remove_disallowed_letters($error->description);

    $error_detail->appendText($description);

    $fault_response->appendChild($request_id);
    $fault_response->appendChild($error_code);
    $fault_response->appendChild($error_detail);

    $detail->appendChild($fault_response);

    return decode('utf8',
        $self->_soap_fault_sender->({
            faultcode => $soap_fault_code,
            faultstring => $error->text,
            detail => $detail
        }, 'UTF-8')->toString
    );
}

sub _soap_fault_sender {
    my $self = shift;
    return $self->{soap_fault_sender} //= $self->services->soap_server
        ->compileMessage('SENDER',
            body => [
                Fault => "{http://schemas.xmlsoap.org/soap/envelope/}Fault"
            ]
        );
}

1;
