package API::Request;

use strict;
use warnings;
use utf8;

use Try::Tiny;
use TextTools qw/normalize_login/;
use Locale::Util qw/parse_http_accept_language/;
use Yandex::I18n;

use Settings;
use IpTools qw/is_ip_in_list/;
use HttpTools qw/http_server_host/;

use Direct::Errors::Messages;

use API::Exception::IllegalRequest;

use Mouse;
use Mouse::Util::TypeConstraints;

=pod

    $Id$

=head1 NAME

    API::Request

=head1 SYNOPSIS

    my $request = API::Request::JSON->new($plack_request);
    if(my $e = $request->error) {
        die $e; # Direct::Defect
    } else {
        print Dumper $request->params;
    }

=head1 DESCRIPTION

    Базовый класс для работы с запросом в API. Предоставляем интерфейс как к
    HTTP запросу, так и структуру методов для получения данных из запросов
    разных протоколов (SOAP/JSON);

    При инициализации запрос обрабатывается и в случае ошибки поле error будет
    содержать соответствующий Direct::Defect объект

=head1 METHODS

=head2 _parsed

    Ленивый атрибут, вычисляется в конструкторе содержит структуру с данными
    запроса после парсинга. Конструктор атрибута Должен либо вернуть структура,
    либо упасть с ошибкой Direct::Defect, которая будет поймана при
    инициализации и помещена в аттрибут error

    В случае непредвиденной ошибки может падать с любым exception-ом, который
    будет обработан в конструкторе класса

=cut

=head2 protocol

    протокол запроса: {json|soap}, должно быть переопределно в
    наследниках

=cut

has protocol => (is => 'ro', isa => 'Str', default => sub { die 'redefine' });

=head2 operation

    Имя операции, должно быть переопределено в потомках

=cut

has operation => (is => 'ro', isa => 'Str', default => sub { die 'redefine' });

=head2 default_locale

    Локаль по умолчанию: en

=cut

has default_locale => ( is => 'ro', default => sub {'en'} );

=head2 error

    Read-Write
    Ошибка при парсинге запроса, содержит объект типа Direct::Defect 

=cut

has error => (is => 'rw', default => sub { undef } );

=head2 service_name

    Имя сервиса

=cut

has service_name => (is => 'ro', lazy => 1, default => sub {
        my $self = shift;
        my $path = $self->_plack_request->path_info;
        $path =~ /\/([a-zA-Z0-9]+)\/*$/;
        return $1;
});

has _parsed => (is => 'ro', isa => 'HashRef', lazy => 1, default => sub {
    my $self = shift;

    my $parsed;

    try {
        $parsed = $self->_parse_request;
    } catch {
        API::Exception::IllegalRequest->raise($_);
    };

    return $parsed;
});

has [qw/_plack_request _services/] => ( is => 'ro', isa => 'Object', required => 1 );

=head2 content

    Тело запросa: строка

=cut

=head2 headers

    http-заголовки plack-запроса

=cut

foreach my $part (qw/content headers/) {
    has $part => (is => 'ro', lazy => 1, default => sub {
        shift->_plack_request->$part
    })
}

=head2 http_host

    Id запроса (reqid plack-запроса)

=cut

has http_host => (is => 'ro', isa => 'Str', lazy => 1, default => sub {
    http_server_host(shift->_plack_request, no_port => 1);
});

=head2 id

    Id запроса (reqid plack-запроса)

=cut

has id => (is => 'ro', isa => 'Str', lazy => 1, default => sub {
    shift->_plack_request->env->{reqid}
});

=head2 remote_ip

    IP клиента

=cut

has remote_ip => (is => 'ro', isa => 'Str', lazy => 1, default => sub {
    shift->_plack_request->address
});

=head2 from_internal_network

    true если запрос из внутренней сети (ip-адрес клиента принадлежит одной из
    внутренних сетей)

=cut

has from_internal_network => ( is => 'ro', isa => 'Str', lazy => 1, default => sub {
    my $remote_ip = shift->remote_ip;
    is_ip_in_list($remote_ip, $Settings::INTERNAL_NETWORKS) ? 1 : 0;
});

=head2 locale

    Локаль указанная в запросе (уже обработанная)

=cut

has locale => (is => 'ro', isa => 'Str', lazy => 1 , default => sub {
    my $self = shift;
    foreach my $lang (parse_http_accept_language( $self->header('Accept-Language') ) ) {
        if($lang =~ /^(\w+)\-/) {
            $lang = $1;
        }
        return $lang if Yandex::I18n::is_valid_lang($lang);
    }
    return $self->default_locale;
});

=head2 token

    Токен

=cut

has token => (is => 'ro', isa => 'Str', lazy => 1, default => sub {
    my $self = shift;
    my $auth_header = $self->header('Authorization') or return '';
    $auth_header =~ /Bearer\s+(.+)\s*$/ or return '';
    return $1;
});

=head2 is_token_persistent

    является ли токен persistent-ым

=cut

has is_token_persistent => (is => 'ro', isa => 'Bool', lazy => 1, default => sub {
    my $self = shift;
    return (lc( $self->header('Token-Type') || '' ) eq 'persistent') ? 1 : 0;
});

=head2 fake_login

    Логин для fake-авторизации, в случае если указан весь запрос будет
    происходить от имени указанного пользователя

=cut

has fake_login => (is => 'ro', isa => 'Str', lazy => 1, default => sub {
    shift->_login_from_header('Fake-Login')||'';
});

=head2 login

    Логин из заголовка запроса, возвращается сразу нормализованным

=cut

has login => (is => 'ro', isa => 'Maybe[Str]', lazy => 1, default => sub {
    shift->_login_from_header('Client-Login');
});

sub _login_from_header {
    my ($self, $header) = @_;
    my $raw_login = $self->header($header) or return;
    return normalize_login($raw_login);
}

=head2 use_operator_units

    Возвращает строковое значение 'true', 'false' или 'auto', указывающее, списывать ли баллы с оператора (определяется по HTTP-заголовку Use-Operator-Units)

=cut

has use_operator_units => (is => 'ro', isa => enum([qw/true false auto/]), lazy => 1, default => sub {
    my $self = shift;
    my $value;
    if (my $header_value = $self->header('Use-Operator-Units')) {
        $header_value = lc($header_value);
        if ($header_value eq 'true' || $header_value eq 'auto') {
            $value = $header_value;
        }
    }
    return $value // 'false';
});

has prefer_perl_implementation => (is => 'ro', isa => 'Bool', lazy => 1, default => sub {
    my $self = shift;
    my $value = $self->header('X-Prefer-Perl-Implementation');
    return defined $value && lc($value) eq "true" ? 1 : 0;
});

sub BUILDARGS {
    my ($class, $plack_request, $services) = @_;

    return {
        _plack_request => $plack_request,
        # нужно для десериализации тела запроса
        _services => $services,
        parsing_error => '',
    };
}

sub BUILD {
    my $self = shift;
    Yandex::I18n::init_i18n($self->locale);
    try {
        $self->_parsed
    } catch { # сохраняем ошибку, или формируем и сохраняем
        my $error = $_;
        if(UNIVERSAL::isa($error, 'Direct::Defect')) { # умерли сами
            $self->error($error);
        } elsif (UNIVERSAL::isa($error, 'API::Exception::IllegalRequest')) {
            $self->error( $error->reason );
        } else {
            warn "unexpected error while request parsing: $error";
            $self->error(error_BadRequest());
        }
    };
}

=head2 trace

    Объект трейсинга

=cut

sub trace { shift->_plack_request->env->{trace} }

=head2 service_operation_check_or_die($operation)

    Проверяет наличие операции в текущем сервиcе и заданность самого сервиса, в
    случае несоответствий кидает соотв. эксепшн

=cut

sub service_operation_check_or_die {
    my $self = shift;
    my $operation = shift;

    die error_BadRequest(iget('Сервис не найден'))
        unless defined $self->service_name;

    $self->_services->service_operation_exists($self->service_name, $operation)
        or die error_OperationNotFound;
}

=head2 header($header_name)

    По имени http-заголовка возрващает его значение

=cut

sub header {
    my ($self, $header_name) = @_;
    return $self->headers->header($header_name);
}

=head2 is_use_operator_units_true_or_auto

    Возвращает 1 в случае, если для запроса в заголовке Use-Operator-Units задано значение true или auto

=cut

sub is_use_operator_units_true_or_auto {
    my ($self) = @_;
    return $self->use_operator_units eq 'true' || $self->use_operator_units eq 'auto' ? 1 : 0;
}

# redefine _parsed params and operation

__PACKAGE__->meta->make_immutable();
1;
