package API::Request::SOAP;

use strict;
use warnings;
use utf8;

use Try::Tiny;

use Yandex::I18n;

use API::Request::SOAPAction;

use Mouse;
extends 'API::Request::XML';

use Direct::Errors::Messages;

use API::WSDL::ErrorsTranslation;

=pod

    $Id$

=head1 NAME

    API::Request::SOAP

=head1 SYNOPSIS

    # см базовый класс L<API::Request>

=head1 DESCRIPTION

    Класс для работы с SOAP-запросом, наследован от API::Request

=head1 METHODS

=cut

has protocol => (is => 'ro', isa => 'Str', default => sub { 'soap' });

# имя сервиса из HTTP заголовка SOAPAction должно соответствовать текущему uri
has service_name => (is => 'ro', isa => 'Str', default => sub {
    my $self = shift;
    my $uri_service_name = $self->SUPER::service_name;
    my $soap_action_service_name = $self->soap_action->service or return '';
    return ( $uri_service_name eq $soap_action_service_name ) ? $uri_service_name : '';
});


=head2 soap_action

    Почищенный SOAPAction из HTTP-заголовка

=cut

has soap_action => (is => 'ro', isa => 'Object', lazy => 1, default => sub {
    my $self = shift;

    my $soap_action = $self->header_soap_action;
    $soap_action =~ s/^["']//;
    $soap_action =~ s/["']$//;
    return $self->{soap_action} //= API::Request::SOAPAction->new(
        $soap_action
    );
});


=head2 header_soap_action

    SOAPAction из HTTP-заголовка, в сыром виде

=cut

sub header_soap_action { shift->header('SOAPAction')||''; }

has operation => (is => 'ro', isa => 'Maybe[Str]', lazy => 1, default => sub {
    shift->soap_action->operation
});

sub _parse_request {
    my ($self) = @_;

    die error_BadRequest(iget('Заголовок SOAPAction не задан'))
        unless $self->header_soap_action;

    $self->service_operation_check_or_die($self->operation);

    my ($header, $body, $error);
    try {
        ($header, $body)= $self->_services->request_from_xml(
            $self->service_name,
            $self->operation,
            $self->content
        );
    } catch {
        die $self->xml_compile_parsing_exception_to_defect($_);
    };

    die error_BadRequest(iget('Отсутствует тело SOAP-запроса')) unless $body;

    return {
        header => $header,
        body => $body
    };
}

has params => (is => 'ro', isa => 'HashRef', lazy => 1, default => sub {
    shift->_parsed->{body}
});

__PACKAGE__->meta->make_immutable();

1;
