package API::Request::XML;

use Direct::Modern;

use Try::Tiny;

use Yandex::I18n;

use Mouse;
extends 'API::Request';

use XML::Compile::Schema;

use API::WSDL::ErrorsTranslation;
use Direct::Errors::Messages;

use API::WSDLFiles qw/xsd_definitions/;

=pod

    $Id$

=head1 NAME

    API::Request::XML

=head1 SYNOPSIS

    # см базовый класс L<API::Request>

=head1 DESCRIPTION

    Класс для работы с XML-запросом, наследован от API::Request

=head1 METHODS

=cut

use constant API_VERSION => 5;

my ( %readers_by_format, %writers_by_format );

=head2 preload_xsd_readers_and_writers

Инициализировать модуль: загрузить xsd-файлы, нужные для валидации запросов,
и превратить их в нужные структуры памяти.

Надо вызывать из скрипта, который инициализирует веб-сервер для обработки запросов,
например, из модуля, который подгружается в конфиге Apache.

=cut

sub preload_xsd_readers_and_writers {
    my $yandex_schema = XML::Compile::Schema->new( [ grep {/\b(reports|general)\b/} xsd_definitions(API_VERSION) ] );
    $readers_by_format{yandex} = $yandex_schema->compile( READER => '{http://api.direct.yandex.com/v5/reports}ReportDefinition' );
    $writers_by_format{yandex} = $yandex_schema->compile( WRITER => '{http://api.direct.yandex.com/v5/reports}reportDownloadError',
        prefixes => { 'http://api.direct.yandex.com/v5/reports' => 'reports' } );

    # TODO заполнить $readers_by_format{google}, там надо будет reportdefinition.xsd и другой elem
}

=head2 get_xsd_writer

Возвращает процедуру, которая сможет записать данные запроса в переданный XML::Document.

TODO: по месту вызова всегда $format = 'yandex', надо починить

=cut

sub get_xsd_writer {
    my ($format) = @_;
    return $writers_by_format{$format};
}

has protocol => (is => 'ro', isa => 'Str', default => sub { 'xml' });

sub _parse_request {
    my ($self) = @_;

    my $parameters = $self->_plack_request->parameters;
    my ($request_data, $reader);
    if (exists $parameters->{__rdxml}) { # запрос в формате Google, передается в параметре __rdxml
        $reader = $readers_by_format{google};
        $request_data = $parameters->{__rdxml};
    } else {
        $request_data = $self->_plack_request->content;
        $reader = $readers_by_format{yandex};
    }
    die error_BadRequest( iget('Параметры отчета не заданы') ) unless $request_data;

    my $parsed;
    try {
        $parsed->{params} = $reader->(\$request_data);
    } catch {
        die $self->xml_compile_parsing_exception_to_defect($_);
    };

    $parsed->{method} = 'create' if $self->service_name eq 'reports';

    die error_BadRequest(iget('Операция не задана')) unless exists $parsed->{method};

    return $parsed;
}

has params => (is => 'ro', lazy => 1, default => sub {
    my $self = shift;
    return $self->_parsed_params($self->_parsed);
});

has operation => (is => 'ro', lazy => 1, default => sub {
    my $self = shift;
    return $self->_parsed_method($self->_parsed);
});

sub _parsed_method { $_[1]->{method} }
sub _parsed_params { $_[1]->{params} }

=head2 xml_compile_parsing_exception_to_defect($error)

    Обертка для ошибки парстинга, проверяет что ошибка -- действительно ошибка
    парсинга и возвращает соответствующий объект ошибки с правильной
    детализацией, иначе возвращает ошибку с пустой детализацией

=cut

sub xml_compile_parsing_exception_to_defect {
    my ($self, $error) = @_;
    if(ref $error) {
        return UNIVERSAL::isa($error, 'Direct::Defect') ?  $error : $self->_xml_compile_error_text($error)
    } else {
        warn "unexpected error while request parsing: $error";
        return error_BadRequest();
    }
}

=head2 _xml_compile_error_text($soap_error)

    Извлекает текст ошибки из ошибки парсинга XML::LibXML::Error, и преобразует
    в детализацию ошибки API, либо пустую строку

=cut

sub _xml_compile_error_text {
    my ($self, $error) = @_;
    die "wrong error type, must be XML::LibXML::Error" unless ref $error eq 'XML::LibXML::Error';
    my $message = ref $error->message
            ? $error->message->toString
            : $error->message;
    # Тут возможно будет преобразование $message в $translated
    my $translated = '';
    warn "unrecognized error $message" unless $translated;
    return error_BadRequest($translated);
}

__PACKAGE__->meta->make_immutable();

1;
