package API::RequestLog::Log;

use strict;
use warnings;
use utf8;

=pod

    $Id$

=head1 NAME

    API::RequestLog

=head1 SYNOPSIS

    my $log = API::RequestLog::Log->new();

    $log->add_request_id($env->{req_id});

    $log->add_common_request_info($request);

    # if parsed
    $log->add_parsed_request_info($request);

    $log->add_auth_info($auth);

    if($error) {
        $log->add_error_info($error);
    }

=head1 DESCRIPTION

    Класс в который сохраняются данные для логирования
    
    Так как поля, которые мы мспользуем для логов, становятся известны
    в разных местах кода, а мы, что бы ни случилось, должны записать
    в лог максимум полей
    Обект этого класса создается почти пустым, и по мере того как поля
    становятся известны мы его дозаполняем, и после завершения запроса
    передаем в функцию API::RequestLog::Logger::log, каторая запишет
    все в базу и syslog
    

=head1 METHODS

    Для заполненя объекта можно использовать ассессоры А можно методы которые
    могут заполнить группы свойств которые как правило становятся известны в
    одном месте кода методы вида add_smth_info

=head2 start_time

    Время запроса

=head2 vesrion

    Версия API

=head2 protocol

    Протокол запроса json/soap

=head2 application_id

    Id приложения

=head2 request_id

    Id запроса (Plack request_id)

=head2 operator_login

    Логин авторизованного пользователя

=head2 operator_uid

    Id авторизованного пользотеля

=head2 chief_rep_uid

    Id главного представителя владельца данных; т.е. главный представитель от логина, указанного в Client-Login. Client-Login равен аунтефицированному пользователю, если не указан.

=head2 cids

    ArrayRef массив кампаний над которыми (или над объектами которых) производится операция

=head2 error_status

    Id код ошибки

=head2 error_details

    Описание ошибки

=head2 units_stats

    Статистика по баллам [ портачено ; осталось от суточного лимита ; размер суточного лимита ]

=head2 units

    Потраченно баллов за последнюю операцию

=head2 params

    HashRef с параметрами запроса

=cut

use Time::HiRes;
use Mouse;
use Mouse::Util::TypeConstraints;

use API::Version;

has start_time => (
    is => 'ro',
    isa => 'Num',
    default => sub {Time::HiRes::time()},
    required => 1
);

has version => (
    is => 'ro',
    isa => 'Int',
    default =>sub {$API::Version::number},
    required => 1
);


has protocol => ( is => 'rw', isa => enum([qw/json soap xml/]) );

has method => ( is => 'rw', isa => 'Str', default => '' );

has operator_login => ( is => 'rw', isa => 'Str', default => sub {''} );

has application_id => ( is => 'rw', isa => 'Str', default => sub {''} );

has request_id => ( is => 'rw', isa => 'Int', default => sub {0} );

has operator_uid => ( is => 'rw', isa => 'Int', default => sub {0} );

has chief_rep_uid => ( is => 'rw', isa => 'Maybe[Int]', default => sub {undef} );

has error_status => ( is => 'rw', isa => 'Int', default => sub {0} );

has error_details => ( is => 'rw', isa => 'Str', default => sub {''} );

has units_stats => ( is => 'rw', isa => 'ArrayRef', default => sub {[]} );

has units => ( is => 'rw', isa => 'Int', default => sub {0} );

has units_spending_user_client_id => ( is => 'rw', isa => 'Int', default => sub {0} );

has cids => ( is => 'rw', isa => 'ArrayRef', default => sub {[]} );

has response_ids => ( is => 'rw', isa => 'Maybe[ArrayRef]', default => sub { undef } );

has params => ( is => 'rw', isa => 'HashRef', default => sub {{}} );

has response => ( is => 'rw', isa => 'Maybe[HashRef]', default => sub { undef } );

has response_headers => ( is => 'rw', isa => 'Maybe[HashRef]', default => sub { undef } );

has request_headers => ( is => 'rw', isa => 'Maybe[HashRef]', default => sub { undef } );

has error_details => ( is => 'rw', isa => 'Str', default => sub {''} );

has remote_ip => ( is => 'rw', isa => 'Str', default => sub {''} );

has error_object_count => (is => 'rw', isa => 'Int', default => sub { 0 } );
has warning_object_count => (is => 'rw', isa => 'Int', default => sub { 0 } );

=head2 add_error_info($auth)

    Логируем ошибку

=cut

sub add_error_info {
    my ($self, $error) = @_;
    $self->error_details($error->description);
    $self->error_status($error->code,);
}

=head2 add_units_info($spent, $balance, $limit)

    Логируем потраченные баллы
    $spent - потрачено за вызов последней операции
    $balance - осталось от суточного лимита
    $limit -> суточный лимит

=cut

sub add_units_info {
    my ($self, $spent, $balance, $limit) = @_;
    $self->units($spent);
    $self->units_stats([$spent, $balance, $limit]) if defined $balance;
}

=head2 add_auth_info($auth)

    Логируем авторизационные данные API::Authorization

=cut

sub add_auth_info {
    my ($self, $auth) = @_;
    $self->operator_login($auth->operator_login);
    $self->application_id($auth->application_id);
    $self->operator_uid($auth->operator_uid);
}

=head2 add_common_request_info($request)

    Записываем данные запроса, которые не зависят от результатов его парсинга,
    в method предварительно записываем имя сервиса, которое при корректном парсинге запроса дополняется именем операции,
    request_id рекомендуется логировать заранее, чтобы он был всегда

=cut

sub add_common_request_info {
    my ($self, $request) = @_;
    $self->protocol($request->protocol);
    $self->remote_ip($request->remote_ip);
    $self->method($request->service_name);
    return;
}

=head2 add_parsed_request_info($request)

    Добавляем данные запроса API::Request, зависящие от результатов его парсинга

=cut

sub add_parsed_request_info {
    my ($self, $request) = @_;
    $self->method($request->service_name.'.'.$request->operation);
    $self->params($request->params);
    return;
}

=head2 add_response_ids([$id, $id, ...])

=cut

sub add_response_ids {
    my ($self, $ids) = @_;
    $self->response_ids($ids);
}

=head2 add_error_object_count($count)

    Сохраняем в логе количество элементов ответа содержащих одну или более ошибок

=cut

sub add_error_object_count {
    my ($self, $errors_count) = @_;
    $self->error_object_count($errors_count||0);
}

=head2 add_warning_object_count($count)

    Сохраняем в логе количество элементов ответа содержащих одно и более предупреждений

=cut

sub add_warning_object_count {
    my ($self, $warnings_count) = @_;
    $self->warning_object_count($warnings_count||0);
}


__PACKAGE__->meta->make_immutable();

1;
