package API::RequestLog::Logger;

use strict;
use warnings;
use utf8;
use feature 'state';

=pod

    $Id$

=head1 NAME

    API::RequestLog

=head1 SYNOPSIS


=head1 DESCRIPTION

    Модуль для логирования запросов в API

=head1 METHODS

    log($log_object) Записывает логи в базу и syslog
        принимает - объект API::RequestLog::Log

=cut

use Time::HiRes;
use EnvTools;
use YAML qw(Dump Load);

use Yandex::Log;
use Yandex::TimeCommon;
use Settings;
use Tools;

use API::RequestLog::Log;

use Data::Dumper;

=head2 log

    Логирование запросов в API
    Записывает параметры запроса в syslog

=cut

sub log {
    my ($log) = shift;
    my $time = time();

    if ( $log->request_headers ) {
	$log->params( { %{$log->params}, ( 'request_headers' => $log->request_headers ) } );
    }

    my $mdata = {
        logtime => unix2mysql($time),
        cid => [ sort @{$log->cids} ],
        bid => 0, # в API5 не имеет смысла
        ip => $log->remote_ip,
        cmd => $log->method,
        runtime => defined $log->start_time ? (Time::HiRes::time() - $log->start_time) : 0,
        param => $log->params,
        http_status => $log->error_status,
        cluid => $log->chief_rep_uid,
        reqid => $log->request_id,
        uid => $log->operator_uid,
        host => $EnvTools::hostname,
        proc_id => $$,
        sleeptime => undef, # появится когда будем прикручивать тормозилку 
        error_detail => substr( $log->error_details || '', 0, 200 ),
        units => $log->units,

        units_stats => $log->units_stats,
        units_spending_user_client_id => $log->units_spending_user_client_id,

        api_version => $log->version,
        interface => $log->protocol,
        application_id => $log->application_id,

        error_object_count => $log->error_object_count,
        warning_object_count => $log->warning_object_count,
        response_headers => $log->response_headers,
    };

    if ( my $response_ids = $log->response_ids ) {
        $mdata->{response_ids} = $response_ids;
    }

    if ( my $response = $log->response ) {
        $mdata->{response} = $response;
    }

    _save_log_to_syslog($mdata);
}

=head2 _save_log_to_syslog

=cut

sub _save_log_to_syslog {
    my $mdata = shift;

    # короткий вариант syslog_prefix => $Settings::BETA_SYSLOG_PREFIX || 'PPCLOG' использовать нельзя,
    # т.к. переменная $Settings::BETA_SYSLOG_PREFIX есть только на бетах и в других местах при запуске unit-тестов
    # выдается предупреждение "used only once: possible typo"
    state $log_syslog;
    $log_syslog //= Yandex::Log->new(
        use_syslog    => 1,
        no_log        => 1,
        syslog_prefix => $Settings::BETA_SYSLOG_PREFIX ? $Settings::BETA_SYSLOG_PREFIX : 'PPCLOG',
        log_file_name => 'ppclog_api.log'
    );

    $mdata->{param} = Tools::trim_log_data($mdata->{param});
    $mdata->{log_type} = 'api';

    eval {
        $log_syslog->out($mdata);
    };
    if( $@ ) {
        print STDERR Dumper ['$@',$@];
    }
}

1;
