package API::Service::CampaignsAvailabilityChecker;

use Direct::Modern;

=pod

=encoding utf8

    $Id$

=head1 NAME

    API::Service::CampaignsAvailabilityChecker

=head1 SYNOPSIS

    my $checker = API::Service::CampaignsAvailabilityChecker->new(
        subclient_uid => $subclient_uid,
        cids => \@cids,
        supported_camp_kind => 'api5_edit',
        subclient_currency => $subclient_currency,
        error_not_found => error_NotFound_Campaign(),
        error_not_supported => error_NotSupported_CampaignType(),
        error_is_archived => error_BadStatus_ArchivedCampaign(),
    );

    if (my $error = $checker->get_error($cid)) {
        $item->add_error($error);
    }

=head1 DESCRIPTION

    Класс для проверки доступности кампаний сабклиенту. Проверяет является ли
    указанный сабклиент владельцем кампании, находится ли кампания в архиве, обрабатывается ли тип кампании в API5 вообще,
    поддерживается ли тип кампании в заданном надтипе.

    Не выполняет проверку прав в RBAC, она должна делаться отдельно.

=head1 METHODS

=cut

use Mouse;

use Yandex::DBTools;

use Settings;

use API::Service::CampaignsAvailabilityChecker::ResultItem;
use API::Service::CampaignsAvailabilityChecker::ResultItem::NotFound;

has subclient_uid => (is => 'ro', isa => 'Int', required => 1);
has subclient_currency => (is => 'ro', isa => 'Str', default => '');
has ppc_shard => ( is => 'ro', lazy => 1, default => sub { PPC(uid => shift->subclient_uid) });
has cids => (is => 'ro', isa => 'ArrayRef[Int]', required => 1);
has supported_camp_kind => (is => 'ro', isa => 'Str', required => 1); # название надтипа поддерживаемых типов кампаний
has pass_archived => (is => 'ro', isa => 'Bool', default => 0);
has items => (is  => 'rw', isa => 'HashRef', lazy => 1, builder => 'populate_items', reader => '_items');
has campaign_info_map => (is => 'ro', isa => 'HashRef', lazy => 1, builder => 'get_campaign_info_map_from_db', reader => '_campaign_info_map');
has available_cids => (isa => 'ArrayRef', lazy => 1, default => sub { my ($self) = @_; [ grep { $self->is_available($_) } @{$self->cids} ] }, reader => 'get_available_cids', auto_deref => 1); # список cid'ов доступных кампаний
has unavailable_cids_map => (isa => 'HashRef', lazy => 1, default => sub { my ($self) = @_; return { map { $_ => 1 } grep { !$self->is_available($_) } @{$self->cids} } }, reader => 'get_unavailable_cids_map', auto_deref => 1); # хэш, где в качестве ключей cid'ы недоступных кампаний
has error_not_found => (is => 'rw', isa => 'Maybe[Direct::Defect]', predicate => 'has_error_not_found');
has error_not_supported => (is => 'rw', isa => 'Maybe[Direct::Defect]', predicate => 'has_error_not_supported');
has error_is_archived => (is => 'rw', isa => 'Maybe[Direct::Defect]', predicate => 'has_error_is_archived');

=head2 get_campaign_info_map_from_db()

    Получаем из БД данные по кампаниям

=cut

sub get_campaign_info_map_from_db {
    my ($self) = @_;

    return get_hashes_hash_sql($self->ppc_shard,
        [
            "SELECT cid, uid, type, archived, source, IFNULL(currency, 'YND_FIXED') currency FROM campaigns",
            WHERE => {
                cid => $self->cids,
                'statusEmpty__not_in' => ['Yes']
            }
        ]
    );
}

=head2 populate_items()

    Формируем внутренние объекты проверок в зависимости от данных полученных из БД

=cut

sub populate_items {
    my ($self) = @_;

    my %items;
    foreach my $cid (@{$self->cids}) {
        my $campaign_info = $self->_campaign_info_map->{$cid};
        if ($campaign_info) {
            $items{$cid} = API::Service::CampaignsAvailabilityChecker::ResultItem->new(
                subclient_uid => $self->subclient_uid,
                subclient_currency => $self->subclient_currency,
                pass_archived => $self->pass_archived,
                supported_camp_kind => $self->supported_camp_kind,
                %$campaign_info
            );
        } else {
            $items{$cid} = API::Service::CampaignsAvailabilityChecker::ResultItem::NotFound->new(cid => $cid);
        }
    }

    return \%items;
}

=head2 get_result($cid)

    Возвращает объект результата проверки доступности указанной в $cid кампании

=cut

sub get_result {
    my ($self, $cid) = @_;
    return $self->_items->{$cid};
}

=head2 is_available($cid)

    Возвращает доступность указанной в $cid кампании

=cut

sub is_available {
    my ($self, $cid) = @_;
    return $self->get_result($cid)->is_available;
}

=head2 get_error($cid)

    Возвращает объект ошибки из заданного набора соответствующий результату проверки доступности кампании с указанным $cid

=cut

sub get_error {
    my ($self, $cid) = @_;

    my $item = $self->get_result($cid);

    return $self->error_not_found if $self->has_error_not_found && !$item->is_visible;
    return $self->error_not_supported if $self->has_error_not_supported && !$item->is_supported;
    return $self->error_is_archived if $self->has_error_is_archived && !$item->pass_archived && $item->is_archived;

    return;
}

__PACKAGE__->meta->make_immutable;

1;
