package API::Service::CampaignsAvailabilityChecker::ResultItem;

use Direct::Modern;

=pod

=encoding utf8

    $Id$

=head1 NAME

    API::Service::CampaignsAvailabilityChecker::ResultItem

=head1 SYNOPSIS

    my $checker = API::Service::CampaignsAvailabilityChecker::ResultItem->new(
        cid => 123,
        uid => 42,
        type => 'text',
        archived => 'Yes',
        currency => 'YND_FIXED',
        pass_archived => 0,
        subclient_uid => 42,
        subclient_currency => 'YND_FIXED',
        supported_camp_kind => 'api5_edit',
    );

    if (!$camp_check->is_available) {
        $item->add_error(
            $camp_check->get_error(
                not_found => error_NotFound_Campaign(),
                not_supported => error_NotSupported_CampaignType(),
                archived => error_BadStatus_ArchivedCampaign(),
            )
        );
    }

=head1 DESCRIPTION

    Класс для хранения результатов проверки доступности кампании клиенту.

    Наиболее важные методы:
        is_visible - клиенту кампания видна (найдена и принадлежит), но может не поддерживаться в API
        is_supported - кампания в принципе поддерживается в API5
        is_available - видна И поддерживается в API5

=cut

use Campaign::Types;

use Mouse;
use Moose::Util::TypeConstraints;
extends 'API::Service::CampaignsAvailabilityChecker::ResultItem::NotFound';

has uid => (is => 'ro', isa => 'Int', required => 1);
has type => (is => 'ro', isa => 'Str', required => 1);
has source => (is => 'ro', isa => 'Str', required => 1);
has archived => (is => 'ro', isa => enum(['Yes', 'No']), required => 1);
has currency => (is => 'ro', isa => 'Str', required => 1);
has pass_archived => ( is => 'ro', isa => 'Bool', default => 0); # считать архивные кампании доступными
has subclient_uid => (is => 'ro', isa => 'Int', required => 1);
has subclient_currency => (is => 'ro', isa => 'Str', default => '');
has supported_camp_kind => (is => 'ro', isa => 'Str', required => 1);

has is_found => ( is => 'ro', isa => 'Bool', lazy => 1, default => sub { shift->cid ? 1 : 0 } ); # кампания найдена в клиентском шарде
has is_archived => ( is => 'ro', isa => 'Bool', lazy => 1, default => sub { shift->archived eq 'Yes' ? 1 : 0 } ); # кампания перемещена в архив
has is_supported => ( is => 'ro', isa => 'Bool', lazy => 1,
    default => sub { my ($self) = @_; (camp_kind_in(type => $self->type, $self->supported_camp_kind) && $self->source ne 'uac' && $self->source ne 'widget' && $self->source ne 'zen') ? 1 : 0 });
    # кампания входит в заданный надтип и не создана в Мастере кампаний или Дзеном
has is_kind_api5_visible => ( is => 'ro', isa => 'Bool', lazy => 1, default => sub { camp_kind_in(type => shift->type, 'api5_visible') ? 1 : 0 }); # кампания обрабатывается в API5
has is_owned => ( is => 'ro', isa => 'Bool', lazy => 1, default => sub { my ($self) = @_; $self->uid == $self->subclient_uid ? 1 : 0 }); # сабклиент является владельцем кампании
has is_currency_matched => ( is => 'ro', isa => 'Bool', lazy => 1, default => sub { my ($self) = @_; $self->currency eq $self->subclient_currency ? 1 : 0 }); # валюта кампании совпадает с валютой клиента
has is_visible => ( is => 'ro', isa => 'Bool', lazy => 1, default => sub { my ($self) = @_; $self->is_found && $self->is_kind_api5_visible && $self->is_owned ? 1 : 0 }); # клиент имеет какой-то доступ к кампании
has is_available => ( is => 'ro', isa => 'Bool', lazy => 1, default => sub { my ($self) = @_; $self->is_visible && ($self->pass_archived || !$self->is_archived) && $self->is_supported ? 1 : 0 }); # клиент имеет указанный тип доступа к кампании

__PACKAGE__->meta->strict_constructor(1);
__PACKAGE__->meta->make_immutable;

1;
