package API::Service::ResultSet::Base;

use Direct::Modern;

=pod

    $Id$

=head1 NAME

    API::Service::ResultSet::Base

=head1 SYNOPSIS

    use_example

=head1 DESCRIPTION

    Класс для формирования ответов API-сервисов состоящий из списка объектов со
    списком ошибок и ворнингов для каждого.
    Не предназначен для использования на прямую, нужен для как базовый для
    написания своих классов

=head1 METHODS

=head2 new(@list)

    По списку значений (любого типа), создает ResultSet, являющийся блеснутой
    ссылкой на массив объектов API::Service::ResultSet::Item (может быть
    переопределен в item_class)

=head2 item_class

    Возвращает имя класса объектов сета, объект сам решает как ему хранить
    значения из списка переданного в конструктор

=head2 ids

    Id объектов, список

=head2 has_errors($id)

    True если объект по такому Id содержит ошибки

=head2 has_warnings($id)

    True если объект по такому Id содержит предупреждения

=head2 first

    Первый объект сета

=head2 first_ok

    Первый объект сета без ошибок

=head2 has_ok

    True если сет содержит объекты без ошибок

=head2 list

    Все объекты сета списком

=head2 list_ok

    Объекты сета списком, без ошибок

=head2 count

    Кол-во объектов в списке

=head2 count_ok

    Кол-во объектов в списке без ошибок


=head2 count_failed

    Кол-во объектов списка содержащих ошибку

=head2 list_failed

    Список объектов содержащих ошибку

=head2 list_ok_ids

    Список Id объектов без ошибок

=head2 list_failed_ids

    Список Id объектов содержащих ошибку

=head2 list_warned

    Список объектов содержащих предупреждение

=head2 item_arrays_by_ids

    Возвращают ссылку на хэш содержащий массивы объектов по Id в виде ключа.
    Id может быть не уникальным, например в пользовательском запросе, и данный метод
    помогает пометить ошибкой дубликаты.

=head2 list_id_duplicated_items

    Список объектов сета, чей id внутри сета не уникален

=head2 add_error_for_id_dups($error)

    Добавляет Direct::Defect ошибку всем объектам сета с неуникальным Id.

=head2 add_warning_for_id_dups($error)

    Добавляет Direct::Defect предупреждение всем объектам сета с неуникальным Id.

=head2 add_error($error, @ids)

    Добавляем Direct::Defect ошибку всем объектам с Id из переданного списка @ids

=head2 add_warning($error, @ids)

    Добавляем Direct::Defect предупрежедние всем объектам с Id из переданного списка @ids

=head2 add_error_when($error, $predicate, $without_errors)

    Добавляет указанную ошибку типа Direct::Defect всем объектам
    (или всем объектам без ошибок - зависит от значения флага
    without_errors), для которых указанная функция вернет истину.

=head2 add_warning_when($warning, $predicate, $without_warnings)

    Добавляет указанное предупреждение типа Direct::Defect всем объектам
    (или всем объектам без предупреждений - зависит от значения флага
    without_warnings), для которых указанная функция вернет истину.

=head2 get($id)

    Возвращает объект с заданным $id в случае дупликатов возвращается первый

=head2 id_item_to_xml($id, $item)

    Возвращает структуру ActionResult для объекта, массив которых вернем клиенту

=head2 import

    Импорт чтобы использовать как use API::Service::ResultSet::Base, без use base;

=head2 add_isa_to_subclass($pkg, $subclass)

    Добавляет пакет в ISA подкласса, для использования в import

=cut

use List::Util qw//;

use API::Error::ToExceptionNotification qw/action_errors action_warnings/;

use API::Service::ResultSet::Item;

sub ids {
    my $self = shift;
    return
        map { $self->_id_from_object($_->object) }
        $self->list;
}

sub _id_from_object {
    my $self = shift;
    my $object = shift;
    return $object->{Id};
}

sub has_errors { # $id -> Bool
    my $self = shift;
    my $id = shift;
    return $self->get($id)->has_errors;
}

sub has_warnings { # $id -> Bool
    my $self = shift;
    my $id = shift;
    return $self->get($id)->has_warnings;
}

sub list { @{$_[0]} }

sub count { $#{$_[0]}+1 }

sub count_failed {
    scalar( shift->list_failed )
}

sub list_failed { # @list
    my $self = shift;
    return grep {
        $_->has_errors
    } @$self;
}

sub list_failed_ids { # @list
    my $self = shift;
    return map {
        $self->_id_from_object($_->object)
    } $self->list_failed;
}

sub list_warned { # @list
    my $self = shift;
    return grep {
        $_->has_warnings
    } @$self;
}

sub count_ok {
    scalar( shift->list_ok )
}

sub list_ok { # @list
    my $self = shift;
    return grep {
        !$_->has_errors
    } @$self;
}

sub list_ok_ids { # @list
    my $self = shift;
    return map {
        $self->_id_from_object($_->object)
    } $self->list_ok;
}

sub item_arrays_by_ids {
    my $self = shift;
    my $by_id;
    foreach my $i ($self->list) {
        my $id = $self->_id_from_object($i->object);
        next unless defined $id;
        push @{ $by_id->{$id} //= [] }, $i;
    }
    return $by_id;
}

sub list_id_duplicated_items {
    my $self = shift;
    my $by_id = $self->item_arrays_by_ids;
    my @dups;
    while( my ($id, $items) = each %$by_id ) {
        push @dups, @$items if $#$items > 0;
    }
    return @dups;
}

sub add_error_for_id_dups {
    my $self = shift;
    my $error = shift;

    $_->add_error($error) foreach $self->list_id_duplicated_items;
}

sub add_error { # ($error, @ids) -> undef
    my $self = shift;
    my $error = shift or die "error not set";
    $error->code or die "not a error object";
    my @ids = @_;
    foreach(@ids) {
        $self->get($_)->add_error($error);
    }
}

sub add_warning_for_id_dups {
    my $self = shift;
    my $warning = shift;

    $_->add_warning($warning) foreach $self->list_id_duplicated_items;
}

sub add_warning { # ($warning, @ids) -> undef
    my $self = shift;
    my $warning = shift or die "warning not set";
    $warning->code or die "not a warning object";
    my @ids = @_;
    foreach(@ids) {
        $self->get($_)->add_warning($warning);
    }
}

sub add_error_when {
    my ( $self, $error, $predicate, $without_errors ) = @_;

    croak 'no error given'     unless $error;
    croak 'no predicate given' unless ( $predicate and ref( $predicate ) eq 'CODE' );

    my @items = $without_errors ? $self->list_ok : $self->list;
    foreach my $item ( @items ) {
        my $obj = $item->object;
        $item->add_error( $error ) if $predicate->( $self->_id_from_object( $obj ), $obj );
    }
}

sub add_warning_when {
    my ( $self, $warning, $predicate, $without_warnings ) = @_;

    croak 'no warning given'   unless $warning;
    croak 'no predicate given' unless ( $predicate and ref( $predicate ) eq 'CODE' );

    my @items = $self->list;
    if ( $without_warnings ) {
        @items = grep { $_->has_warnings } @items;
    }

    foreach my $item ( @items ) {
        my $obj = $item->object;
        $item->add_warning( $warning ) if $predicate->( $self->_id_from_object( $obj ), $obj );
    }
}

sub get {
    my $self = shift;
    my $id = shift;
    foreach(@$self) {
        return $_ if
            $self->_id_from_object($_->object) eq $id;
    }
    return; # not found
}

sub item_class { 'API::Service::ResultSet::Item' }

sub new {
    my $class = shift;
    my @list = @_;
    my $idx = 0;

    die 'Item could not be an array' if List::Util::any { ref $_ eq 'ARRAY' } @list;

    return bless [
        map { $class->item_class->new( $idx++, $_ ) } @list
    ], $class
}

sub import {
    my $pkg = shift;
    my $caller = caller(0);
    return if $pkg ne __PACKAGE__;

    $pkg->add_isa_to_subclass($caller);
}

sub add_isa_to_subclass {
    my $pkg = shift;
    my $subclass = shift;
    no strict 'refs';
    push @{"${subclass}::ISA"}, $pkg;
}

sub first { @{$_[0]}[0] }

sub first_ok {
    my $self = shift;
    return List::Util::first {
        !$_->has_errors
    } @$self;
}

sub has_ok {
    return shift->first_ok ? 1 : 0; 
}

sub id_item_to_xml {
    my ($self, $id, $item) = @_;
    my $r = {};

    # you either die young by error, or live long to have an id
    die 'no id or errors' unless $id || $item->has_errors;

    # if there is a error we hiddin id even if we have one
    $r->{Id} = $id if $id && !$item->has_errors;
    $r->{Errors} = action_errors($item->list_errors) if $item->has_errors;
    $r->{Warnings} = action_warnings($item->list_warnings) if $item->has_warnings;

    return $r;
}

1;

__END__
