package API::Services;

use strict;
use warnings;
use utf8;

use Encode;
use Hash::Util qw/lock_hash_recurse/;

use Module::Load qw/load/;
use Module::Loaded;

use XML::Compile::SOAP11::Server;
use XML::Compile::WSDL11;
use XML::Compile::Util qw/unpack_type/;

use Yandex::Trace;

use WSDL::JSON::Schema;

# настроечная информация для XSD-сервисов, которую нельзя извлечь из WSDL
my %XSD_SERVICE_MAP = (
    reports => { class_name => 'Reports', operations => { create => {} } },
);
lock_hash_recurse(%XSD_SERVICE_MAP);

sub new {
    my ($class, $wsdl_files, $xsd_definitions) = @_;

    my $profile = Yandex::Trace::new_profile('api:services:constructor');

    my $wsdl11 = XML::Compile::WSDL11->new(
        undef,
        opts_rw => {
            sloppy_floats => 1,
            sloppy_integers => 1
        }
    );
    $wsdl11->addWSDL($_) foreach(@$wsdl_files);
    $wsdl11->importDefinitions($_) foreach(@$xsd_definitions);

    my $soap_server = XML::Compile::SOAP11::Server->new(schemas => $wsdl11);

    my @ops = $wsdl11->operations;

    my $services = { map { $_ => $XSD_SERVICE_MAP{$_}->{class_name} } keys %XSD_SERVICE_MAP };
    my $operations = { map { $_ => $XSD_SERVICE_MAP{$_}->{operations} } keys %XSD_SERVICE_MAP };

    foreach my $operation (@ops) {
        my $action_url = $operation->action;
        $action_url =~ s/\/$//;
        my @path = split('/', $action_url);
        my $service_uri = $path[-2];
        my $operation_name = $path[-1];

        my $in = $operation->{input_def}{body}{parts}[0]{element};
        my $out = $operation->{output_def}{body}{parts}[0]{element};
        my ($in_type, $in_element) = unpack_type($in);
        my ($out_type, $out_element) = unpack_type($out);
        my $operation_info = {
            input_body => [ $in_element => $in ],
            output_body => [ $out_element => $out ],
            action => $action_url,
        };

        if(my $in_header = $operation->{input_def}{header}) {
            my $in_header = $operation->{input_def}{header}[0]{parts}[0]{element};
            my ($in_header_type, $in_header_element) = unpack_type($in_header);
            $operation_info->{input_header} = [ $in_header_element => $in_header ];
        }
        $operations->{ $service_uri }{ $operation_name } = $operation_info;

        $services->{$service_uri} = $operation->serviceName;
    }

    return bless {
        soap_server => $soap_server,
        services => $services,
        operations => $operations
    }, $class;
}

=head2 namespaces

    $self->soap_server->schemas->namespaces
    см. XML::Compile::Schema::NameSpaces

=cut

sub namespaces { $_[0]->soap_server->schemas->namespaces }

sub json_schemas {
    my $self = shift;
    my $nss = $self->namespaces;
    unless($self->{json_schemas}) {
        my $profile = Yandex::Trace::new_profile('api:services:json_schemas_loading');
        $self->map_thru_service_operation_ns_parts(sub {
            my ($service, $operation, $ns, $request, $response) = @_;
            my $schema = WSDL::JSON::Schema->new($nss, $ns);
            $self->{json_schemas}{$service}{$operation}{request} = $schema->build($request);
            $self->{json_schemas}{$service}{$operation}{response} = $schema->build($response);
        });
    }

    return $self->{json_schemas};
}

=head2 map_thru_service_operation_ns_parts($sub)

    Для каждого неймспейса, имени элемента запроса и имени элемента ответа,
    каждого сервиса каждой операции вызывает $sub->($service, $operation, $ns, $request, $response)

=cut

sub map_thru_service_operation_ns_parts {
    my ($self, $sub) = @_;
    foreach my $service (sort keys %{$self->_services}) { # sort нужен если делаем print в $sub
        next if exists $XSD_SERVICE_MAP{$service};
        foreach my $operation ( sort keys %{$self->_service_operations($service)} ) {
            my ($request, $ns) = $self->ns_name_split(
                $self->request_body_element_ns( $service, $operation )
            );
            my ($response) = $self->ns_name_split(
                $self->response_body_element_ns( $service, $operation )
            );
            $sub->($service, $operation, $ns, $request, $response);
        }
    }
}

sub json_schema_request {
    my ($self, $service_name, $operation) = @_;
    return $self->json_schemas->{$service_name}->{$operation}->{request};
}

sub json_schema_response {
    my ($self, $service_name, $operation) = @_;
    return $self->json_schemas->{$service_name}->{$operation}->{response};
}

sub ns_name_split {
    $_[1] =~ /^\{(.+?)\}(\w+)$/ or die "wrong element $_[1]";
    return ($2, $1);
}

sub soap_server { shift->{soap_server} }

sub service_operation_exists {
    my ($self, $service_uri, $operation_name) = @_;
    return $self->service_exists($service_uri)
        && $self->_service_operations($service_uri)->{$operation_name};
}

sub service_exists {
    my ($self, $service_uri) = @_;
    return $self->_services->{$service_uri};
}

sub _services { shift->{services} }

sub _service_operations {
    my ($self, $service_uri) = @_;
    return $self->{operations}{$service_uri};
}

sub operation {
    my ($self, $service, $operation) = @_;
    my $operations = $self->_service_operations($service) or return;
    return $operations->{$operation};
}

sub operation_by_input_body_name {
    my ($self, $service, $input_body_name) = @_;
    my $operations = $self->_service_operations($service) or return;
    foreach my $o (keys %$operations) {
        return $o if $operations->{$o}->{input_body}->[0] eq $input_body_name;
    }
    return;
}

=head2 service_name_by_uri($uri)

    Получаем имя сервиса по $uri, вытащенному из HTTP запроса

=cut

sub service_name_by_uri {
    my $self = shift;
    my $uri = shift;

    return $self->_services->{$uri};
}

sub init {
    my $self = shift;
    my $service_name = shift;

    unless($self->{loaded}->{$service_name}) {
        my $service_module = "API::Service::$service_name";
        load $service_module unless is_loaded($service_module);
        $self->{loaded}->{$service_name} = $service_module->new($service_name);
    }

    return $self->{loaded}->{$service_name};
}

sub response_to_xml {
    my $self = shift;
    my $service = shift;
    my $operation = shift;
    my $params = shift;
    die 'wrong params' unless defined $params;

    return Encode::decode('utf8',
        $self->soap_server->compileMessage('SENDER',
            body => $self->_service_operations($service)->{$operation}{output_body}
        )->($params, 'UTF-8')->toString
    );
}

sub request_from_xml {
    my $self = shift;
    my $service = shift;
    my $operation = shift;
    my $xml = shift or die 'wrong xml';
    my $op_info = $self->operation($service, $operation);
    my $body = $op_info->{input_body};
    my $body_element = $body->[0];
    my $request = $self->soap_server->compileMessage('RECEIVER',
            body => $body
    )->($xml);
    return (undef, $request->{$body_element});
}

sub respones_from_xml {
    my $self = shift;
    my $service = shift;
    my $operation = shift;
    my $xml = shift or die 'wrong xml';
    return $self->soap_server->compileMessage('RECEIVER',
            body => $self->response_body($service, $operation)
    )->($xml);
}

sub _op_part {
    my $self = shift;
    my $service = shift;
    my $operation = shift;
    my $part = shift;
    return $self->operation($service, $operation)->{$part}
}

sub response_body {
    my $self = shift;
    return $self->_op_part(@_, 'output_body');
}

sub response_body_element {
    my $self = shift;
    return $self->response_body(@_)->[0];
}

sub response_body_element_ns {
    my $self = shift;
    return $self->response_body(@_)->[1];
}

sub request_body {
    my $self = shift;
    return $self->_op_part(@_, 'input_body');
}

sub request_body_element {
    my $self = shift;
    return $self->request_body(@_)->[0];
}

sub request_body_element_ns {
    my $self = shift;
    return $self->request_body(@_)->[1];
}

1;
