package API::Units::BucketManager;

use Direct::Modern;

=pod

=encoding utf8

    $Id$

=head1 NAME

    API::Units::BucketManager

=head1 SYNOPSIS

    my $bucket_manager = API::Units::BucketManager->new();
    $bucket_manager->req_id($req_id);
    $bucket_manager->auth($auth);
    $bucket_manager->subclient($subclient);
    $bucket_manager->create_bucket_for_failure();
    my $bucket = $bucket_manager->get_bucket();

=head1 DESCRIPTION

    Класс для формирования корзины, по данным которой будут списывать баллы.

=head1 METHODS

=head2 new()

    Конструктор без параметров

=head2 req_id($req_id)

    Идентификатор запроса.

=head2 _is_operator_failure($bool)

    $bool - true если запрос не удалось распарсить, флаг объясняющий отсутствие части данных в логе

=head2 is_server_fault($bool)

    Устанавливаем флаг из кода в случае возникновения серверной ошибки при обработке запроса

=head2 use_operator_units($use_operator_units)

    Строковое значение 'true', 'false', 'auto' в зависимости от переданного HTTP-заголовка Use-Operator-Units

=head2 auth($auth)

    Объект API::Authorization для получения данных по оператору и брендам

    Должно быть заполнено перед вызовом create_bucket()

=head2 client_login($client_login)

    Логин субклиента из поля Client-Login заголовка запроса

=head2 subclient($subclient_user)

    API::Authorization::User главный представитель сабклиента. Пользователь с
    данными которого ведется работа

=head2 get_bucket($bucket)

    API::Units::Bucket объект корзины баллов из которой в итоге будут списан баллы

    Должно быть заполнено перед вызовом get_log_data()

=cut

use Mouse;
use Mouse::Util::TypeConstraints;

use API::Units::Bucket;
use Units;

use constant TYPE_OPERATOR => 'operator';
use constant TYPE_OPERATOR_BRAND => 'operator brand';
use constant TYPE_SUBCLIENT => 'subclient';
use constant TYPE_SUBCLIENT_BRAND => 'subclient_brand';

has req_id => (is => 'rw', isa => 'Str');
has auth => (is => 'rw', isa => 'Maybe[API::Authorization]');
has client_login => (is => 'rw', isa => 'Maybe[Str]', default => undef);
has subclient => (is => 'rw', isa => 'Maybe[API::Authorization::User]', default => undef);
has _is_operator_failure => (is => 'rw', isa => 'Bool', default => 0);
has _is_bucket_switched_to_operator => (is => 'rw', isa => 'Bool', default => 0);
has is_server_fault => (is => 'rw', isa => 'Bool', default => 0);
has use_operator_units => (is => 'rw', isa => enum([qw/true false auto/]), default => 'false');
has _bucket => (is => 'rw', isa => 'Maybe[API::Units::Bucket]', reader => 'get_bucket', writer => '_set_bucket', predicate => 'has_bucket');
has _bucket_type => (is => 'rw', isa => enum([TYPE_OPERATOR, TYPE_OPERATOR_BRAND, TYPE_SUBCLIENT, TYPE_SUBCLIENT_BRAND]));
has _bucket_user => (is => 'rw', isa => 'API::Authorization::User');
has _units_stat => (is => 'rw', isa => 'Maybe[HashRef]', default => undef);
has _subclient_brand_user => (is => 'rw', isa => 'Maybe[API::Authorization::User]', default => undef);
has _operator_brand_user => (is => 'rw', isa => 'Maybe[API::Authorization::User]', default => undef);

=head2 create_bucket_for_operator_failure

    Создает объект корзины баллов для оператора/его бранда, используется при обработке ошибок оператора.
    Сохраняет корзину в поле _bucket, в _bucket_type записывает для какого клиента создана корзина.
    Падает при попытке повторного создания корзины.
    Возвращает созданный объект корзины.

=cut

sub create_bucket_for_operator_failure {
    my ($self) = @_;

    die "Bucket already exists" if $self->has_bucket;

    $self->_is_operator_failure(1);

    return $self->create_bucket()
}

=head2 can_switch_to_operator_bucket

    Доступно ли автоматическое списание баллов с оператора при недостатке баллов у клиента

=cut

sub can_switch_to_operator_bucket {
    my ($self) = @_;
    return $self->use_operator_units eq 'auto' ? 1 : 0;
}

=head2 switch_to_operator_bucket

    Меняем корзину на операторскую для случая, когда недостаточно клиентских баллов

=cut

sub switch_to_operator_bucket {
    my ($self) = @_;

    $self->_is_bucket_switched_to_operator(1);
    my ($bucket_type, $bucket_user) = $self->_process_operator();
    my $bucket_owner_login = $self->auth->operator_login;

    $self->_bucket_user($bucket_user);
    $self->_bucket_type($bucket_type);
    my $bucket = API::Units::Bucket->new(
        id    => $bucket_user->ClientID,
        limit => $bucket_user->units_limit,
        owner_login => $bucket_owner_login,
    );
    $self->_set_bucket($bucket);

    return $bucket;
}


=head2 create_bucket

    Создает объект корзины баллов, выбирая нужного клиента (оператора/его бренд или субклиента/его бренд),
    сохраняет корзину в поле _bucket, в _bucket_type записывает для какого клиента создана корзина.
    Падает при попытке повторного создания корзины.
    Возвращает созданный объект корзины.

=cut

sub create_bucket {
    my ($self) = @_;

    die "Bucket already exists" if $self->has_bucket;

    my $auth = $self->auth;
    my ($bucket_user, $bucket_type, $bucket_owner_login);
    if ($self->_is_operator_failure || $self->use_operator_units eq 'true') {
        ($bucket_type, $bucket_user) = $self->_process_operator();
        $bucket_owner_login = $auth->operator_login;
    } else {
        my $subclient = $self->subclient;
        my $subclient_brand_user = $auth->get_brand_chief_rep_user_by_clientid($subclient->ClientID);
        $self->_subclient_brand_user($subclient_brand_user);
        if ($subclient_brand_user && $subclient_brand_user->units_limit >= $subclient->units_limit) {
            $bucket_type = TYPE_SUBCLIENT_BRAND;
            $bucket_user = $subclient_brand_user;
        } else {
            $bucket_type = TYPE_SUBCLIENT;
            $bucket_user = $subclient;
        }
        $bucket_owner_login = $self->client_login || $subclient->login;
    }

    $self->_bucket_user($bucket_user);
    $self->_bucket_type($bucket_type);
    my $bucket = API::Units::Bucket->new(
        id    => $bucket_user->ClientID,
        limit => $bucket_user->units_limit,
        owner_login => $bucket_owner_login,
    );
    $self->_set_bucket($bucket);

    return $bucket;

};

=head2 _process_operator

    Обрабатываем оператора, пытаемся определить бренд
    Возвращаем тип оператора (под брендом или нет) и соответствующий объект пользователя

=cut

sub _process_operator {
    my ($self) = @_;

    my $operator = $self->auth->chief_rep_user;
    my $operator_brand_user = $self->auth->get_brand_chief_rep_user_by_clientid($operator->ClientID);
    $self->_operator_brand_user($operator_brand_user);

    my ($bucket_type, $bucket_user);
    if ($operator_brand_user && $operator_brand_user->units_limit >= $operator->units_limit) {
        $bucket_type = TYPE_OPERATOR_BRAND;
        $bucket_user = $operator_brand_user;
    } else {
        $bucket_type = TYPE_OPERATOR;
        $bucket_user = $operator;
    }

    return ($bucket_type, $bucket_user);
}

=head2 get_log_data()

    Возвращает хэш с данными о списании баллов готовый к логированию

=cut

sub get_log_data {
    my ($self) = @_;

    my $auth = $self->auth;
    my $operator = $auth && $auth->chief_rep_user;
    my $operator_brand_user = $self->_operator_brand_user;
    my $subclient = $self->subclient;
    my $subclient_brand_user = $self->_subclient_brand_user;
    my $bucket_user = $self->_bucket_user;
    my $units_stat = $self->_units_stat;
    my $bucket = $self->get_bucket;

    my $log_data = {
        reqid => $self->req_id,
        is_operator_failure => $self->_is_operator_failure,
        is_server_fault => $self->is_server_fault,
        use_operator_units => $self->use_operator_units,
        is_bucket_switched_to_operator => $self->_is_bucket_switched_to_operator,

        operator => {
            operator_uid => $auth && $auth->operator_uid,
            operator_login => $auth && $auth->operator_login,
            client => {
                operator_client_id => $operator && $operator->ClientID,
                operator_client_login => $operator && $operator->login,
                operator_client_role => $auth && $auth->role,
                operator_client_units_daily => $operator && $operator->api5_units_daily,
                operator_client_units_manual => $operator && $operator->api_units_daily,
                operator_client_relation => $auth && $auth->operator_client_relation_type,
            },
            brand => {
                operator_brand_client_id => $operator_brand_user && $operator_brand_user->ClientID,
                operator_brand_login => $operator_brand_user && $operator_brand_user->login,
                operator_brand_units_daily => $operator_brand_user && $operator_brand_user->api5_units_daily,
                operator_brand_units_manual => $operator_brand_user && $operator_brand_user->api_units_daily,
            },
        },
        client_login => $self->client_login,
        subclient => {
            client => {
                subclient_client_id => $subclient && $subclient->ClientID,
                subclient_uid => $subclient && $subclient->uid,
                subclient_login => $subclient && $subclient->login,
                subclient_units_daily => $subclient && $subclient->api5_units_daily,
                subclient_units_manual => $subclient && $subclient->api_units_daily,
            },
            brand => {
                subclient_brand_client_id => $subclient_brand_user && $subclient_brand_user->ClientID,
                subclient_brand_login => $subclient_brand_user && $subclient_brand_user->login,
                subclient_brand_units_daily => $subclient_brand_user && $subclient_brand_user->api5_units_daily,
                subclient_brand_units_manual => $subclient_brand_user && $subclient_brand_user->api_units_daily,
            },
        },
        bucket => {
            bucket_client_id => $bucket_user && $bucket_user->ClientID,
            bucket_login => $bucket_user && $bucket_user->login,
            bucket_units_spent => $units_stat && $units_stat->{spent},
            bucket_units_balance => $units_stat && $units_stat->{balance},
            bucket_units_limit => $bucket_user && ($bucket_user->units_limit || $Units::DEFAULT_LIMIT),
            bucket_type => $self->_bucket_type,
        },
        units_used_login => $bucket && $bucket->owner_login,
    };

    return $log_data;
}

=head2 units_stat($units_stat)

    Сохраняем в объекте списанные балы, балланс и лимит корзины.

=cut

sub units_stat {
    my ($self, $units_stat) = @_;

    $self->_units_stat({ spent => $units_stat->[0], balance => $units_stat->[1], limit => $units_stat->[2] });

    return;
}

__PACKAGE__->meta->make_immutable;

1;
