package API::UserConnectionLock;

use strict;
use warnings;
use utf8;

=pod

    $Id$

=head1 NAME

    API::UserConnectionLock

=head1 SYNOPSIS

    my $lock = API::UserConnectionLock->get($uid, 10)
        or return error_MaxConnectionsExceed;

    do_stuff();

    $lock->release;

=head1 DESCRIPTION

    Модуль для взятия лока на соединение по ключу. На каждый ключ может быть
    не больше локов, чем указано во втором параметре get.

=head1 METHODS

=cut

use Yandex::Memcached::Lock;

use Settings;

use Mouse;

has key => ('is' => 'ro', required => 1);
has max_connections => ('is' => 'ro', isa => 'Int', required => 1);

has _locker => ('is' => 'ro', isa => 'Object', lazy => 1, default => sub {
    my $self = shift;
    return Yandex::Memcached::Lock->new(
        servers => $Settings::MEMCACHED_SERVERS,
        entry => $self->key,
        no_auto_lock => 1,
        max_locks_num => $self->max_connections
    );
});

sub BUILDARGS {
    my ($class, $lock_id, $max_connections) = @_;

    return {
        key => $lock_id,
        max_connections => $max_connections
    };
}

=head2 get

    Взять лок. Возвращает true если лок на соединение взят успешно или если
    сервера обслуживающие локи не доступны. В случае если уже взято максимальное
    кол-во локов на соединения, возвращает false

=cut

sub get {
    my $self = shift;

    # Лок взяли автоматически при инициализации
    my $lock_status = $self->_locker->lock;

    # сервер для локов недоступен, позволяем бесконечное число соединений
    return 1 unless defined $lock_status;

    return $lock_status;
}

=head2 release

    Явно освобождаем лок. В случае если лок не постоянный (не было вызова
    L<make_persistent>), то лок вернется автоматически при удалении объекта.

=cut

sub release {
    my $locker = shift->_locker;
    $locker->unlock;
}

__PACKAGE__->meta->make_immutable;

1;

__END__
