package API::UserConnectionLock::Fake;

use Direct::Modern;

=pod

=encoding utf8

    $Id:$

=head1 NAME

    API::UserConnectionLock::Fake

=head1 SYNOPSIS

    my $limiter = API::UserConnectionLock::Fake($subclient, $subclient->allowed_concurrent_calls);
    my $max = $subclient->allowed_concurrent_calls;
    foreach my $i (1..$max) {
        ok($limiter->get, "$i lock ok");
    }

    ok(!$limiter->get, ($max + 1) . " lock fails");
    $limiter->release;
    ok($limiter->get, $limiter->current . " lock ok again after release");

=head1 DESCRIPTION

    Модуль для взятия локов для ограничения параллельных соединений, работает
    на глобальной переменной не использует мемкеш, нужен для написания юнит
    тестов

=head1 METHODS

=head2 new($user, $max)

    $user - API::Authorization::User,
    $max - int, кол-во параллельых соединений

=head2 max

    максимальное кол-во соединений с клиента

=head2 current

    текущее кол-во соединений с клиента

=head2 get

    Взять лок, инкрементирует текущее кол-во соединений и возвращает его, если
    соединений уже максимум, то возвращает undef

=head2 release

    Отпустить лок

=head2 FLUSH

    Сбрасывает текущие счетчики

=head2 STORAGE

    Хэш счетчиков вида { client_id => count }

=cut

our $USERS = {}; # client_id => connections_count

sub new {
    my ($class, $user, $max) = @_;
    return bless {
        max => $max,
        current => 0,
        user => $user,
        client_id => $user->client_id
    }, $class;
}

sub max { shift->{max} }

sub current { $USERS->{shift->{client_id}} //= 0 }

sub get {
    my $self = shift;
    return if $self->current >= $self->max;
    return ++$USERS->{$self->{client_id}};
}

sub release { --$USERS->{shift->{client_id}} }

sub FLUSH { $USERS = {} } # не является частью интерфейса API::UserConnectionLock
sub STORAGE { $USERS }

1;
