package API::WSDLFiles;

use strict;
use warnings;
use utf8;

use lib::abs;

use Carp qw/croak/;
use Exporter qw/import/;

our @EXPORT_OK = qw/wsdl_directory wsdl_files xsd_definitions/;

=pod

    $Id$

=head1 NAME

    API::WSDLFiles;

=head1 SYNOPSIS

    use API::WSDLFiles qw/wsdl_directory wsdl_files/;

    my $API_VERSION = 5;

    my $dir = wsdl_directory($API_VERSION);

    my @wsdls = wsdl_files($API_VERSION);

=head1 DESCRIPTION

    Модулья для получения списка WSDL файлов определенной версии API

=head1 METHODS

=head2 wsdl_directory($version_major_number)

    Возвращает директорию с wsdl-файлами для указанной версии API

=head2 wsdl_files($version_major_number)

    Возвращает список wsdl-файлов (с полными путями) для указанной версии API

=head2 xsd_definitions($version_major_number)

    Возвращает список xsd-файлов (с полными путями) для указанной версии API

=cut

sub _check_version {
    my ($version) = @_;
    unless($version && $version =~ /^\d+$/) {
        croak "Version should be interger";
    }
    return $version;
}

sub wsdl_directory {
    my $version = _check_version(@_); # number
    my $wsdls_dir = lib::abs::path("../../../data/soap-root/wsdl/v$version");
    -d $wsdls_dir or die "no wsdl dir found: $!";
    return $wsdls_dir;
}

sub wsdl_files {
    my $version = _check_version(@_); # number
    return _files($version, qr/\.wsdl$/);
}

sub xsd_definitions {
    my $version = _check_version(@_); # number
    return _files($version, qr/\.xsd$/);
}

sub _files {
    my $version = shift;
    my $re = shift;
    my $wsdls_dir = wsdl_directory($version);
    opendir(my $WSDLS_DH, "$wsdls_dir") or die "can't open directory with wsdl files: $!";
    return map {
        "$wsdls_dir/$_"
    } grep {
        $_ =~ $re && -f "$wsdls_dir/$_"
    } readdir($WSDLS_DH);
}

1;
