package Campaign::Creator;

=pod
=encoding utf8

    $Id:$

=head1 NAME

    Campaign::Creator

=head1 SYNOPSIS

    package Campaigns;

    ..
        my creator = Campaign::Creator->new(
            operator_role => 'manager',
            operator_uid => 1235,
            subclient_uid => 4567,
            subclient_client_id => 890,
        );
        if(my $error = $creator->why_cant_create_campaign) {
            return $creator->error;
        } else {
            my $campaign = Direct::Model::Campaign->new(
                client_fio => 'Ivanov Ivan Ivanovich',
                email => 'ivanov@ya.ru',
                currency => 'USD',
                campaign_type => 'text'
            );
            if(!$creator->create( $campaign ) ) {
                die $creator->error; # unexpected error
            }
        }
        return undef; # ok

    package Sitelinks;

    ..
        if($self->campaign_creator->can_create_campaign) {
            return $self->create_sitelinks();
        } else {
            return error_NoRights(iget('Нет прав на создание быстрых ссылок для клиента'));
        }

=head1 DESCRIPTION

    Класс отвечающий на вопрос "Может ли оператор создать кампанию саб-клиенту",
    и создающий кампанию с правильным типом сервисируемости (самоходную,
    менеджерскую, агентскую)

=head1 METHODS

=head2 new(%params)
    Параметры:
        * operator_role - роль залогиненного пользователя (client, manager, agency)
        * operator_uid - uid залогиненного пользователя
        * subclient_uid - uid главного представителя клиента для которого
                            создаем кампанию
        * subclient_client_id - client_id под которым будет хранится кампания

=head2 can_create_campaign

    Возвращает (true) если оператор может создавать кампании для данного клиента

=head2 why_cant_create_campaign

    Возвращает ошибку если оператор не может созадвать кампании для данного клиента или undef

=head2 create($campaign)

    $campaign - объект Direct::Model::Campaign, с заданными полями
            client_fio,
            email,
            currency,
            campaign_type

    В случае успеха возвращает cid, иначе ошибка доступная через метод error в
    виде объекта Direct::Defect

=head2 error

    Ошибка вызова метода create, если вызов не успешен, иначе undef

=head2 is_agency_or_under

    True если оператор является агентством или проверяемый сабклиент находится под агентством
    в обоих случаях кампания должна быть создана из под агентства или не создана вообще

=cut

use Carp qw/croak/;
use List::MoreUtils qw/any/;

use Mouse;

use Direct::Modern;
use Yandex::I18n;

use RBACDirect qw/
    rbac_has_agency
    rbac_get_agencies_for_create_camp_of_client
    rbac_is_owner

/;

use Rbac qw/:const/;

use Campaign qw/can_create_camp create_empty_camp del_camp/;
use Campaign::Creator::Types;
use Direct::Errors::Messages;

# поля
has operator_role => (isa => 'RBACRoles', is => 'ro', required => 1);
has operator_uid => (isa => 'Int', is => 'ro', required => 1);
has subclient_uid => (isa => 'Int', is => 'ro', required => 1);
has subclient_client_id => (isa => 'Int', is => 'ro', required => 1);
has is_operator_readonly => (isa => 'Bool', is => 'ro', required => 1);

# интерфейс

has is_agency_or_under => (isa => 'Bool', is => 'ro', lazy => 1, default => sub {
    my $self = shift;
    # в обоих случаях кампания будет создана из под агентства или не создана вообще
    return $self->operator_role eq 'agency' || rbac_has_agency(undef, $self->subclient_uid);
});

has error => (isa => 'Maybe[Direct::Defect]', is => 'rw');

sub can_create_campaign {
    shift->_can_create_campaign_cache->{error} ? 0 : 1;
}

sub why_cant_create_campaign { shift->_can_create_campaign_cache->{error} }

sub create {
    my ($self, $campaign) = @_;

    my %campaign_data = (
        product_type     => ($campaign->campaign_type eq 'cpm_banner') ? 'cpm_banner' : 'text',
        client_chief_uid => $self->subclient_uid,
        ClientID         => $self->subclient_client_id,
        type             => $campaign->campaign_type,
        currency         => $campaign->currency,
        client_fio       => $campaign->client_fio,
        client_email     => $campaign->email,
    );

    my $create_by = $self->create_by_role or die $self->error; # either check before calling create or be ready to die here

    if ($create_by eq 'agency') {
        $campaign_data{agency_uid} = $self->create_by_agency_uid;
    } elsif ($create_by eq 'manager') {
    # DIRECT-117871: Отключаем логику привязки новой кампании к менеджеру-оператору
    #   Сейчас кампания должна привязываться к "главному менеджеру", если он задан. Эта логика есть в ядре
    #     $campaign_data{manager_uid} = $self->operator_uid;
    } elsif ($create_by ne 'client') {
        croak "unknown create for";
    }

    my $error = can_create_camp( %campaign_data );
    if ($error) {
        warn "RBAC error creating new campaign: " . $error;
        $self->error( error_OperationFailed(iget("Ошибка создания кампании")) );
        return;
    }

    return create_empty_camp( %campaign_data );
}

has create_by_role => (is => 'ro', isa => 'RBACRoles', lazy => 1, default => sub {
    shift->_can_create_campaign_cache->{create_by_role};
});

has create_by_agency_uid => (is => 'ro', isa => 'Maybe[Int]', lazy => 1, default => sub {
    shift->_can_create_campaign_cache->{create_by_agency_uid};
});

# внутренности

has _can_create_campaign_cache => ('is' => 'ro', isa => 'CreateCheckResult', lazy => 1, default => sub {
    my $self = shift;
    my $result = {
        create_by_role => '',
        create_by_agency_uid => undef,
        error => undef
    };

    my $role = $self->operator_role;
    if( !$self->_has_rights_to_create_campaign
        || !(any {$_ eq $self->operator_role} qw/super placer manager support agency client/)
    ) {
        $result->{error} = error_NoRights_CantWrite( iget('Нет прав на создание кампании') );
        return $result;
    }

    if ($self->is_agency_or_under) {
        my $agencies_for_client = $self->_agencies_allowed_to_create_campaigns_under;

        if (!$agencies_for_client || !@$agencies_for_client) { # клиент под агентством на которое нет прав у оператора
            my $manager_has_rights = (($role eq $ROLE_MANAGER) && rbac_is_owner(undef, $self->operator_uid, $self->subclient_uid)) ? 1 : 0;
            if (!$manager_has_rights) {
            my $msg = $role eq 'agency'
                ? iget('Создание кампании невозможно. Этот логин занят другим агентством или обслуживание клиента завершено.')
                : iget('Нет прав на создание кампании');
            $result->{error} = error_NoRights_CantWrite( $msg );
            }
        } elsif (@$agencies_for_client > 1) {
            $result->{error} = error_NoRights_CantWrite( iget('Клиент не может находиться на обслуживание у более чем одного агентства'));
        }

        unless ($result->{error}) {
            $result->{create_by_role} = 'agency';
            $result->{create_by_agency_uid} = $agencies_for_client->[0];
        }
    } else {
        $result->{create_by_role} = $role eq 'manager' ? 'manager' : 'client';
    }

    return $result;
});


has _agencies_allowed_to_create_campaigns_under => (isa => 'ArrayRef[Int]', is => 'ro', lazy => 1, default => sub {
    my $self = shift;

    return rbac_get_agencies_for_create_camp_of_client(undef, $self->operator_uid, $self->subclient_uid) // [];
});

has _has_rights_to_create_campaign => (isa => 'Bool', is => 'ro', lazy => 1, default => sub {
    my $self = shift;

    # падает т.к. в role_perm нет записей для RBAC_CreateSCampaign. Но проверка не нужна, т.к.
    # право на клиента провереятся в авторизации опретора, а право на создание кампании под агентством
    # менеджером будет проверено _agencies_allowed_to_create_campaigns_under
    return 1 if $self->operator_role eq 'manager';
    return 0 if $self->is_operator_readonly;

    return RBACDirect::rbac_is_owner(undef, $self->operator_uid, $self->subclient_uid);
});

__PACKAGE__->meta->strict_constructor(1);
__PACKAGE__->meta->make_immutable;

1;

