package WSDL::JSON::Schema;

use strict;
use warnings;
use utf8;

use XML::Compile::Iterator;

use WSDL::JSON::Schema::Sequence;

=pod

    $Id$

=head1 NAME

    WSDL::JSON::Schema

=head1 SYNOPSIS

    use XML::Compile::WSDL11;

    my $wsdl11 = XML::Compile::WSDL11->new(
        undef,
        opts_rw => {
            sloppy_floats => 1,
            sloppy_integers => 1
        }
    );

    $wsdl11->addWSDL($_) foreach(@wsdl_files);
    $wsdl11->importDefinitions($_) foreach(@xsd_files);


    my $ns = 'http://direct.yandex.com/api/v5/adgroups';

    my $builder = WSDL::JSON::Schema->new(
        $wsdl11->namespaces,
        $ns
    );

    my $schema = builder("{$ns}GetRequest");


=head1 DESCRIPTION

    Класс для построения схемы валидации по WSDL-описанию Зависит от
    XML::Compile. Понимание WSDL-я ограниченное. Предназначено в первую очередь
    для валидации перл структур полученных десереализацией JSON-а, но может
    использоваться и в других целях

=head1 METHODS

=cut

=head2 new($namespaces, $local_path)

    namespaces - объект XML::Compile::Schema::NameSpaces, по текущим WSDL-ям
    local_path - строка, неймспейс по умолчанию, напр http://direct.yandex.com/api/v5/adgroups

=cut

sub new {
    my ($class, $namespaces, $local_path) = @_;
    return bless {
        namespaces => $namespaces,
        local_path => $local_path
    }, $class;
}

=head2 ns

    XML::Compile::Schema::NameSpaces объект

=cut
sub ns { shift->{namespaces} }

=head2 find($type, $path)

    $type - строка, тэг (element, complexType, simpleType)
    $path - название тэга {http://direct.yandex.com/api/v5/adgroups}GetRequestGeneral,
    в случае короткого названия будет подставлен локальный нейспейс из $self->path

=cut

sub find {
    my ($self, $type, $name) = @_;
    return $self->ns->find($type => $self->full_name($name));
}

=head2 path

    строка, дефолтный namespace

=cut

sub path { shift->{local_path} }

=head2 full_name($name)

    Возвращает полное имя элемента добавив namespace по умолчанию, в случае
    если неймспейс элемента не задан

=cut

sub full_name {
    my ($self, $name) = @_;

    return ($name =~ /\{.+\}/)
        ? $name
        : '{' . $self->path . "}$name";
}

=head2 build($element_name)

    строит схему для елемента $element_name(строка)

    формат схемы
    element = {
        min => $int,
        max => $int,
        nillable => $bool,
        type => $type
    }

    где type
        - скаляр для встроенных типов decimal int long string;
        - WSDL::JSON::Schema::Sequence для сложных типов
        - ссылка на массив для ENUM-ов

=cut

sub build {
    my ($self, $element_name) = @_;

    my $found = $self->find(element => $element_name)
        or die "element '$element_name' not found";
    my $node = $found->{node};

    my $full_name = $self->full_name($element_name);
    return $self->build_by_node($full_name => $node);
}

=head2 build_by_node($full_name => $node)

    Строим схему валидации по полному имени элемента и его объкту XML::LibXML::Node
    my $schema = build_by_node({http://api.direct.yandex.com/v5/ads}ResumeRequest => $resume_request_element)

=cut

sub build_by_node {
    my ($self, $full_name, $node) = @_;

    my $schema = $self->element(
        $self->iterator($node, $full_name)
    );

    # document literal - запрос всегда один элемент,
    # не может отстутствовать, не может быть ниллом
    foreach my $attr (qw/min max/) {
        die "$full_name can't have ${attr}Occur other than 1"
            unless $schema->{$attr} ne 'unbounded' && $schema->{$attr} == 1;
    }
    die "$full_name can't be nillable" if $schema->{nillable};

    return $schema;
}

=head2 iterator($node, $path)

    $node - XML::LibXML::Node
    $path - полное имя элемента

=cut

sub iterator {
    my ($self, $node, $path) = @_;
    return XML::Compile::Iterator->new(
        $node,
        $path,
        sub {
            my $n = shift;
            $n->isa('XML::LibXML::Element')
              && $n->namespaceURI eq $node->namespaceURI
              && $n->localName !~ qr/(?^:^(?:notation|annotation|(?^:unique|key|keyref)|(?^:assert|report))$)/
        }
    );
}

=head2 complexType($iterator)

    $iterator - XML::Compile::Iterator для ноды complexType
    строим описание для complexType
    Поддерживаются только complexType вида
        <complexType>
            <sequence>
                <element />
                ..
            </sequence>
        </complexType>
    или
        <complexType>
            <complexContent>
                <extentsion base="someBaseComplexType">
                    <sequence>
                        <element />
                        ..
                    </sequence>
                </extentsion>
            </complexContent>
        </complexType>
    или (пустой extension для будущего расширения)
        <complexType>
            <complexContent>
                <extentsion base="someBaseComplexType" />
            </complexContent>
        </complexType>
=cut

sub complexType {
    my ($self, $iter) = @_;
    $iter->nrChildren == 1 or die $self->_idebug($iter, "complexType can have only 1 children");

    my $child_name = $iter->firstChild->localName;
    if($child_name eq 'sequence') {
        return $self->sequence($iter->descend);
    } elsif($child_name eq 'complexContent') {
        my $content = $iter->descend;
        $self->_only_child_check($content, 'extension');

        my $extension = $content->descend;
        my $base_type = $extension->node->getAttribute('base')
            or die $self->_idebug($iter, 'extension must have base attribute');

        my $type = $self->get_type_by_name($extension->node, $base_type);

        if($extension->node->hasChildNodes) {
            $self->_only_child_check($extension, 'sequence');
            my $sequence = $self->sequence( $extension->descend );
            foreach my $name ($sequence->element_names) {
                $type->add($name, $sequence->element($name));
            }
        }
        return $type;
    } else {
        die $self->_idebug($iter, "complexType must either consists of sequence or consists of complexContent");
    }
}

=head2 simpleType($iterator)

    $iterator - XML::Compile::Iterator для ноды simpleType

    <simpleType>
        <restriction> /* аттрибут base игнорируется */
            <enumeration>ONE</enumeration>
                ..
            <enumeration>N</enumeration>
        </restriction>
    </simpleType>

    возвращает массив значений ENUM-а

=cut

sub simpleType {
    my ($self, $iter) = @_;
    $self->_only_child_check($iter, 'restriction');
    my $restriction = $iter->descend;

    my $values = [];
    foreach my $enum ($restriction->childs) {
        die $self->_idebug($iter, "wrong tag in restriction", $enum->localName) unless $enum->localName eq 'enumeration';
        push @$values, $enum->getAttribute('value');
    }
    return $values;
}

=head2 _only_child_check($iter, $childname)

    $iterator - XML::Compile::Iterator для проверяемой ноды Проверяет что нода
    имеет только один тэг типа $childname, иначе кидает die (схема считается не
    валидной)

=cut

sub _only_child_check {
    my ($self, $iter, $child_name) = @_;
    die $self->_idebug($iter, "must consists of one $child_name") unless $iter->nrChildren == 1 and $iter->firstChild->localName eq $child_name;
    return;
}

=head2 _idebug($iter, @msg)

    Возвращает отладочное сообщение с перфиксом для текущего элемента в итераторе

=cut

sub _idebug {
    my ($self, $iter, @msg) = @_;
    return $iter->path . ": " . $iter->nodeLocal . " " . join(' ', @msg);
}

=head2 sequence($iter)

    $iterator - XML::Compile::Iterator для ноды <sequence>
    возвращает массив описаний элементов каждого элемента последовательнсти

=cut

sub sequence {
    my ($self, $iter) = @_;
    my $sequence = WSDL::JSON::Schema::Sequence->new();
    foreach my $child ($iter->childs) {
        my ($name, $element) = $self->element($iter->descend($child));
        $sequence->add( $name, $element );
    }
    return $sequence;
}

=head2 element($iter)

    $iterator - XML::Compile::Iterator для ноды <element>

    Поддерживаемые виды элементов
    <element name="SomeName" type="xsd:some_built_in_type" minOccurs=i maxOccurs=j nillable=bool />

    Для maxOccurs > 1 nillable не может быть true. Если нужны nillable массивы,
    то заворачиваем их в объект, вроде Object => { Items => [] }, где Object nillable

    <element name="SomeName">
        <complexType>
            см. описание complexType
        </complexType>
    </element>

    <element name="SomeName">
        <simpleType>
            см. описание simpleType
        </simpleType>
    </element>

=cut

sub element {
    my ($self, $iter) = @_;
    my $e = $iter->node;
    my $attrs = {
        minOccurs => 'min',
        maxOccurs => 'max',
        nillable => 'nillable',
        name => 'name',
        type => 'type',
    };

    my $name = $e->getAttribute('name') or die $self->_idebug($iter, "element has no name attribute");

    my $constraints = {};

    unless(!$iter->nrChildren || $iter->nrChildren == 1) {
        die "$e element must have complexType child or no children at all"
    }

    my $type;
    if($iter->nrChildren == 1) {
        if ($iter->firstChild->localName eq 'complexType') {
            $constraints->{type} = $self->complexType($iter->descend);
        } elsif ($iter->firstChild->localName eq 'simpleType') {
            $constraints->{type} = $self->simpleType($iter->descend);
        } else {
            die $self->_idebug($iter, "$e element can only consist of one complexType or simpleType sub element");
        }
    } elsif(my $type_name = $e->getAttribute('type')) {
        $constraints->{type} = $self->get_type_by_name($e, $type_name);
    } else {
        die $self->_idebug($iter, "element $e must either consist of type definition or has type attribute");
    }
    my $has_attributes = {
        map { $_->name => $_->value } $e->attributes
    };

    my $accept_attributes = {};
    ## no critic (Freenode::DollarAB)
    foreach my $a (keys %$attrs) {
        $accept_attributes->{$attrs->{$a}} = delete $has_attributes->{$a}
            if exists $has_attributes->{$a};
    }

    if(scalar keys %$has_attributes) {
        die $self->_idebug($iter, "unrecognized attributes " . join(', ', keys %$has_attributes));
    }

    foreach (qw/min max/) {
        $constraints->{$_} = exists $accept_attributes->{$_}
            ? $accept_attributes->{$_} : 1; # min/max Occurence default is 1
    }

    unless($constraints->{max} eq 'unbounded') {
        die $self->_idebug($iter, "maxOccurs must be greater than 0") unless $constraints->{max} > 0;
        die $self->_idebug($iter, "maxOccurs must be 1 or unbounded, other numbers are unsupported") unless $constraints->{max} == 1;
    }

    die $self->_idebug($iter, "minOccurs must be greater than 0 or equal") if $constraints->{min} < 0;

    die $self->_idebug($iter, "maxOccurs must be greater or equal to minOccurs")
        if  $constraints->{max} ne 'unbounded' && $constraints->{max} < $constraints->{min};

    # Возможно отключим в будущем. Проверяем после равнения min и max чтобы не маскировать их несоответствие
    die $self->_idebug("minOccurs must be 0 or 1") if $constraints->{min} > 1;

    $constraints->{nillable} =
        ($accept_attributes->{nillable}||'') eq 'true' ? 1 : 0; # default is false

    if($constraints->{max} eq 'unbounded' && $constraints->{nillable}) {
        die $self->_idebug($iter, "element", $e->getAttribute('name'), ": array can't be nillable");
    }

    return $name => $constraints;
}

=head2 get_type_by_name($node, $type_name)

    $node - XML::LibXML::Node объект тэга родителя
    $type_name тип (строка) который ищем

    возвращает описание типа, по имени

=cut

sub get_type_by_name {
    my ($self, $node, $type_name) = @_;
    my $builtInTypes = { map { $_ => 1} qw/decimal int long string base64Binary/ };
    if($type_name =~ /^xsd:(\w+)$/) {
        die "$1 unknown simple datatype" unless $builtInTypes->{$1};
        return $1;
    }

    my ($uri, $local);
    if($type_name =~ m/^(.+?)\:(.*)/) {
        $uri = $node->lookupNamespaceURI($1);
        $local = $2;
    } else {
        $uri = $self->path;
        $local = $type_name;
    }

    my $full_type_name  = $uri ? "{$uri}$local" : $local;
    my $type = $self->find(complexType => $full_type_name)
            || $self->find(simpleType => $full_type_name);
    $type or die "$full_type_name not found";
    my $type_iter = $self->iterator($type->{node}, $type->{full});

    return $type->{type} eq 'complexType'
        ? $self->complexType($type_iter)
        : $self->simpleType($type_iter);
}

1;
