package WSDL::JSON::Schema::Printer;

use strict;
use warnings;
use utf8;

=pod

    $Id:$

=head1 NAME

    WSDL::JSON::Schema::Printer

=head1 SYNOPSIS

    use WSDL::JSON::Schema::Printer qw/schema_dump/;
    my $namespaces = API::Service->new(
        '/path/to/some.wsdl'
    )->namespaces;
    my $namespace = 'http://api.direct.yandex.com/v5/adgroups';
    my $schema = WSDL::JSON::Schema->new($namespaces, $namespace)->build;
    print schema_dump(Request => $schema);

=head1 DESCRIPTION

    Класс для вывода схемы в текстовом виде

=head1 METHODS

=head2 schema_dump($element_name, $element_definition)

    $elemenent_name - строка с именем первого элемента
    $element_definition - описание структуры элемента в формате возвращаемом WSDL::JSON::Schema->build

=head2 $TAB

    Строка, с одним отступом

=cut

use WSDL::JSON::Validate;
use Exporter qw/import/;

our @EXPORT_OK = qw/schema_dump/;

my $IDENT; # счетчик вложенности

our $TAB = '    ';

# конвертация имен простых типов
my $TYPEMAP = {
    'int' => 'integer'
};

sub schema_dump {
    my ($name, $schema) = @_;
    $IDENT = 0;
    return _dump_element($name, $schema) . "\n";
}

sub _dump_element {
    my ($name, $element) = @_;
    my $ident = $IDENT++;
    my $type = $element->{type} or die "$name: no type";

    my $res;
    if(ref $type eq 'WSDL::JSON::Schema::Sequence') { # object
        my @subelements;
        foreach my $subelement_name ($type->element_names) {
            my $sub_element = $type->element($subelement_name);
            push @subelements, _dump_element($subelement_name, $sub_element);
        }
        my $subelements_s = join(",\n", @subelements);

        my ($bkt_left, $bkt_right) = qw/{ }/;
        if($element->{max} eq 'unbounded' || $element->{max} > 1) {
            $bkt_left = "[$bkt_left";
            $bkt_right = qq~$bkt_right, .. ($element->{max})]~;
        }
        $res = _ipr("\"$name\" : $bkt_left", _cmt($element)) 
            . "$subelements_s\n" 
            . _ip($bkt_right);
    } elsif (!ref $type || ref $type eq 'ARRAY') { # simpleton or ENUM
        $res = _pel($name, $element, $type);
    } else {
        die "die strange type $name => $type";
    }
    $IDENT--;
    return $res;
}

sub _pel { # print element
    my ($name, $element, $type) = @_;
    my $definition_string = _bkts(
        ref $type eq 'ARRAY' # ENUM
            ? join('|', map { qq~"$_"~ } @$type)
            : _typemap($type) );
    if(_is_array($element)) {
        $definition_string = "[ $definition_string ]";
    }
    return _ip(
        "\"$name\" : ",
        $definition_string,
        _cmt($element)
    );
}

sub _cmt {
    my $element = shift;
    my @features;
    if($element->{min} > 0) {
        push @features, 'required';
    }
    if($element->{nillable}) {
        push @features, 'nillable';
    }
    return @features ? join(' ', ' /*', @features, '*/') : '';
}

sub _typemap { $TYPEMAP->{$_[0]} || $_[0] }
sub _bkts{ _concat( "(", @_ , ")" ) }

sub _concat { join('', @_) }

sub _ip { # ident print
    return _concat((map { $TAB } (1..$IDENT)) , @_);
}

sub _ipr { #indent print return
    return _ip(@_, "\n");
}

sub _is_array { WSDL::JSON::Validate::as_array(shift) }

1;

__END__
