package WSDL::JSON::Validate;

use strict;
use warnings;
use utf8;

=pod

    $Id$

=head1 NAME

    WSDL::JSON::Validate

=head1 SYNOPSIS

    use WSLD::JSON::Schema;
    use WSLD::JSON::Validate;

    my $builder = 'WSDL::JSON::Schema'->new($ns, 'http://direct.yandex.com/api/v5/adgroups');
    WSDL::JSON::Validate::validate(GetRequest => $builder->build('GetRequest'), $data);

=head1 DESCRIPTION

    Пакет для проверки перловой структуры (на самом деле десериализованного
    JSON-а), согласно WSDL схеме сконвертированной в правила проверки благодаря
    WSDL::JSON::Schema

    При валидации также происходит преобразование типов для скаляров (+0, +0.0,
    "$_[0]"), а также не заданные списковые ключи определяются как пустой
    массив

=head1 METHODS

=cut

use List::MoreUtils 'any';
use MIME::Base64 qw/decode_base64/;
use Scalar::Util qw/blessed/;

use Yandex::Validate qw/is_valid_float is_valid_int is_valid_base64/;

use Yandex::I18n;
use Direct::Errors::Messages;

=head2 validate($name, $definition, $data)

    Проверяем структуру на соответствие схеме, в случае ошибки кидаем
    эксепшн Direct::Defect. Отсутствие эксепшена говорит об успешности прохождения проверки

    $name - имя проверяеемой структуры, т.е. имя ключа хэша под которым
    расположены данные в родительской структуре или имя запроса
    $definition - определение данных, схема WSDL::JSON::Schema
    $data - сами данные (скаляр, ссылка на массив или хэш)

    Метод проверяет значение на null, а также кол-во элементов для массивов, и
    вызывает проверку типа.

    Для ключей значения которых не обязательны (minOccurs = 0), данный метод
    вызываться не должен, потому как иначе занчением будет считаться undef,
    котоый соответствует null в json, и значение будет проверено на is_nillable
    в схеме.

=cut

sub validate {
    my ($name, $definition, $data) = @_;

    # json null === perl undef
    if(!defined $data) {
        if($definition->{nillable}) {
            return $data; #ok
        } else {
            error(iget_noop("%s не может иметь значение null"), $name);
        }
    }

    if($definition->{max} eq 'unbounded' || $definition->{max} > 1) { # as_array performance optimization
        error(iget_noop("%s должен содержать массив"), $name)
            unless ref $data eq 'ARRAY';

        error(iget_noop("%s должен содержать не менее %d элемента"), $name, $definition->{min})
            if scalar(@$data) < $definition->{min};

        error(iget_noop("%s не может содержать более %d элемента"), $name, $definition->{max})
            if  $definition->{max} ne 'unbounded' # !_infinite performance optimization
                && scalar(@$data) > $definition->{max};

        foreach my $i (0..$#$data) {
            error(iget_noop("Элемент массива %s не может иметь значение null"), $name)
                unless defined $data->[$i];

            $data->[$i] = validate_type($name, $definition->{type}, $data->[$i], 1);
        }
    } else {
        error(iget_noop("%s не может содержать массив"), $name)
            if ref $data eq 'ARRAY';
        $data = validate_type($name, $definition->{type}, $data);
    }

    return $data;
}

=head2 validate_type($name, $type, $value, | $is_array)

    Проверяем значение $value на соответствие типу $type (см. типы в
    WSDL::JSON::Schema) - скаляр, объект, значение ENUM.

    $name - имя поля в родительской структуре нужно для выдачи ошибок
    $is_array - если ошибка на элемент массива, влияет на текст ошибок, не обязательный параметр

=cut

sub validate_type {
    my ($name, $type, $value, $is_array) = @_;
    my $get_item_name = sub { $is_array ? iget("Элемент массива %s", $name) : $name }; # что?
    my $get_item_name_where = sub { $is_array ? iget("элементе массива %s", $name) : $name }; # Где?

    if(!ref $type) { # built-in type
        if($type eq 'int' || $type eq 'long') {
            # is_valid_int здесь заинлайнена для производительности
            error(iget_noop("%s должен содержать целочисленное значение"), $get_item_name->())
                unless $value =~ /^-?[0-9]+$/; # is_valid_int performance optimization
            return $value+0; # без return будет warning
        } elsif ($type eq 'decimal') {
            error(iget_noop("%s должен содержать дробное значение"), $get_item_name->())
                unless is_valid_float($value);
            return $value + 0.0;
        } elsif ($type eq 'string') {
            error(iget_noop("%s должен содержать строку"), $get_item_name->()) if ref $value;
            return "$value";
        } elsif ($type eq 'base64Binary') {
            error(iget_noop("%s должен содержать данные, закодированные с помощью Base64"), $get_item_name->())
                unless is_valid_base64($value);
            return decode_base64($value);
        } else {
            die "unknown type $type";
        }
    } elsif (ref $type eq 'ARRAY') { # ENUM
        error(iget_noop("%s содержит неверное значение перечисления"), $get_item_name->())
            unless any { $_ eq $value } @$type;
    } elsif (blessed($type) && $type->can('name') && $type->name eq 'sequence') {
        error(iget_noop("%s должен содержать объект"), $get_item_name->())
            unless ref $value eq 'HASH';

        my $elements = $type->elements;

        # проверяем ключи и из значения
        foreach my $key (sort keys %$value) { # сортировка важна, иначе результат каждый раз разный
            $elements->{$key} or error(iget_noop("%s содержит неизвестное поле %s"), $get_item_name->(), $key);
        }

        # проверяем наличие всех обязательных ключей
        foreach my $key ($type->element_names) {
            my $definition = $elements->{$key};
            if (exists $value->{$key}) {
                $value->{$key} = validate($key, $definition, $value->{$key});
                # Вырезаем пустые массивы из запроса. Грязных хак, обусловлен кросс-протокольностью API,
                # т.к. в XML не может придти пустой массив, то мы делаем вид что в JSON их тоже нет,
                # чтобы отдельно не валидировать этот момент в контроллере
                delete $value->{$key} if ref $value->{$key} eq 'ARRAY' && !@{$value->{$key}};
            } elsif (is_required($definition)) {
                error(iget_noop("В %s отсутствует обязательное поле %s", $get_item_name_where->(), $key));
            }
        }
    } else {
        die "strange type $type, something wrong";
    }

    return $value; # ok
}

=head2 error($msg)

    Формируем ошибку и умираем с ней. Метод для формирования ошибок валидации
    Только ошибки из этого метода должны трактоваться как ошибки валидации, все
    остальное -- не предвиденые ошибки

=cut

sub error { die error_BadRequest(iget(@_)) }

=head2 _infinite($schema)

    Истина если схема (WSDL::JSON::Schema) описывает бесконечный массив

=head2 is_required ($schema)

    Истина если схема (WSDL::JSON::Schema) описывает обязательное поле

=head2 as_array($schema)

    Истина если схема (WSDL::JSON::Schema) описывает массив

=cut

sub _infinite { shift->{max} eq 'unbounded' }

sub is_required {
    my $def = shift;
    return $def->{min} != 0;
}

sub as_array {
    my $def = shift;
    return _infinite($def) || $def->{max} > 1;
}

1;

__END__
