use Direct::Modern;

use Plack::Request;

use lib::abs "./../../../protected";
use lib::abs "./../../lib";
use lib::abs "./../../services/v5";

use API::Services;
use API::WSDLFiles qw/wsdl_files xsd_definitions/;
use WSDL::JSON::Schema;
use WSDL::JSON::Schema::Printer qw/schema_dump/;

my $VERSION = 5;
my $services = API::Services->new( [ wsdl_files($VERSION) ], [ xsd_definitions($VERSION) ]);
my $nss = $services->namespaces;

my $app = sub {
    my ($env) = @_;

    my $req = Plack::Request->new($env);

    my ($service, $part) = parse_path($req->path);

    if ($service && $part) {
        if ($req->user_agent =~ /Mozilla/) {
            return html_ok(
                tag_html(
                    '<a href="/schema/v5/">Back to services list</a>'
                    . "<h2>$service : $part</h2>"
                    . code_json(get_part($service, $part))
                )
            )
        } else {
            return json_ok(get_part($service, $part));
        }
    } else {
        return html_ok(tag_html(list_services()));
    }
};


sub json_ok {
    my ($json) = @_;
    return [
        '200',
        [ 'Content-Type' => 'text/json' ],
        [ $json ]
    ];
}

sub html_ok {
    my ($html) = @_;
    return [
        '200',
        [ 'Content-Type' => 'text/html' ],
        [ $html ]
    ];
}


sub get_part {
    my ($service, $part) = @_;
    return builder_element_print(
        WSDL::JSON::Schema->new( $nss, "http://api.direct.yandex.com/v$VERSION/" . lc($service) ), $part
    );
}

sub builder_element_print {
    my ($builder, $element) = @_;
    my $schema = $builder->build($element);
    return schema_dump( ($element =~ /Request/ ? "params" : "result") => $schema);
}

sub parse_path {
    my $path = shift;
    $path =~ s/^\///;
    my ($service, $part, @a) = split('/', $path);
    return unless $service && $part;
    $service =~ s/[^a-zA-Z0-9]//g;
    $part =~ s/[^a-zA-Z0-9]//g;
    return ($service, $part);
}

sub list_services {
    my $res = '';

    my $service_operations;
    $services->map_thru_service_operation_ns_parts(sub {
        my ($service_name, $operation, $ns, $request_element, $response_element) = @_;
        $service_operations->{$service_name} //= { ns => $ns, operations => [] };
        push @{$service_operations->{$service_name}{operations}}, { name => $operation, request => $request_element, response => $response_element };
    });

    foreach my $s (sort keys %$service_operations) {
        my $ns = $service_operations->{$s}{ns};
        $res .= "<h3>$s " . wsdl_link($ns) . "</h3>\n";
        $res .= '<table class="operations">';
        foreach my $o (@{$service_operations->{$s}->{operations}}) {
            $res .= "<tr>";
            $res .= td($o->{name});
            $res .= td(operation_part_link($s, $o->{request}));
            $res .= td(operation_part_link($s, $o->{response}));
            $res .= "</tr>\n";
        }
        $res .= "</table>\n";
    }
    return $res;
}

sub td {
    return "<td>$_[0]</td>";
}

sub operation_part_link {
    my ($service, $part) = @_;
    return qq~<a href="/schema/v5/$service/$part">$part</a>~;
}

sub wsdl_link {
    my $url = shift;
    $url =~ s/^http/https/;
    $url =~ s/api\.direct\.yandex\.com/api.direct.yandex.ru/;

    return qq~<a href="$url?wsdl">wsdl</a>~;
}

sub code_json {
    my $code = shift;
    return qq~<pre><code class="javascript">$code</code></pre>~;
}

sub tag_html {
    my $body = shift;

    return
q~
<html>
<head>
    <link rel="stylesheet"
          href="//s3.mds.yandex.net/direct-qa/static/highlight-javascript/styles/atom-one-light.css">
    <script src="//s3.mds.yandex.net/direct-qa/static/highlight-javascript/highlight.pack.js"></script>
    <script>hljs.initHighlightingOnLoad();</script>
    <style>
        h3 {
            margin-top: 30px;
            margin-bottom: 10px;
        }
        table.operations {
            width: 600px;
        }
        body {
            padding: 10 10 50 20;
        }
    </style>
</head>
<body>~
. $body .
q~
</body>
</html>
~;
}
