package API::Service::AdImages;

# $Id: AdImages.pm 107863 2016-02-11 21:28:14Z hmepas $

=encoding utf-8

=head1 NAME

API::Service::AdImages - методы работы с хранимыми картинками Директа в API5

=head1 DESCRIPTION

Модуль предоставляет возможность создавать (add), читать (get) и удалять (delete) картинками через вызовы API

=cut

use Direct::Modern;

use parent qw/API::Service::Base/;

use List::MoreUtils qw/any/;
use Math::Round qw/round/;

use Yandex::DBShards;
use Yandex::I18n;

use BannerImages::Pool qw//;
use BannerImages qw/banner_prepare_save_image/;
use Direct::Errors::Messages;
use Direct::Model::ImageFormat;
use Direct::Model::ImageFormat::Manager;
use Direct::Model::ImagePool;
use Direct::Model::ImagePool::Manager;
use Direct::Validation::Image;
use HashingTools qw//;

use API::Service::Request::Get;
use API::Service::Request::Delete;
use API::Service::AdImages::ResultSet::Add;
use API::Service::AdImages::ResultSet::Delete;
use API::Version;
use Property;

our $ADIMAGES_ADD_LIMIT    //= 100;
our $ADIMAGES_DELETE_LIMIT //= 10_000;
our $ADIMAGES_GET_LIMIT    //= 10_000;

# TODO: наверное, место этому где-нибудь в моделях
our $MAX_NAME_LENGTH //= 255;

my $logo_in_api_is_enabled = Property->new('logo_in_api_is_enabled');

=head2 get(self, request)



=cut

sub get {
    my ($self, $request) = @_;

    my $req = API::Service::Request::Get->new( $request );
    if (my $e = $req->validate(
        limits => {
            AdImageHashes => $ADIMAGES_GET_LIMIT,
        }
    )) {
        return $e;
    }

    my $res = {};

    my ($limit, $offset) = ($req->page_limit, $req->page_offset);

    my $condition = {};

    # флаг: если взведён в 1, на самом деле в базу не ходим, отдаём клиенту пустой массив
    my $simulate_empty_result;

    if($req->has_selection_criteria) {
        my %crit = %{$req->selection_criteria};
        if (exists $crit{AdImageHashes}) {
            for my $image_hash (@{$crit{AdImageHashes}}) {
                if ($image_hash eq '') {
                    return error_InvalidField_EmptyArrayItem(undef, field => "AdImageHashes");
                }
            }

            # Fail Fast: если клиент дал неправдоподобный ImageHash, базу про него не спрашиваем
            $condition->{image_hash} = [ grep { HashingTools::is_valid_md5_base64ya($_) } @{ $crit{AdImageHashes} } ];

            # ... и если правдоподобных ImageHash не осталось, базу вообще не спрашиваем
            unless ( @{ $condition->{image_hash} } ) {
                $simulate_empty_result = 1;
            }
        }
        if (exists $crit{Associated}) {
            $condition->{associated} = ($crit{Associated} eq 'YES' ? 1 : 0);
        }
    }

    if (!exists $condition->{image_hash}) {
        $condition->{source} = 'direct';
    }

    my %O;
    if (exists $condition->{associated} || any {$_ eq 'Associated'} @{$request->{FieldNames}}) {
        $O{associated_info} = 1;
    }

    if (any {$_ =~ /Type|Subtype|PreviewUrl|OriginalUrl/} @{$request->{FieldNames}}) {
        $O{size_info} = 1;
    }

    my @images;
    unless ($simulate_empty_result) {
        my $are_logos_enabled = $logo_in_api_is_enabled->get(60);
        @images = grep { $are_logos_enabled || $_->{'image_type'} ne 'logo'} @{get_images($self, $condition, %O)};
    }

    my $count = scalar @images;
    if ($count > $offset) {
        @images = splice @images, $offset, $limit;

        $res->{LimitedBy} = $offset + $limit if ($count - $offset > $limit);
        $res->{AdImages} = $self->_filter_get_response(\@images, $request->{FieldNames});
    }

    $self->units_withdraw_for_objects('adimage' => scalar(@images));

    return $res;
}

=head2 get_images(self, condition, O)

Выбрать информацию о картинках по условию отбора:
client_id — обязательно,
associated (true/false) — выбрать только те, что привязаны / только те, что не привязаны к объявлениям,
image_hash.
Вернуть хеш, имя, тип, url для предпросмотра, url с оригинальным изображением и флаг привязанности к хотя бы одному объявлению

O{associated_info} — если нет или неправда, то не смотреть на associated и не возвращать его
O{size_info} — также вернуть данные из banner_images_formats: image_type и десериализованный formats

=cut

sub get_images {
    my ($self, $condition, %O) = @_;
    my %bip_get_options;
    if ($O{associated_info}) {
        $bip_get_options{with_assigned_info} = 1;
        $bip_get_options{clientid2chiefuid} = {$self->subclient_client_id => $self->subclient_uid};
    }
    if ($O{size_info}) {
        $bip_get_options{with_size} = 1;
    }
    if (exists $condition->{associated}) {
        $bip_get_options{assigned} = $condition->{associated};
    }
    if (exists $condition->{source}) {
        $bip_get_options{source} = $condition->{source};
    }
    my $image_hash = (exists $condition->{image_hash}) ? $condition->{image_hash} : undef;
    my $pool_items = BannerImages::Pool::get_items([$self->subclient_client_id], $image_hash, %bip_get_options);

    return $pool_items;
}

my %type_translation = (small => 'SMALL', wide => 'WIDE', regular => 'REGULAR', 'image_ad' => 'FIXED_IMAGE', 'logo' => 'LOGO');

=head2 _filter_get_response(self, response, field_names)

Преобразовать response во внешний формат,
вернуть массив хешей с теми field_names, что указаны

=cut

sub _filter_get_response {
    my ($self, $response, $field_names) = @_;
    my @result;

    my %fn = map {$_ => undef} @$field_names; # field names hash for faster filtering
    my $hostname = $API::Version::is_production ? '' : $self->http_host;

    foreach my $pool_item (@$response) {
        my $item;
        if (exists $fn{AdImageHash}) {
            $item->{AdImageHash} = $pool_item->{image_hash};
        }
        if (exists $fn{Name}) {
            $item->{Name} = $pool_item->{name} // '';
        }
        if (exists $fn{Associated}) {
            $item->{Associated} = $pool_item->{assigned} ? 'YES' : 'NO';
        }
        if (exists $fn{Type}) {
            $item->{Type} = $pool_item->{image_type} ? $type_translation{$pool_item->{image_type}} // 'UNFIT' : 'UNFIT';
        }

        my $base_url;

        my $is_image_ad_image = $pool_item->{image_type} eq 'image_ad';

        if (exists $fn{OriginalUrl}) {
            $base_url //= BannerImages::get_image_url( $pool_item, { host => $hostname } );
            $item->{OriginalUrl} = $base_url;
        }

        if (exists $fn{PreviewUrl}) {
            $base_url //= BannerImages::get_image_url( $pool_item, { host => $hostname } );

            if ($is_image_ad_image) {
                $item->{PreviewUrl} = $base_url;
            } else {
                if (($pool_item->{image_type} // '') eq 'wide' && exists $pool_item->{formats}->{wx300}) {
                    $item->{PreviewUrl} = $base_url.'/wx300';
                } elsif (exists $pool_item->{formats}->{x90}) {
                    $item->{PreviewUrl} = $base_url.'/x90';
                } else {
                    $item->{PreviewUrl} = 'NIL';
                }
            }
        }

        if (exists $fn{Subtype}) {
            if ($is_image_ad_image) {
                state $inaccuracy_property = Property->new('INACCURACY_IN_PERCENTS_FOR_PROPORTIONALLY_LARGER_IMAGES');
                my $inaccuracy = $inaccuracy_property->get(60) // 3;
                my $dimensions = $pool_item->{formats}->{orig};
                foreach my $size (@Direct::Validation::Image::VALID_SIZES_ORIGIN) {
                    my $ratio = round($dimensions->{width} / $size->[0]);
                    if (abs($ratio * $size->[0] - $dimensions->{width}) <= $dimensions->{width} * $inaccuracy / 100
                        &&
                        abs($ratio * $size->[1] - $dimensions->{height}) <= $dimensions->{height} * $inaccuracy / 100) {
                        $item->{Subtype} = 'IMG_' . $ratio * $size->[0] . '_' . $ratio * $size->[1];
                        last;
                    }
                }
                if (!defined $item->{Subtype}) {
                    die "Invalid image_ad size: $dimensions->{width}x$dimensions->{height}";
                }
            } else {
                $item->{Subtype} = 'NONE';
            }
        }

        push @result, $item;
    }

    return \@result;
}

=head2 add

=cut

sub add {
    my ($self, $request) = @_;

    my $rs = API::Service::AdImages::ResultSet::Add->new(@{$request->{AdImages}});

    if (my $e = $self->_validate_add_request($rs)) {
        return $e;
    }

    my $clientid = $self->subclient_client_id;

    # первая валидация отдельных изображений: смотрит на всякие размеры файлов и сколько
    # клиент может загрузить
    $self->_validate_add_request_items($rs);

    # дорогая операция посмотреть и заполнить размеры-типы изображений
    # для изображений, которые забракованы первой валидацией, не нужна
    for my $item ($rs->list_ok) {
        my $image_info = Direct::Validation::Image::check_image($item->object->{ImageData});
        $item->object->{_check_result} = $image_info;

        unless ($image_info->{error}) {
            $item->object->{_image_size_valid_for_image_ad} =
                Direct::Validation::Image::is_image_size_valid_for_image_ad($image_info->{width}, $image_info->{height});
        }
    }

    # вторая валидация, которая исходит из размеров-типов изображений
    $self->_validate_checked_add_request_items($rs);

    for my $item ($rs->list_ok) {
            my $format = Direct::Model::ImageFormat->new(
                image => $item->object->{ImageData}, name => $item->object->{Name}, info => $item->object->{_check_result}
            );
            $item->object->{image_type} = $format->image_type;
    }
    $self->_validate_image_ad_size($rs);

    # AdImageHash => { ClientID => 123, image_hash => 'FOOBAR', name => 'file.jpg' }, ...
    my %items_to_add_to_images_pool;

    # AdImageHash => Direct::Model::ImageFormat
    my %images_to_save_for_image_ads;

    # AdImageHash => [ item from ResultSet, ... ]
    my %rs_items_by_image_hash;

    for my $item ($rs->list_ok) {
        my $ad_image_hash = $item->object->{_check_result}->{md5};
        my $fname = $item->object->{Name};

        $rs_items_by_image_hash{$ad_image_hash} ||= [];
        push @{ $rs_items_by_image_hash{$ad_image_hash} }, $item;

        if ($item->object->{_image_size_valid_for_image_ad}) {
            $images_to_save_for_image_ads{$ad_image_hash} = Direct::Model::ImageFormat->new(
                image => $item->object->{ImageData},
                name => $fname,
                namespace => $item->object->{image_type} eq 'LOGO' ? 'direct' : 'direct-picture'
            );

        } else {
            my $img_data = banner_prepare_save_image($item->object->{ImageData}, { ClientID => $clientid }, keep_jpeg => 1);

            my $bipid;
            if ($img_data->{error}) {
                $item->add_error(error_OperationFailed($img_data->{error}));
                next;
            }

            $items_to_add_to_images_pool{$ad_image_hash} = {
                ClientID => $clientid,
                image_hash => $img_data->{md5},
                name => $fname,
            };
        }
    }

    # add_items добавляет в структуры в значениях %items_to_add_to_images_pool поле imp_id
    if ( values %items_to_add_to_images_pool ) {
        BannerImages::Pool::add_items( [ values %items_to_add_to_images_pool ] );
    }

    my %image_ad_pool_item_by_image_hash;
    if ( values %images_to_save_for_image_ads ) {
        my %hashes_to_add_to_pool = map { $_ => 1 } keys %images_to_save_for_image_ads;

        my %images_by_namespaces;
        for my $item (values %images_to_save_for_image_ads) {
            push @{ $images_by_namespaces{ $item->namespace() } }, $item;
        }

        for my $namespace (keys %images_by_namespaces) {
            my $format_manager = Direct::Model::ImageFormat::Manager->new(items => $images_by_namespaces{$namespace});
            my $errors = $format_manager->save( shard => get_shard(ClientID => $clientid), namespace => $namespace );

            if (%$errors) {
                for my $ad_image_hash ( keys %$errors ) {
                    for my $item ( @{ $rs_items_by_image_hash{$ad_image_hash} } ) {
                        next if $item->has_errors;
                        $item->add_error(error_OperationFailed(iget("Ошибка при сохранении файла")));
                    }

                    delete $hashes_to_add_to_pool{$ad_image_hash};
                }
            }
        }

        my @pool_items;
        for my $ad_image_hash ( keys %hashes_to_add_to_pool ) {
            my $image_model = $images_to_save_for_image_ads{$ad_image_hash};
            my $pool_item = Direct::Model::ImagePool->new(
                client_id => $clientid,
                name => $image_model->name,
                hash => $ad_image_hash,
            );

            push @pool_items, $pool_item;
            $image_ad_pool_item_by_image_hash{$ad_image_hash} = $pool_item;
        }

        my $pool_manager = Direct::Model::ImagePool::Manager->new(items => \@pool_items);
        $pool_manager->create;
    }

    for my $image_hash (keys %rs_items_by_image_hash) {
        my $items = $rs_items_by_image_hash{$image_hash};

        for my $item (@$items) {
            next if $item->has_errors;

            if ($item->object->{_image_size_valid_for_image_ad}) {
                my $pool_item = $image_ad_pool_item_by_image_hash{$image_hash};
                if ($pool_item->pool_id) {
                    $item->object->{AdImageHash} = $image_hash;
                } else {
                    # "такого быть не должно" -- исключительная ситуация, напишем в error log, кто-нибудь посмотрит и поправит
                    warn "AdImages.add: error adding image to pool";
                    $item->add_error(error_OperationFailed('Не удалось сохранить изображение'));
                }
            } else {
                my $bimp_item = $items_to_add_to_images_pool{$image_hash};
                if ($bimp_item->{imp_id}) {
                    $item->object->{AdImageHash} = $bimp_item->{image_hash};
                } else {
                    # "такого быть не должно" -- исключительная ситуация, напишем в error log, кто-нибудь посмотрит и поправит
                    warn "AdImages.add: error adding image to pool";
                    $item->add_error(error_OperationFailed('Не удалось сохранить изображение'));
                }
            }
        }
    }

    my @image_hashes = map { $_->object->{AdImageHash} } $rs->list;

    $self->units_withdraw_for_results( adimage => $rs );
    return {
        AddResults => $rs->prepare_for_xml_with_ids(@image_hashes),
    };
}

sub _validate_add_request {
    my ($self, $rs) = @_;

    unless ($self->can_write_client_objects()) {
        return error_NoRights_CantWrite();
    }

    if ($rs->count > $ADIMAGES_ADD_LIMIT) {
        return error_RequestLimitExceeded(
            iget('Разрешено загружать не более %s изображений в одном запросе', $ADIMAGES_ADD_LIMIT));
    }

    return;
}

sub _validate_add_request_items {
    my ($self, $rs) = @_;

    my $clientid = $self->subclient_client_id;

    my $limits = BannerImages::Pool::get_users_pool_limits( { $clientid => $self->subclient_uid } );
    my ($limits_for_clientid) = grep { $_->{ClientID} == $clientid } @$limits;
    die "Couldn't get limits for ClientID=$clientid" unless $limits_for_clientid;

    my $allowed_images = $limits_for_clientid->{total} - $limits_for_clientid->{cnt};

    for my $item ($rs->list_ok) {
        $allowed_images--;
        if ( $allowed_images < 0 ) {
            $item->add_error(error_ReachLimit(iget('Достигнуто максимальное количество изображений')));
            next;
        }

        unless ($item->object->{Name} =~ /\S/) {
            $item->add_error(error_BadParams(iget('Строка в поле Name не должна быть пустой или состоять только из пробелов')));
        }

        if (length($item->object->{Name}) > $MAX_NAME_LENGTH) {
            $item->add_error(error_MaxLength(undef, field => 'Name', length => $MAX_NAME_LENGTH));
        }
        if (length $item->object->{ImageData} > $BannerImages::MAX_IMAGE_FILE_SIZE) {
            $item->add_error(error_MaxFileSizeExceeded(iget(
                "Размер файла изображения больше допустимого (%d МБ)", $BannerImages::MAX_IMAGE_FILE_SIZE_IN_MEGABYTES
            )));
            next;
        }
    }

    return;
}

sub _validate_checked_add_request_items {
    my ($self, $rs) = @_;
    for my $item ($rs->list_ok) {
        my $check_result = $item->object->{_check_result};
        if ($check_result->{error}) {
            $item->add_error(error_InvalidFormat($check_result->{error}));
        } else {
            unless ($item->object->{_image_size_valid_for_image_ad}) {
                my $image_type = BannerImages::banner_image_check_size($check_result->{width}, $check_result->{height});

                if (!$image_type || $image_type eq 'small') {
                    $item->add_error(error_InvalidFormat(iget('Размер изображения некорректен')));
                }
            }
        }
    }

    return;
}

sub _validate_image_ad_size {
    my ($self, $rs) = @_;
    for my $item ($rs->list_ok) {
        if ($item->object->{image_type} eq 'image_ad') {
            $item->add_error(error_MaxFileSizeExceeded(iget(
                "Для изображения с разрешением %dx%d размер файла больше допустимого (%d КБ)",
                $item->object->{_check_result}{width},
                $item->object->{_check_result}{height},
                $BannerImages::MAX_IMAGEAD_FILE_SIZE_IN_KILOBYTES
            ))) if length $item->object->{ImageData} > $BannerImages::MAX_IMAGEAD_FILE_SIZE;
            next;
        }
    }

    return;
}

=head2 delete

=cut

sub delete {
    my ($self, $request) = @_;

    unless ($self->can_write_client_objects()) {
        return error_NoRights_CantWrite();
    }

    my $req = API::Service::Request::Delete->new( $request );

    # check ids per request limit
    my $limits_err = $req->validate( limits => { AdImageHashes => $ADIMAGES_DELETE_LIMIT } );
    return $limits_err if defined $limits_err;

    my $rs = API::Service::AdImages::ResultSet::Delete->new(
        map { +{ AdImageHash => $_ } } $req->selection_attribute('AdImageHashes')
    );

    for my $item ($rs->list_ok) {
        my $image_hash = $item->object->{AdImageHash};

        if ($image_hash eq '') {
            $item->add_error( error_BadParams("AdImageHash не может быть пустым") );
            next;
        }

        # Fail Fast: если ImageHash неправдоподобный, базу про него не спрашиваем, потому что его там нет
        # дальше все циклы ходят по $rs->list_ok, так что эту запись с ошибкой больше никто обрабатывать не будет
        if (!HashingTools::is_valid_md5_base64ya($image_hash)) {
            $item->add_error( error_NotFound( iget('Изображение не найдено') ) );
            next;
        }
    }

    # check duplicates
    $rs->add_error_for_id_dups(
        error_Duplicated(
            iget('AdImageHash присутствует в запросе более одного раза')
        )
    );

    # AdImageHash => объект из $rs->list
    # элементы, у которых уже есть ошибка (про дублирование), в этом хэше отсутствуют
    my %image_hash_to_result_item = map { $_->object->{AdImageHash} => $_ } $rs->list_ok;

    # найти image pool ids
    my $pool = $self->get_images(
        { image_hash => [ keys %image_hash_to_result_item ] },
        associated_info => 1,
    );

    # AdImageHash => объект из $pool
    my %image_hash_to_pool_item = map { $_->{image_hash} => $_ } @$pool;

    # если какие-то картинки не нашлись или привязаны к объявлению, записать в них ошибку;
    # остальные пометить к удалению
    my @imp_ids_to_delete;
    for my $item ($rs->list_ok) {
        my $image_hash = $item->object->{AdImageHash};

        unless ( exists $image_hash_to_pool_item{$image_hash} ) {
            $item->add_error( error_NotFound( iget('Изображение не найдено') ) );
            next;
        }

        my $pool_item = $image_hash_to_pool_item{$image_hash};
        if ($pool_item->{assigned}) {
            $item->add_error( error_CantDelete('Невозможно удалить привязанное изображение'));
            next;
        }

        push @imp_ids_to_delete, $image_hash_to_pool_item{$image_hash}->{imp_id};
    }

    # удалить image pool ids
    if (@imp_ids_to_delete) {
        BannerImages::Pool::delete_items(\@imp_ids_to_delete);
    }

    $self->units_withdraw_for_results( adimage => $rs );

    return { DeleteResults => $rs->prepare_for_xml };
}

1;
