package API::Service::Base;

=pod

    $Id$

=head1 NAME

    API::Service::Base

=head1 SYNOPSIS


=head1 DESCRIPTION

    Базовый класс для сервисов API5.

=head1 METHODS

=cut

use Direct::Modern;

use Carp;
use Module::Loaded qw/is_loaded/;
use Module::Load qw/load/;
use Try::Tiny;

use API::Converter;
use API::Service::CampaignsAvailabilityChecker;
use API::Units::Costs;
use API::Units::FreeOfChargeRoles;

use RBACDirect qw/
    rbac_check_allow_delete_camps
    rbac_check_allow_show_camps
    rbac_user_allow_edit_camps_detail
/;
use Rbac qw/$REP_READONLY/;

use Settings;
use Tools; # _save_vars($uid)

use UnitsFactory;
use Yandex::DBTools;
use Yandex::Log::Messages;
use Yandex::Trace;

use Campaign::Creator;
use Client qw/get_client_currencies/;
use DirectContext;
use Direct::Errors::Messages;

sub new {
    my ($class, $service_name) = @_;
    my $self = bless {
        _stash => {},
        _service_name => $service_name
    }, $_[0];
    return $self;
}

sub service_name { shift->{_service_name} }
sub current_operation { shift->{_current_operation} }

=head2 is_available_on_production

    Возвращает true, служит для переопределния в сервисах, которые нужно
    временно отключить на продакшн-среде, для скрытого запуска

=cut

sub is_available_on_production { 1 }

=head2 access_allowed_to_application

    Возвращает true, служит для переопределения в сервисах, которые нужно
    временно отключить на продакшн-среде для приложений не из белого списка
    для скрытого запуска

=cut

sub access_allowed_to_application {
    my ( $self, $application_id ) = @_;
    return 1;
}

# запускать перед обработкой каждого реквеcта
sub flush {
    my $self = shift;

    $self->{_stash} = {};

    delete @$self{qw/
        _authorization
        _locale
        _ppc_shard
        _subclient
        _subclient_role
        _units
        _units_spent
        _costs
        _affected_cids
        _units_multiplier
        _http_host
        _units_bucket
        _campaign_creator
        _http_request_headers
        _http_response_headers
        _direct_context
	_http_request_headers_for_log
    /};
}

sub stash { shift->{_stash} }

sub name {
    my $self = shift;

    unless ($self->{_name}) {
        my $pkg = ref $self;
        $pkg =~  /::(.+?)$/;
        $self->{_name} = $1;
    }

    return $self->{_name};
}

sub set_current_operation { $_[0]->{_current_operation} = $_[1] }

sub set_authorization {
    my $self = shift;
    my $auth = shift;
    $self->{_authorization} = $auth;

    Tools::_save_vars($self->operator_uid); # FIXME проверить нужен ли нам Tools, и если да, то критично ли нам заданность $UID
}

=head2 set_subclient($subclient)

    Запоминаем сабклиента (указанного в Client-Login или представитель оператора)
    $subclient - объект API::Authorization::User

=cut

sub set_subclient {
    my ($self, $subclient) = @_;
    $self->{_subclient} = $subclient;
    $self->{_subclient_role} = $subclient->role;
}

=head2 set_units_bucket($bucket)

    Устанавливает объект API::Units::Bucket как настройки корзины, из которой списываются баллы за операцию

=cut

sub set_units_bucket {
    my ($self, $bucket) = @_;
    $self->{_units} = undef;
    $self->{_units_bucket} = $bucket;
}

=head2 units_spent

    Баллов потрачено за текущую (последнюю) операцию

=cut

sub units_spent { shift->{_units_spent} || 0 }

=head2 units_limit

    Лимит баллов на клиента, за период (сутки на момент написания)

=cut

sub units_limit { shift->units->limit }

=head2 units_balance

    Осталось баллов, после операции

=cut

sub units_balance { shift->units->balance }

=head2 affected_cids(|\@cids)

    Ассесор для списка id кампаний использовавшихся в последнем запросе. Должен
    заполнятся в процессе выполнения операции, везде где это применимо. Отсюда
    после выполнения операции список попадает в log. Автоматически флешится на
    каждый запрос. Если cid-ы не заданы возвращает []

    Пишем кампани и на которые у оператора есть права. Для Get запросов пишем тоже.

=cut

sub affected_cids {
    my ($self, $cids) = @_;

    if(defined $cids) {
        die "cids should be an array" unless ref $cids eq 'ARRAY';
        $self->{_affected_cids} = $cids;
    }

    return exists $self->{_affected_cids} ? $self->{_affected_cids} : [];

}

=head2 $service->response_ids($response)

    Достать из запроса список ID, которые он обработал; потом они запишутся в лог запросов.

    Возвращать должен arrayref или undef; если undef, в лог ID не пишутся.

    Метод в родительском классе хорошо умеет извлекать и не извлекать ID для вызовов add/delete/get/update.
    Если в сервисе есть другие методы, может быть, стоит переопределить.

=cut

sub response_ids {
    my ($self, $response) = @_;

    return undef if $self->current_operation eq 'get';

    ## что в ответе 1 элемент (например: AddResponse), так что осмысленное содержание ответа
    ## содержится в единственном значении в values %$response
    my ($response_data) = values %$response;

    my @ids;
    for my $response_entry (@$response_data) {
        if ( $response_entry->{Id} ) {
            push @ids, $response_entry->{Id};
        }
    }
    return \@ids;
}

=head2 count_error_warning_objects_in_reponse($reponse)

    считает количество записей ответа содержащий ошибку и количество объектов
    содержащий ворнинг, количество ворнингов и ошибок внутри объекта не имеет
    значения

    Возвращает ($errors_count, $warnings_count)
=cut

sub count_error_warning_objects_in_reponse {
    my ($self, $response) = @_;

    return undef if $self->current_operation eq 'get';

    ## что в ответе 1 элемент (например: AddResponse), так что осмысленное содержание ответа
    ## содержится в единственном значении в values %$response
    my ($response_data) = values %$response;

    my ($errors_count, $warnings_count);
    for my $response_entry (@$response_data) {
        $warnings_count++ if exists $response_entry->{Warnings};
        $errors_count++ if exists $response_entry->{Errors};
    }
    return ($errors_count, $warnings_count);
}

=head2 $service->include_response_in_logs()

    Надо ли записывать полный ответ сервиса в текстовые логи

=cut

sub include_response_in_logs {
    my ($self) = @_;

    if ( $self->current_operation eq 'get' ) {
        return 0;
    }

    return 1;
}

=head2 $service->include_response_in_logs()

    Надо ли записывать заголовки запроса сервиса в текстовые логи

=cut

sub include_request_headers_in_logs {
    return 0;
}

sub _subclient { shift->{_subclient} }

sub _authorization { shift->{_authorization} or die "no authorization set" }

=head2 operator_login
=head2 operator_client_id
=head2 operator_uid
=head2 operator_role
=head2 is_operator_client
=head2 is_operator_agency
=head2 is_operator_super
=head2 is_operator_superreader

    operator_login и operator_uid залогиненого пользователя
    могут отличаются от operator_chief_rep_login в случае если
    залогиненый пользователь является представителем
    отличным от главного - operator_chief_rep_{login|uid}
    будут логин и uid главного представителя)

=cut

sub operator_login { shift->_authorization->operator_login }
sub operator_uid { shift->_authorization->operator_uid }
sub operator_client_id { shift->_authorization->operator_user->ClientID }
sub operator_role { shift->_authorization->role }
sub is_operator_client { shift->_authorization->is_role_client }
sub is_operator_agency { shift->_authorization->is_role_agency }
sub is_operator_super { shift->_authorization->is_role_super }
sub is_operator_superreader { shift->_authorization->is_role_superreader }

sub operator_chief_rep_login { shift->_authorization->chief_rep_login }
sub operator_chief_rep_client_id { shift->_authorization->chief_rep_user->ClientID }
sub operator_chief_rep_uid { shift->_authorization->chief_rep_uid }

=head2 application_id

    ID приложения

=cut

sub application_id { shift->_authorization->application_id }

sub rights { shift->_authorization->rights }

=head2 subclient

    subclient - владелец объектов над которыми производится операция (т.е. чей uid указан в
    кампаниях, которым принадлежат объекты операции).

    subclient - главный представитель пользователя, чей логин указан в заголовке Client-Login.
    Для прямых клиентов равен главному представителю залогиненного пользователя.

=head2 subclient_login

    Логин главного представителя указанного в http-заголовке Client-Login пользователя

=head2 subclient_uid

    Uid главного представителя указанного в http-заголовке Client-Login пользователя

=head2 subclient_role

    Строка, роль главного представителя указанного в http-заголовке Client-Login пользователя

=head2 subclient_client_id { shift->_subclient->ClientID }

    ClientId клиента.

=head2 subclient_units_limit

    Лимит баллов указанного клиента. undef если не задан в БД, тогда будет использовано дефолтное из Units.pm.

=cut

sub subclient_login { shift->_subclient->login }
sub subclient_uid { shift->_subclient->uid }
sub subclient_role{ shift->{_subclient_role} }
sub subclient_client_id { shift->_subclient->ClientID }
sub subclient_units_limit { shift->_subclient->units_limit }

=head2 set_header_client_login
=head2 header_client_login

    Методы для сохранения и доступа к указанному в http-заголовке Client-Login login-у пользователя

    Это значение иногда требуется, например, при запросе агентством данных конкретно этого пользователя
    (в subclient_* данные главного представителя этого пользователя).

=cut

sub set_header_client_login {
    my ($self, $login) = @_;
    $self->{_header_client_login} = $login;
}

sub header_client_login { shift->{_header_client_login} }

=head2 units_bucket

    Объект API::Units::Bucket - настройки корзины, из которой будут списаны баллы за операцию

=cut

sub units_bucket { shift->{_units_bucket} }

=head2 ppc_shard

    объект с описание шарда с данными в котором мы работаем в рамках текущей операции

    дефакто PPC( ClientID => $self->subclient_client_id )

=cut

sub ppc_shard {
    my $self = shift;
    return $self->{_ppc_shard} ||= PPC( ClientID => $self->subclient_client_id );
}

sub locale { shift->{_locale} }

sub set_protocol {
    my $self = shift;
    $self->{_protocol} = shift;
}

sub protocol { shift->{_protocol} }

=head2 set_http_host($string)

    Запоминаем текущий хост из запроса, нужно для формирования URL, на файлы

=cut

sub set_http_host {
    my $self = shift;
    $self->{_http_host} = shift;
}

=head2 set_http_request_headers($headers)

    Запоминаем заголовки текущего запроса

=cut

sub set_http_request_headers {
    my ($self, $headers) = @_;
    $self->{_http_request_headers} = $headers;
}

=head2 get_http_request_header($header_name)

    Возвращает значение одного заголока текущего запроса

=cut

sub get_http_request_header {
    my ($self, $header_name) = @_;
    return $self->{_http_request_headers}->header($header_name);
}

=head2 http_response_headers($headers)

    Получение/изменение объекта с заголовками текущего ответа

=cut

sub http_response_headers {
    my ($self, $headers) = @_;

    if ($headers) {
        $self->{_http_response_headers} = $headers;
    }

    return $self->{_http_response_headers};
}

=head2 http_request_headers_for_log($headers)

    Получение/изменение объекта с заголовками текущего запроса (которые необходимо записать в логи)

=cut

sub http_request_headers_for_log {
    my ($self, $headers) = @_;

    if ($headers) {
        $self->{_http_request_headers_for_log} = $headers;
    }

    return $self->{_http_request_headers_for_log};
}

=head2 http_host

    Текущий хост из запроса

=cut

sub http_host { shift->{_http_host} }

sub nil {
    my $protocol = shift->protocol;
    return $protocol eq 'soap' || $protocol eq 'xml' ? 'NIL' : undef;
}

sub is_nil {
    my ($self, $value) = @_;
    my $protocol = $self->protocol;
    if($self->protocol eq 'soap' || $self->protocol eq 'xml') {
        if (defined $value) {
            if (ref $value eq 'HASH' && defined $value->{_}) {
                return $value->{_} eq 'NIL';
            } else {
                return "$value" eq $self->nil;
            }
        }
    } else { #json
        return !defined $value;
    }
}

sub set_locale {
    my ($self, $locale) = @_;
    $self->{_locale} = ($locale || 'en');
    Yandex::I18n::init_i18n($self->locale);
}


=head2 not_owned_campaign_ids(@campaign_ids)

    По списку id-шников кампаний возвращает список тех из них,
    что не принадлежат пользователю, указанному в заголовке.

    !!! Данный метод не имеет ничего общего с проверкой прав, которую надо
    делать от имени залогиненного пользователя !!!

=cut

sub not_owned_campaign_ids {
    my ($self, @campaign_ids) = @_;
    return () unless @campaign_ids;

    my $campaign_ids_to_uids = get_hash_sql(
        PPC(ClientID => $self->subclient_client_id),
        [
            "select cid, uid from campaigns",
            where => {
                cid => \@campaign_ids,
                'statusEmpty__not_in' => ['Yes']
            }
        ]
    );
    my @not_owned_campaigns;
    foreach my $cid (@campaign_ids) {
        push @not_owned_campaigns, $cid
            unless exists $campaign_ids_to_uids->{$cid}
                and $campaign_ids_to_uids->{$cid} eq $self->subclient_uid;
    }

    return @not_owned_campaigns;
}

=head2 get_campaigns_availability_checker($cids, %options)

    Возвращает объект для проверки доступности кампаний, указанных в $cids.
    В $supported_camp_kind задается надтип кампаний, который определяет какие типы кампаний поддерживаются.
    Опции:
        supported_camp_kind => 'api5_edit' надтип поддерживаемых типов кампаний,
        check_currency => 1 включает сравнение валюты кампании с валютой клиента,
        pass_archived => 1 включает доступность архивных кампаний,
        error_not_found объект ошибки для ненайденных кампаний,
        error_not_supported объект ошибки для неподдерживаемых кампаний,
        error_is_archived объект ошибки для архивных кампаний.
    Замечаие: get не использует этот метод. Пользуемся этим, чтобы поддержать кампании Дзена в get и не поддержать в других методах

=cut


sub get_campaigns_availability_checker {
    my ($self, $cids, %options) = @_;

    return API::Service::CampaignsAvailabilityChecker->new(
        subclient_uid => $self->subclient_uid,
        cids => $cids,
        supported_camp_kind => $options{supported_camp_kind} // 'api5_edit',
        subclient_currency => $options{check_currency} ? get_client_currencies( $self->subclient_client_id )->{work_currency} : '',
        pass_archived => $options{pass_archived} // 0,
        map { $options{$_} ? ( $_ => $options{$_} ) : ()} qw/error_not_found error_not_supported error_is_archived/
    );
}

=head2 get_unavailable_cids_map($cids, %options)

    Возвращаем хэш с идентификаторами недоступных (с учетом поддерживаемых типов, заданных надтипом $camp_kind) кампаний из заданных в $cids.
    Опции:
        pass_archived => 1 включает доступность архивных кампаний,
        supported_camp_kind => 'api5_edit' надтип поддерживаемых типов кампаний

=cut

sub get_unavailable_cids_map {
    my ($self, $cids, %options) = @_;
    return $self->get_campaigns_availability_checker($cids, supported_camp_kind => $options{supported_camp_kind}, pass_archived => $options{pass_archived})->get_unavailable_cids_map;
}

sub not_owned_campaign_ids_hash {
    my ($self, @campaign_ids) = @_;
    return map { $_ => 1 } $self->not_owned_campaign_ids(@campaign_ids);
}

=head2 check_campaigns_read(@campaign_ids)

    По списку id-шников кампаний возвращает хеш тех из них,
    что оператор не может просматривать.

    Id кампаний на которые есть права на чтение -- записываем в affected_cids.

=cut

sub check_campaigns_read {
    my ($self, @campaign_ids) = @_;
    return () unless @campaign_ids;
    my %has_access = %{rbac_check_allow_show_camps(undef, $self->operator_uid, \@campaign_ids)};
    my %no_access = map {$_ => undef} grep {!$has_access{$_}} keys %has_access;

    $self->affected_cids([ keys %has_access ]); # кампании участующие в запросе, пишем в лог

    return %no_access;
}

sub _no_rights_if_mediaplanner {
    my ($self, @campaign_ids) = @_;

    if($self->operator_role eq 'media') {
        return ( map {$_ => undef} @campaign_ids );
    }

    return;
}

=head2 check_campaigns_write_detail(@campaign_ids)

    По списку id-шников кампаний возвращает хеш тех из них,
    что оператор не может редактировать; значения в хеше буквенные:

    n — (not found/no rights) кампания не видна пользователю,
    r — (read only) кампания видна пользователю, нет права на запись.

    Id кампаний на которые есть хоть какие-то права -- записываем в affected_cids.

=cut

sub check_campaigns_write_detail {
    my ($self, @campaign_ids) = @_;
    return () unless @campaign_ids;

    my %ret = ();

    my %read_access = %{rbac_check_allow_show_camps(undef, $self->operator_uid, \@campaign_ids)};
    foreach my $cid(@campaign_ids) {
        next if exists $ret{$cid};
        unless ($read_access{$cid}) {
            $ret{$cid} = 'n';
        }
    }

    my %can_write;
    if(my %r = $self->_no_rights_if_mediaplanner(@campaign_ids)) {
        %can_write = %r;
    } else {
        %can_write = %{rbac_user_allow_edit_camps_detail(undef, $self->operator_uid, \@campaign_ids)};
    }

    foreach my $cid(@campaign_ids) {
        next if exists $ret{$cid};
        unless ($can_write{$cid}) {
            $ret{$cid} = 'r';
        }
    }

    my $affected_cids = [ grep { !exists $ret{$_} || $ret{$_} eq 'r' } @campaign_ids ];
    $self->affected_cids($affected_cids);

    return %ret;
}

=head2 grep_not_found(%camp_rights)

    по резульатату из check_campaings_write возвращает список campaign_id
    которые не найдены (не существуют или нет прав даже на чтение)

=cut

sub grep_not_found {
    my ($self, %camp_rights) = @_;
    return grep {
        $self->is_campaign_not_found($camp_rights{$_})
    } keys %camp_rights;
}

=head2 grep_no_write_rights(%camp_rights)

    по резульатату из check_campaings_write возвращает список campaign_id
    для которых у данного клиента нет прав на запись

=cut

sub grep_no_write_rights {
    my ($self, %camp_rights) = @_;
    return grep {
        $self->is_no_write_rights_on_campaign($camp_rights{$_})
    } keys %camp_rights;
}

sub is_campaign_not_found {$_[1] eq 'n' }
sub is_no_write_rights_on_campaign {$_[1] eq 'r' }

=head2 campaign_creator

    Объект Campaign::Creator, создается один раз на запрос

=cut

sub campaign_creator {
    my $self = shift;
    return $self->{_campaign_creator} //= Campaign::Creator->new(
        operator_role => $self->rights->role,
        operator_uid => $self->operator_uid,
        subclient_uid => $self->subclient_uid,
        subclient_client_id => $self->subclient_client_id,
        is_operator_readonly => $self->_authorization->operator_user->rep_type eq $REP_READONLY ? 1 : 0
    );
}

=head2 can_write_client_objects()

    Понять может ли клиент создавать и редактировать явно не связанные с кампаниями объекты,
    например сайтлинки.

    Права на клиента уже проверили, здесь проверим роли.

    Ответ:
    1 — есть право,
    0 — нет права.

=cut

sub can_write_client_objects {
    my $self = shift;

    # можем создавать объекты если можем создавать кампании
    return $self->campaign_creator->can_create_campaign ? 1 : 0;
}

=head2 check_campaigns_delete(@campaign_ids)

    По списку id кампаний возвращает хеш тех из них, которые оператор не может удалить

=cut

sub check_campaigns_delete {
    my ($self, @campaign_ids) = @_;

    return () unless @campaign_ids;

    my %r = $self->_no_rights_if_mediaplanner(@campaign_ids);
    if ( %r ) {
        return %r;
    }

    my $can_delete = rbac_check_allow_delete_camps( undef, $self->operator_uid, \@campaign_ids );
    my %cant_delete = map {$_ => undef} grep {!$can_delete->{$_}} keys %$can_delete;

    return %cant_delete;
}

=head2 costs

    Возвращает объект для взаимодействия с подсистемой расчета стоимости операций

    Параметры:
        нет

    Результат:
        Ссылка на объект API::Units::Costs

=cut

sub costs {
    my $self = shift;

    return $self->{_costs} //= API::Units::Costs->new( $self->service_name, $self->current_operation );
}

=head2 set_units_multiplier($coef)

    Устанавливает множитель на списываемые баллы (число с плавающей точкой)

=cut

sub set_units_multiplier {
    my ($self, $coef) = @_;
    $self->{_units_multiplier} = $coef;
}

=head2 units_multiplier

    Множитель на списываемые баллы (число с плавающей точкой)

=cut

sub units_multiplier { shift->{_units_multiplier} }

=head2 units

    Возвращает объект для взаимодействия с подсистемой учета баллов

    Параметры:
        нет

    Результат:
        Ссылка на объект Units

=cut

sub units {
    my $self = shift;

    unless ( $self->{_units} ) {
        $self->{_units} = UnitsFactory::create_adjusted(
                            $self->units_bucket->id,
                            $self->units_multiplier,
                            $self->units_bucket->limit
                        );
    }

    return $self->{_units};
}

=head2 units_enough_for_operation

    Вычисляет требуемое для выполнения операции количество баллов и проверяет их наличие у клиента

    Параметры:
        нет

    Результат:
        bool, 1 | 0

=cut

sub units_enough_for_operation {
    my $self = shift;

    return 1 if $self->is_free_of_charge_operator;

    my $cost = $self->costs->get_cost_for_operation();

    return $self->units->is_available($cost);
}

=head2 units_withdraw_for_operation

    Вычисляет требуемое для выполнения операции количество баллов и списывает их у клиента

    Параметры:
        нет

    Результат:
        bool, 1 | 0

=cut

sub units_withdraw_for_operation {
    my $self = shift;

    my $cost = $self->costs->get_cost_for_operation();

    return $self->units_reset_or_withdraw($cost);
}

=head2 units_withdraw_for_objects

    Вычисляет требуемое для выполнения операции над определенным количеством объектов количество баллов и списывает их у клиента

    Параметры:
        type   - тип объектов, строка, обязательный
        number - количество объектов, число, обязательный

    Результат:
        bool, 1 | 0

=cut

sub units_withdraw_for_objects {
    my ( $self, $type, $number ) = @_;

    my $cost = $self->costs->get_cost_for_objects( $type, $number );

    return $self->units_reset_or_withdraw($cost);
}


=head2 units_withdraw_for_objects_with_errors($number)

    Вычисляет количество баллов, списываемое за объекты с ошибками, и списывает его у клиента

    Параметры:
        $number - количество объектов с ошибками, число, обязательный

    Результат:
        bool, 1 | 0

=cut

sub units_withdraw_for_objects_with_errors {
    my ( $self, $number ) = @_;
    my $cost = $self->costs->get_cost_for_objects_with_errors( $number );

    return $self->units_reset_or_withdraw($cost);
}

=head2 units_withdraw_for_results($objects_type, $resultset)

    Вычисляет количество баллов по результату операции и списывает его у клиента

    Параметры:
        $objects_type - тип объекта
        $resultset    - объект API::Service::ResultSet или наследник

=cut

sub units_withdraw_for_results {
    my ( $self, $objects_type, $result_set ) = @_;

    my $profile = Yandex::Trace::new_profile('api:redis:units_withdraw_for_results');
    $self->units_withdraw_for_objects_with_errors( $result_set->count_failed );
    $self->units_withdraw_for_objects( $objects_type => $result_set->count_ok );
}

=head2 units_withdraw_for_request_error()

    Метод для списания баллов за ошибку запроса (не верно переданы параметр и т.п.)

=cut

sub units_withdraw_for_request_error {
    my ( $self ) = @_;

    my $cost = $self->costs->get_cost_for_request_error();

    return $self->units_reset_or_withdraw($cost);
}

=head2 units_reset_or_withdraw($amount)

    Списываем, если необходимо, $amount баллов с саб-клиента пользователя (оператора или сабклиента)
    Если принимается решение о том, что баллы списывать не нужно то счетчик
    потраченных баллов обнуляется, чтобы не врать в логах, и заголовках ответа

    Успешное списание, возвращаем true

=cut

sub units_reset_or_withdraw {
    my ($self, $amount) = @_;

    if($self->is_free_of_charge_operator) {
        # роли для которых списывать не надо
        $self->{_units_spent} = 0;
    } else {
        my $spent = $self->units->withdraw($amount);
        return unless defined $spent; # мемкэш лежит
        $self->{_units_spent} += $spent;
    }

    return 1; # успех, даже если списывать ничего не надо было
}

=head2 is_free_of_charge_operator

    True если при работе из под текущего оператора (сотрудника яндекса) не
    нужно списывать баллы с сабклиента

=cut

sub is_free_of_charge_operator { return API::Units::FreeOfChargeRoles::is_free_role(shift->operator_role) }

=head2 converter($method, $type)

    Возврвщает конвертер для текущего сервиса и метода

    Параметры:
        $type - тип конвертера internal или external

    Результат:
        API::Converter

=cut

sub converter($$){
    my ($self, $type) = @_;
    my $map_module = 'API::Service::'.$self->service_name.'::ConvertMap';
    try {
        load $map_module unless is_loaded($map_module);
    } catch {
        die "converter for service ".$self->service_name." need module $map_module: $_";
    };
    my $method = $self->current_operation();
    my $convert_map = $map_module->new(method => $method);
    unless (defined $self->{converter}{$method}{$type}){
        if ($type eq 'internal') {
            $self->{converter}{$method}{$type} =
                API::Converter->new(map => $convert_map->to_internal_map);
        } elsif ($type eq 'external') {
            $self->{converter}{$method}{$type} =
                API::Converter->new(map => $convert_map->to_external_map);
        } else { die 'unknown converter'}
    }
    return $self->{converter}{$method}{$type};
}

=head2 write_to_messages_log(@log_data)

    Пришет в messages.log через Yandex::Log::Messages

=cut

sub write_to_messages_log {
    my ($self, @log_data) = @_;
    $self->{messages_log} //= Yandex::Log::Messages->new();
    $self->{messages_log}->bulk_out(@log_data);
    return;
}

=head2 item_value_check_id

    Проверяет id внутри item-а (из ResultSet) на положительность,
    (цело)численность. Item нужен чтобы добавить к нему ошибку в случае провала
    проверки. Метод должен быть в контроллере, чтобы был доступ к
    $self->is_nil.

=cut

sub item_value_check_id {
    my ($self, $item, $field, $value) = @_;
    $item->add_error( error_InvalidField_NotPositive(undef, field => $field) ) unless $self->is_nil($value) || ($value =~ /^\d+$/ && $value > 0);
}

=head2 grep_owned_campaign_ids($self, $cids)

    По списку id'шников кампаний, получает список id'шников кампаний принадлежащих клиенту

=cut

sub grep_owned_campaign_ids {
     my ($self, $cids) = @_; # self, []

     my %no_read_rights = ( $self->not_owned_campaign_ids_hash(@$cids) );
     return [ grep { !exists $no_read_rights{$_} } @$cids ];
 }

=head2 grep_read_access_campaign_ids($self, $cids)

    По списку id'шников кампаний, получает список id'шников кампаний, для которых у клиента есть права на чтение

=cut

sub grep_read_access_campaign_ids {
     my ($self, $cids) = @_; # self, []

     my %no_read_rights = ( $self->check_campaigns_read(@$cids) );
     return [ grep { !exists $no_read_rights{$_} } @$cids ];
 }

=head2 get_direct_context

    Метод возвращает экземпляр-синглтон класса DirectContext

=cut

sub get_direct_context {
    my ($self) = @_;

    if (! defined $self->{_direct_context}) {
        $self->{_direct_context} = DirectContext->new({
            is_direct        => 1,
            UID              => $self->operator_uid,
            uid              => $self->subclient_uid,
            client_chief_uid => $self->subclient_uid, # это и есть uid chief-а
            client_client_id => $self->subclient_client_id,
            login_rights     => $self->rights,
        });
    }

    return $self->{_direct_context};
}

1;
