package API::Service::Campaigns;

use Direct::Modern;

use Try::Tiny;

use Carp qw/croak/;
use DateTime::TimeZone;
use List::MoreUtils qw/each_array uniq any all none/;
use Storable qw/dclone/;

use Yandex::I18n;
use Yandex::HashUtils qw/hash_merge hash_cut/;
use Yandex::ListUtils qw/xminus xisect xdiff/;
use Yandex::Validate qw/is_valid_id is_valid_int/;

use BillingAggregateTools;
use Campaign qw/mass_get_servicing create_empty_camp
    save_camp camp_save_metrika_counters
    mass_get_source_ids_by_cid get_camp_info get_camp_sum_available
    send_camp_to_service is_autobudget mass_camps_has_banners is_campaign_strategy_use_meaningful_goals_optimization
    is_strategy_change_ignore_platform/;
use CampaignTools qw/camp_metrika_counters_multi set_status_empty_to_no/;

use Campaign::Types qw/get_camp_kind_types camp_kind_in get_camp_type_multi/;
use Client qw/
    check_add_client_campaigns_limits
    get_client_currencies
    get_client_data
    get_client_discount
    get_client_limits
/;
use Common qw/:subs/;
use Currencies qw/calc_bonus/;
use Currency::Format qw/format_const/;
use Models::Campaign qw//;
use Models::CampaignOperations qw/mass_check_block_money_camps/;
use PrimitivesIds qw/get_clientid get_cids/;
use Primitives qw/
    get_other_manager_uid
    get_idm_primary_manager_uid
    filter_archived_campaigns
    get_attribution_model_or_default_by_type
/;
use RBACDirect;
use Settings;
use Stat::OrderStatDay;
use TimeTarget;
use User qw/get_user_data/;
use Wallet;
use Property;

use Direct::Errors::Messages;

use API::Converter::ConvertSubs qw/convert_to_money/;

use API::Service::ResultSet::Archive;
use API::Service::ResultSet::Unarchive;

use API::Service::Request::Archive;
use API::Service::Request::Unarchive;

use API::Service::Campaigns::AddResults;
use API::Service::Campaigns::GetRequest;
use API::Service::Campaigns::DeleteRequest;
use API::Service::Campaigns::ResumeRequest;
use API::Service::Campaigns::SuspendRequest;
use API::Service::Campaigns::Statuses qw/
    status_from_model state_from_model status_payment_from_model
    statuses_where states_where status_payment_where
    merge_statuses_where
/;
use API::Service::Campaigns::Strategies qw/
    get_structure_name_by_strategy
    is_strategy_has_no_more_than_one_structures
    is_strategy_consistent
    is_strategy_has_needed_structure
    merge_strategies
    is_network_default_strategy
/;
use API::Service::Campaigns::ConvertSubs qw/
    convert_finance_to_external convert_settings_to_external
    convert_sms_time_string_to_struct convert_placement_types convert_placement_types_to_external
/;
use API::Service::Campaigns::Types qw/
    prepare_type_structure get_structure_name_by_type get_type_by_structure disclose_type_structure
    get_external_types has_type_structure convert_type_to_external get_type_structure_name
    get_type_structures_names convert_types_to_external convert_type
/;
use API::Error::ToExceptionNotification qw/action_errors/;

use API::Service::Campaigns::PermissionChecks;
use API::Service::Campaigns::ResultSet;

use Direct::Campaigns;
use Direct::Campaigns::Dynamic;
use Direct::Campaigns::MobileContent;
use Direct::Campaigns::Text;
use Direct::Model::Campaign;
use Direct::Model::CampaignDynamic;
use Direct::Model::CampaignMobileContent;
use Direct::Model::CampaignText;
use Direct::Model::CampaignCpmBanner;
use Direct::Model::CampaignContentPromotion;
use Direct::Validation::Campaigns qw/
    validate_add_campaign validate_update_campaign validate_delete_campaigns
    validate_resume_campaigns validate_suspend_campaigns
/;
use Direct::Validation::Campaigns::Constraints;
use Direct::Validation::DayBudget qw/validate_camp_day_budget/;

use Direct::Validation::Domains;

use constant SUPPORTED_CAMP_KIND => 'api5_edit_campaigns';

use base qw/API::Service::Base/;

our $ADD_IDS_LIMIT       //= 10;
our $ARCHIVE_IDS_LIMIT   //= 1000;
our $DELETE_IDS_LIMIT    //= 1000;
our $GET_IDS_LIMIT       //= 1000;
our $RESUME_IDS_LIMIT    //= 1000;
our $SUSPEND_IDS_LIMIT   //= 1000;
our $UNARCHIVE_IDS_LIMIT //= 1000;
our $UPDATE_IDS_LIMIT    //= 10;

my %DEFECT_DESCRIPTION_FIELD_MAP = (
    campaign_name     => { field => 'Name', length => $Direct::Validation::Campaigns::MAX_TITLE_LENGTH, },
    client_fio        => { field => 'ClientInfo', length => $Direct::Validation::Campaigns::MAX_CLIENT_FIO_LENGTH, },
    email             => { field => 'Email', length => $Direct::Validation::Campaigns::MAX_EMAIL_LENGTH, },
    start_date        => { field => 'StartDate', },
    finish_date       => { field => 'EndDate', from => 'StartDate', to => 'EndDate' },
    disabled_domains  => { field => 'ExcludedSites', },
    disabled_ips      => { field => 'BlockedIps', },
    timezone_id       => { field => 'TimeZone', },
    time_target       => { field => 'TimeTargeting', },
    minus_words       => { field => 'NegativeKeywords', },
    broad_match_limit => { field => 'BudgetPercent', },
    metrika_counters  => { field => 'CounterIds', },
    meaningful_goals  => { field => 'PriorityGoals', },
    sms_time          => { from => 'TimeFrom', to => 'TimeTo', },
    money_warning_threshold => { field => 'WarningBalance', },
    position_check_interval => { field => 'CheckPositionInterval', },
    broad_match_goal_id => { field => 'OptimizeGoalId' },
);

my %STRATEGY_DEFECT_DESCRIPTION_FIELD_MAP = (
    start  => { field => 'StartDate', },
    finish => { field => 'EndDate', from => 'StartDate', to => 'EndDate' },
);

my $HAS_NETWORK_DEFAULT_LIMIT_PERCENT_FLAG_NAME = 'has_network_default_limit_percent';
my $HAS_MAINTAIN_NETWORK_CPC_SETTING_FLAG_NAME = 'has_maintain_network_cpc_setting';

my $DEFAULT_LIMIT_PERCENT = 0;

my @TIME_TARGET_FIELDS = qw/timeTarget time_target_holiday time_target_holiday_dont_show time_target_holiday_coef
    time_target_holiday_from time_target_holiday_to time_target_working_holiday/;
my @SMS_SETTINGS_FIELDS = qw/sms_time_hour_from sms_time_hour_to sms_time_min_from sms_time_min_to active_orders_money_warning_sms
    active_orders_money_out_sms notify_order_money_in_sms moderate_result_sms notify_metrica_control_sms camp_finished_sms/;
my $TYPES_WITH_NO_EXCLUDED_DOMAIN_ITEMS = {content_promotion => 1};

my $add_metrica_tag_property = Property->new('default_yes_for_add_metrica_tag');
my $spend_with_nds_property = Property->new('return_campaign_spend_with_nds_in_api');
my $save_available_counters_goals_on_campaign_saving_property =
    Property->new('save_available_counters_goals_on_campaign_saving_in_api');

my $BUSINESS_CLASS_BY_TYPE = { map {$_->supported_type => $_} qw/
        Direct::Campaigns::Dynamic
        Direct::Campaigns::Performance
        Direct::Campaigns::MobileContent
        Direct::Campaigns::Text
        Direct::Campaigns::CpmBanner/ };

=head2 get_business_logic_class_by_type

    Получение директовой модели по типу, используемому в campaigns-методах API

=cut

sub get_business_logic_class_by_type {
    my ($self, $type) = @_;
    return $BUSINESS_CLASS_BY_TYPE->{$type};
}

=head2 get

    получение кампаний

=cut

sub get {
    my ($self, $request) = @_;

    my $get_request = API::Service::Campaigns::GetRequest->new($request);

    if (my $e = $get_request->validate(limits => { Ids => $GET_IDS_LIMIT })) { return $e }

    my $client_currencies = get_client_currencies($self->subclient_client_id);
    my $work_currency = $client_currencies->{work_currency};

    my $all_campaigns = $self->_get_campaigns($get_request, $work_currency);

    return { Campaigns => [] } unless @$all_campaigns;

    my $too_many_camps = scalar(@$all_campaigns) > $get_request->page_limit;

    my @limited_campaigns = $too_many_camps
                                ? @$all_campaigns[0..$get_request->page_limit - 1]
                                : @$all_campaigns;

    my $preprocess_campaigns = $self->_preprocess_get(\@limited_campaigns, $get_request, $work_currency);

    my $converted_campaigns = $self->converter('external')->convert($preprocess_campaigns);

    for my $campaign (@$converted_campaigns){
        my $type = get_structure_name_by_type($campaign->{Type});
        my @needed_fields = $get_request->needed_field_names_by_type($type);
        my $is_need_type = $get_request->is_in_field_names('Type');
        if (!$is_need_type) {
            @needed_fields = ('Type', @needed_fields);
        }
        $campaign = hash_cut($campaign, @needed_fields);
        $self->_prepare_nillable_fields($campaign);
        $self->_daily_budget_default_to_nil($campaign);
        $campaign->{$type} = prepare_type_structure($get_request, $campaign);
        $self->_process_pay_for_install($campaign);
        $self->_process_autobudget_avg_cpv($campaign);
        delete $campaign->{$type} unless defined $campaign->{$type};
        if (!$is_need_type) {
            delete $campaign->{Type};
        }
    }

    $self->units_withdraw_for_objects(campaign => scalar(@$converted_campaigns));

    my $result = { Campaigns => $converted_campaigns };

    if ($too_many_camps) {
        $result->{LimitedBy} = $get_request->page_limit + $get_request->page_offset;
    }

    return $result;
}

=head2 _process_autobudget_avg_cpv

    временный метод, заменяющий стратегии autobudget_avg_cpv и autobudget_avg_cpv_custom_period на UNKNOWN, чтобы не падать при валидации по WSDL,
    в котором эти стратегии пока не поддержаны

=cut

sub _process_autobudget_avg_cpv {
    my ($self, $campaign) = @_;

    if ($campaign->{CpmBannerCampaign} && exists $campaign->{CpmBannerCampaign}{BiddingStrategy}) {
        my $bidding_strategy = $campaign->{CpmBannerCampaign}{BiddingStrategy};
        foreach my $ad_network_type (qw/Search Network/) {
            if (exists $bidding_strategy->{$ad_network_type}
                && exists $bidding_strategy->{$ad_network_type}{BiddingStrategyType} && !defined $bidding_strategy->{$ad_network_type}{BiddingStrategyType}) {
                $bidding_strategy->{$ad_network_type}{BiddingStrategyType} = 'UNKNOWN';
                delete $bidding_strategy->{$ad_network_type}{''};
            }
        }
    }
    return;
}

=head2 _process_pay_for_install

    временный метод, заменяющий стратегию PAY_FOR_INSTALL на UNKNOWN, чтобы не падать при валидации по WSDL,
    в котором эта стратегия пока не поддержана

=cut

sub _process_pay_for_install {
    my ($self, $campaign) = @_;

    if ($campaign->{MobileAppCampaign} && exists $campaign->{MobileAppCampaign}{BiddingStrategy}) {
        my $bidding_strategy = $campaign->{MobileAppCampaign}{BiddingStrategy};
        foreach my $ad_network_type (qw/Search Network/) {
            if (exists $bidding_strategy->{$ad_network_type} && $bidding_strategy->{$ad_network_type}{BiddingStrategyType} eq 'PAY_FOR_INSTALL') {
                $bidding_strategy->{$ad_network_type}{BiddingStrategyType} = 'UNKNOWN';
                delete $bidding_strategy->{$ad_network_type}{PayForInstall};
            }
        }
    }
    return;
}


sub _preprocess_get {
    my ($self, $models, $get_request, $work_currency) = @_;

    my $source_ids;
    if ($get_request->is_in_field_names('SourceId')) {
        $source_ids = mass_get_source_ids_by_cid($self->subclient_client_id);
    }

    my @cids = map { $_->{cid} } @$models;

    my $metrika_counters;
    if ($get_request->is_in_field_names_for_all_types('CounterIds') || $get_request->is_in_field_names_for_all_types('CounterId')) {
        $metrika_counters = camp_metrika_counters_multi(\@cids);
    }

    my $can_use_day_budget = 'Yes';

    my ($favorite_camps, $servicing_camps);
    if ($get_request->is_in_field_names_for_all_types('Settings')) {
        $favorite_camps = mass_get_favorite_camps($self->subclient_uid);
        $servicing_camps = mass_get_servicing(\@cids);
    }

    if ($get_request->is_in_field_names('StatusClarification')) {
        if ( my @models_without_banners = grep { ! exists $_->{has_banners} } @$models ) {
            my $camp_has_banners = mass_camps_has_banners(\@models_without_banners);
            for my $model (@models_without_banners) {
                $model->{has_banners} = $camp_has_banners->{ $model->{cid} };
            }
        }
    }

    my $status_clarifications;
    if ($get_request->is_in_field_names('StatusClarification')) {
        $status_clarifications = Campaign::CalcCampStatus_mass($models);
    }
    for my $model (@$models) {
        if ($get_request->is_in_field_names('SourceId')) {
            $model->{source_id} = exists $source_ids->{$model->{cid}} ? $source_ids->{$model->{cid}} : $self->nil;
        }
        if ($get_request->is_in_field_names('RepresentedBy')) {
            $model->{agency_name} = $self->nil unless $model->{agency_name};
            $model->{manager_fio} = $self->nil unless $model->{manager_fio};
        }
        if ($get_request->is_in_field_names('TimeZone')) {
            $model->{time_zone} = TimeTarget::cached_tz_by_id($model->{timezone_id});
        }
        if ($get_request->is_in_field_names('Statistics')) {
            $model->{shows} = $model->{shows} ? $model->{shows} : 0;
            $model->{clicks} = $model->{clicks} ? $model->{clicks} : 0;
        }

        my $strategy = Campaign::define_strategy($model);
        $model->{is_network_default_with_search_autobudget_strategy} =
            is_network_default_strategy($strategy) && is_autobudget($strategy) ? 1 : 0;

        if ($get_request->is_in_field_names_for_all_types('BiddingStrategy')) {
            if ($strategy->{net}{name} eq 'default' && !$model->{is_network_default_with_search_autobudget_strategy}) {
                $strategy->{net}{ContextLimit} = $model->{ContextLimit};
            }
            $model->{strategy} = $strategy;
        } else {
            $model->{strategy} = {};
        }
        if ($get_request->is_in_field_names('Status')) {
            $model->{status} = status_from_model($model);
        }
        if ($get_request->is_in_field_names('State')) {
            $model->{state} = state_from_model($model, $work_currency);
        }
        if ($get_request->is_in_field_names('StatusPayment')) {
            $model->{status_payment} = status_payment_from_model($model);
        }
        if ($get_request->is_in_field_names('Funds')) {
            $model->{finance} = convert_finance_to_external($model, undef, $spend_with_nds_property->get(300));
        }
        if ($get_request->is_in_field_names_for_all_types('Settings')) {
            $model->{is_favorite} = $favorite_camps->{$model->{cid}} ? 1 : 0;
            $model->{is_servicing} = $servicing_camps->{$model->{cid}};
            $model->{can_use_day_budget} = 1;
            $model->{settings} = $self->_convert_settings_to_external($model);
        }
        if ($get_request->is_in_field_names('StatusClarification')) {
            $model->{status_clarification} = $status_clarifications->{$model->{cid}}->{text};
        }
        if ($get_request->is_in_field_names_for_all_types('PlacementTypes')) {
            $model->{placement_types} = convert_placement_types_to_external($model->{placement_types});
        }
        if ($get_request->is_in_field_names_for_all_types('CounterIds')) {
            $model->{metrika_counters} = $metrika_counters->{$model->{cid}};
        }
        if ($get_request->is_in_field_names_for_all_types('CounterId')) {
            my $counters = $metrika_counters->{$model->{cid}};
            $model->{metrika_counter} = ref $counters eq 'ARRAY' ? $counters->[0] : $counters;
        }

        if ($get_request->is_in_field_names_for_all_types('RelevantKeywords')) {
            my %broad_match = (
                flag => delete $model->{broad_match_flag},
                limit => delete $model->{broad_match_limit},
                goal_id => delete $model->{broad_match_goal_id},
            );
            $model->{broad_match} = \%broad_match;
        }
        if ($get_request->is_in_field_names_for_all_types('FrequencyCap') && $model->{type} eq 'cpm_banner') {
            my %frequency_cap = (
                rf => delete $model->{rf},
                rfReset => delete $model->{rfReset},
            );
            $model->{frequency_cap} = \%frequency_cap;
        }
    }

    return $models;
}

sub _convert_settings_to_external {
    my ($self, $model) = @_;
    return convert_settings_to_external($model);
}

sub _prepare_nillable_fields {
    my ($self, $campaign) = @_;
    my @nillable_fields = (qw/StartDate EndDate BlockedIps ExcludedSites CounterIds VideoTarget
                             NegativeKeywords RelevantKeywords FrequencyCap PriorityGoals/,
                             [RelevantKeywords => 'OptimizeGoalId'],
                             [TimeTargeting => 'HolidaysSchedule'],
                             [FrequencyCap => 'PeriodDays'],
                         );
    for my $field (@nillable_fields) {
        my $structure = $campaign;
        if (ref $field) {
            next unless exists $structure->{$field->[0]} && defined $structure->{$field->[0]} && !$self->is_nil($structure->{$field->[0]});
            $structure = $structure->{$field->[0]};
            $field = $field->[1];
        }
        $structure->{$field} = $self->nil
            if exists $structure->{$field} and !defined $structure->{$field};
    }
    if (exists $campaign->{BiddingStrategy}) {
        my $strategies = $campaign->{BiddingStrategy};
        for my $type (qw/Search Network/) {
            my $strategy = $strategies->{$type};
            my $settings_structure_name = get_structure_name_by_strategy($strategy->{BiddingStrategyType});
            my $settings_structure = $strategy->{$settings_structure_name};
            for my $setting (keys %$settings_structure) {
                if (!defined $settings_structure->{$setting}) {
                    $settings_structure->{$setting} = $self->nil;
                }
            }
        }
    }
    return;
}

sub _get_campaigns($$) {
    my ($self, $get_request, $work_currency) = @_;

    # выбираем сконвертированные кампании только если явно попросили - через выборку по соответствующему стейту или по айди
    my $filter_out_converted_camps = !$get_request->has_selection_ids() &&
                                        (none { $_ eq 'CONVERTED' } $get_request->selection_attribute('States'));

    if (!$get_request->has_selection_ids()) {
        my $ids = $self->grep_read_access_campaign_ids( get_cids(uid => $self->subclient_uid) );
        $get_request->set_selection_attribute(Ids => $ids);
    } else {
        $get_request->set_selection_attribute(
            Ids => $self->grep_read_access_campaign_ids(
                $self->grep_owned_campaign_ids([ uniq $get_request->selection_ids('Ids') ])
            )
        );
    }
    if (!$get_request->has_selection_attribute('Types')) {
        $get_request->set_selection_attribute('Types', convert_types_to_external($self->get_api_allowed_camp_types()));
    }

    my $where = _selection_criteria_to_where($self, $get_request, $work_currency, $filter_out_converted_camps);

    my $limit = $get_request->page_limit + 1;
    my $offset = $get_request->page_offset;

    my $campaigns = $self->_get_campaigns_from_db($where, $limit, $offset);

    mix_manager_data($campaigns);
    mix_agency_data($campaigns);

    my $client_discount = get_client_discount($self->subclient_client_id);
    enrich_sums_uni($campaigns, $client_discount, $work_currency);

    return $campaigns;
}

sub _get_campaigns_from_db {
    my ($self, $where, $limit, $offset) = @_;

    my $campaigns = get_user_camps_by_sql($where,
        {order_by => ['c.cid'], limit => $limit, offset => $offset,
         shard => {ClientID => $self->subclient_client_id},
         remove_nds => 1,
         without_spent_today => 1,
         sums_with_include_nds => $spend_with_nds_property->get(300) ? 1 : 0,
         dont_calc_status => 1,
         dont_calc_has_active_banners => 1,
         join => 'LEFT JOIN strategies s ON c.strategy_id = s.strategy_id',
         strategy_fields => ', s.is_public as strategy_is_public',
        }
    )->{campaigns};

    # DIRECT-51986 - get_user_camps_by_sql() convert
    # start_time == "0000-00-00" to start_date == undef
    # as web interface needed, but for api start_date
    # set to "0000-00-00"
    for ( @$campaigns ) {
        $_->{start_date} //= '0000-00-00';
    }

    return $campaigns;
}

=head2 enrich_sums_uni

    Добавляет в структуру sums_uni хеша с кампанией вычисляемые поля для FundsParam
    Принимает: хеш с кампанией и скидку клиента

=cut

sub enrich_sums_uni {
    my ($models, $client_discount, $work_currency) = @_;

    my @campaign_ids = map {$_->{wallet_cid} || (defined $_->{currency} && $_->{currency} eq 'YND_FIXED') ? () : ( $_->{cid} ) } @$models;
    my $forecasts = {};
    if ((scalar @campaign_ids) > 0) {
        $forecasts = Stat::OrderStatDay::get_camp_bsstat_forecast(\@campaign_ids , $work_currency);
    }

    my $blocked_money_camp = mass_check_block_money_camps($models);
    for my $model (@$models) {
        $model->{money_type} = $blocked_money_camp->{$model->{cid}} ? 'blocked' : 'real';
        my $money = $model->{sums_uni};
        if ($client_discount) {
            if (!$model->{wallet_is_enabled}) {
                $money->{bonus} = calc_bonus($money->{"total"}, $client_discount);
            }
        } else {
            $money->{bonus} = 0;
        }

        $money->{total_include_nds} = 0;
        $money->{available_for_transfer} = get_camp_sum_available(
            $model, fast => 1, sums_without_nds => 1, dont_round => 1, forecast => ($forecasts->{ $model->{cid} } // 0),
        );
    }
    return;
}

sub _selection_criteria_to_where {
    my ($self, $get_request, $work_currency, $filter_out_converted_camps) = @_;

    my ($statuses_where, $states_where, $statuses_payment_where) = ({}, {}, {});

    my @statuses = $get_request->selection_attribute('Statuses');
    if (@statuses) {
        $statuses_where = statuses_where(\@statuses);
    }

    my @states = $get_request->selection_attribute('States');
    if (@states) {
        $states_where = states_where(\@states, $work_currency);
    }

    my @statuses_payment = $get_request->selection_attribute('StatusesPayment');
    if (@statuses_payment) {
        $statuses_payment_where = status_payment_where(\@statuses_payment);
    }

    my $where_for_all_statuses = merge_statuses_where($statuses_where, $states_where, $statuses_payment_where);

    my $converter = $self->converter('internal');

    my $selection_criteria = dclone $get_request->selection_criteria;

    my $where = $converter->convert({SelectionCriteria => $selection_criteria})->{SelectionCriteria};

    if ($filter_out_converted_camps) {
        $where->{_NOT} = states_where([qw/CONVERTED/], $work_currency);
    }

    hash_merge($where, $where_for_all_statuses);

    $where->{'c.source__not_in'} = ['uac', 'widget']; # фильтруем кампании, созданные в Мастере кампаний

    return $where;
}

=head2 add

    создание кампаний

=cut
sub add {
    my ($self, $request) = @_;
    my $rs = API::Service::Campaigns::AddResults->new(@{$request->{Campaigns}});

    my $e = $self->_validate_add_request($rs);
    return $e if $e;

    # проверим есть ли у клиента сервисируемые кампании,
    # если есть, то запомним менеджера
    $self->_guess_and_remember_manager();

    my $client_currencies = get_client_currencies($self->subclient_client_id);
    my $work_currency = $client_currencies->{work_currency};

    $self->_validate_add_request_items($rs, $work_currency);

    $self->_enable_relevant_keywords_for_text_camps_by_default($rs);

    my @result;
    my $converter = $self->converter('internal');

    my $user_data = get_user_data($self->subclient_uid, [qw/fio email/]);

    my $wallet_day_budget = $self->_get_wallet_day_budget($work_currency);

    my $meaningful_goals_data = $self->_prefetch_meaningful_goals_data_on_add($rs);
    my $prefetched_goals = $self->_prefetch_goals($rs, 'is_add_request');
    my $source = $API::Settings::APP_ID_TO_CAMPAIGN_SOURCE->{$self->application_id};
    $source = 'api' if not defined $source;

    for my $item ($rs->list) {
        $item->object->{Type} = get_type_by_structure($item->object);

        disclose_type_structure($item->object);

        $self->_preprocess_nillable_fields($item);

        my $converted = $converter->convert($item->object);
        $converted->{source} = $source;

        _preprocess_add_update($converted);
        _mix_user_data($user_data, $converted);
        _mix_currency($work_currency, $converted);

        $self->_add_prepare_model_default_values($item, $converted);

        $self->_validate_campaing_type_with_currency($item, $converted);

        $self->_preprocess_network_default_strategy_params($item, $converted);

        $self->_validate_add_campaign($item, $converted, $wallet_day_budget, $meaningful_goals_data, $prefetched_goals)
            unless $item->has_errors;

        my %result;
        if ($item->has_errors) {
            $result{Errors} = action_errors($item->list_errors)
        } else {
            if (my $create_error = $self->_create_in_rbac($converted)) {
                $result{Errors} = action_errors($create_error);
            } else {
                try {
                    $self->_save($converted, 1);
                    $result{Id} = $converted->{cid};
                } catch {
                    $result{Errors} = action_errors(error_OperationFailed());
                };
            }
        }

        $result{Warnings} = action_errors($item->list_warnings)
            if $item->has_warnings;

        push @result, \%result;
    }

    $self->units_withdraw_for_results( campaign => $rs );

    return {
        AddResults => \@result
    };
}

sub _add_prepare_model_default_values {
    my ($self, $item, $campaign) = @_;

    if ($campaign->{type} && $campaign->{type} eq 'cpm_banner') {
        $campaign->{rf} //= 0;
        $campaign->{rfReset} //= 0;
        $campaign->{mediaType} = $campaign->{type};

        my $strategy_type = $campaign->{strategy}{net}{name};
        my $video_type = $campaign->{eshows_video_type};
        if ($strategy_type eq "autobudget_avg_cpv" || $strategy_type eq "autobudget_avg_cpv_custom_period") {
            $campaign->{eshows_video_type} = undef;
            if (defined $video_type) {
                $item->add_warning(
                    warning_ParamNotUseful(
                        iget('Параметр %s для заданной стратегии не поддерживается', 'VideoTarget')
                    )
                );
            }
        } elsif (!defined $video_type) {
            $campaign->{eshows_video_type} = "completes";
        }
    } else {
        $campaign->{opts}{enable_cpc_hold} //= 1;
    }
    $campaign->{attribution_model} //= get_attribution_model_or_default_by_type($campaign);

    if (!exists $campaign->{status_click_track}
        && $campaign->{type}
        && ( any { $campaign->{type} eq $_ } qw/text dynamic cpm_banner/ )
        && $add_metrica_tag_property->get(300)) {
        $campaign->{status_click_track} = 1; # по умолчанию включено добавление yclid в ссылку
    }

    for my $type (qw/search net/) {
        if ($campaign->{strategy}->{$type}) {
            my $strategy = $campaign->{strategy}->{$type};
            if ($strategy->{name} eq 'autobudget_avg_cpi') {
                $strategy->{goal_id} //= $Settings::DEFAULT_CPI_GOAL_ID;
            }
        }
    }

    return;
}


sub _prefetch_meaningful_goals_data_on_add {
    my ($self, $rs) = @_;

    my @counters;
    for my $item ($rs->list_ok) {
        my $tc = $item->object->{TextCampaign} || $item->object->{DynamicTextCampaign} || $item->object->{SmartCampaign};
        next if !$tc;

        # если нет ключевых целей, то ничего запрашивать не нужно
        next if !$tc->{PriorityGoals};
        next if none {$_->{GoalId} != $Settings::ENGAGED_SESSION_GOAL_ID} @{$tc->{PriorityGoals}->{Items}};

        if ($tc->{CounterIds}) {
            push @counters, @{$tc->{CounterIds}->{Items}};
        }
        if ($tc->{CounterId}) {
            push @counters, $tc->{CounterId};
        }
    }

    return {} if !@counters;
    return Campaign::prefetch_meaningful_goals_data(counters => \@counters);
}


sub _prefetch_meaningful_goals_data_on_update {
    my ($self, $rs) = @_;

    my @counters;
    my @cids;
    for my $item ($rs->list_ok) {
        my $tc = $item->object;
        next if !$tc->{PriorityGoals};
        next if !grep {$_->{GoalId} != $Settings::ENGAGED_SESSION_GOAL_ID} @{$tc->{PriorityGoals}->{Items}};

        push @cids, $item->object->{Id};
        if ($tc->{CounterIds}) {
            push @counters, @{$tc->{CounterIds}->{Items}};
        }
    }

    return {} if !@cids;
    return Campaign::prefetch_meaningful_goals_data(
        counters => \@counters,
        cids => \@cids,
    );
}


sub _validate_meaningful_goals {
    my ($self, $item) = @_;

    my $mg_setting = $item->object->{TextCampaign} && $item->object->{TextCampaign}->{PriorityGoals}
                     || $item->object->{DynamicTextCampaign} && $item->object->{DynamicTextCampaign}->{PriorityGoals}
                     || $item->object->{SmartCampaign} && $item->object->{SmartCampaign}->{PriorityGoals};
    return if !$mg_setting;

    for my $mg_item (@{$mg_setting->{Items}}) {
        if ($mg_item->{Operation} ne 'SET') {
            $item->add_error(
                error_NotSupported(iget('Для поля %s поддерживается только операция %s', 'PriorityGoals', 'SET'))
            );
            last;
        }
    }

    return;
}

sub _prefetch_goals {
    my ($self, $rs, $is_add_request) = @_;

    my @counters;
    my $need_fetch_goals = 0;

    for my $item ($rs->list_ok) {
        my $camp;
        if ($is_add_request) {
            my $type = get_type_structure_name($item->object);
            $camp = $item->object->{$type};
        } else {
            $camp = $item->object;
        }

        my $strategies = $camp->{BiddingStrategy};
        if ($strategies) {
            for my $strategy_type (qw/Search Network/) {
                my $strategy = $camp->{BiddingStrategy}->{$strategy_type};
                if (defined $strategy) {
                    my $structure_name = get_structure_name_by_strategy($strategy->{BiddingStrategyType});
                    if (defined $strategy->{$structure_name} && $strategy->{$structure_name}->{GoalId}) {
                        $need_fetch_goals = 1;
                        if ($camp->{CounterIds}) {
                            push @counters, @{$camp->{CounterIds}->{Items}};
                        }
                        if ($camp->{CounterId}) {
                            push @counters, $camp->{CounterId};
                        }
                        last;
                    }
                }
            }
        }
    }

    if ($need_fetch_goals) {
        return Campaign::prefetch_goals([uniq @counters], $self->subclient_uid);
    }

    return undef;
}

sub _validate_frequency_cap {
    my ($self, $item) = @_;

    return unless exists $item->object->{CpmBannerCampaign} && exists $item->object->{CpmBannerCampaign}{FrequencyCap};

    my $frequency_cap = $item->object->{CpmBannerCampaign}{FrequencyCap};
    return if $self->is_nil($frequency_cap);

    if ($frequency_cap->{Impressions} < $Direct::Validation::Campaigns::Constraints::CPM_RF_MIN
            || $frequency_cap->{Impressions} > $Direct::Validation::Campaigns::Constraints::CPM_RF_MAX) {
        $item->add_error(
            error_InvalidField(
                iget(
                    'Значение поля %s должно быть в диапазоне от %s до %s', 'Impressions',
                    $Direct::Validation::Campaigns::Constraints::CPM_RF_MIN,
                    $Direct::Validation::Campaigns::Constraints::CPM_RF_MAX
                )
            )
        );
    }

    return if $self->is_nil($frequency_cap->{PeriodDays});

    if ($frequency_cap->{PeriodDays} < $Direct::Validation::Campaigns::Constraints::CPM_RF_RESET_MIN
            || $frequency_cap->{PeriodDays} > $Direct::Validation::Campaigns::Constraints::CPM_RF_RESET_MAX) {
        $item->add_error(
            error_InvalidField(
                iget(
                    'Значение поля %s должно быть в диапазоне от %s до %s', 'PeriodDays',
                    $Direct::Validation::Campaigns::Constraints::CPM_RF_RESET_MIN,
                    $Direct::Validation::Campaigns::Constraints::CPM_RF_RESET_MAX
                )
            )
        );
    }

    return;
}

sub _validate_campaign_type {
    my ($self, $item, $is_required) = @_;

    my $count = 0;

    state $types = get_type_structures_names(convert_types_to_external($self->get_api_allowed_camp_types()));
    state $types_str = join(', ', @$types);

    foreach my $type (@$types) {
        $count++ if exists $item->object->{$type};
        if ($count > 1) {
            $item->add_error( error_PossibleOnlyOneField( iget('Кампания может содержать только один из объектов %s', $types_str)) );
            return 1;
        }

    }
    if(!$count && $is_required) {
        $item->add_error( error_RequiredAtLeastOneOfFields( iget('Кампания должна содержать один из объектов %s', $types_str) ) );
        return 1;
    }

    return; # ok
}

sub _validate_campaing_type_with_currency {
    my ($self, $item, $campaign_data) = @_;

    if ($campaign_data->{currency} eq 'YND_FIXED' && $campaign_data->{type} eq 'cpm_banner') {
        $item->add_error( error_NotSupported( iget('Медийные кампании в валюте %s не поддерживаются', 'YND_FIXED') ) );
    }

    return;
}

=head2 _guess_and_remember_manager

    Пытаемся определить сервисирующего менеджера, если получилось - то запоминаем

=cut

sub _guess_and_remember_manager {
    my ($self) = @_;

    my $manager_uid = get_other_manager_uid($self->subclient_uid, get_camp_kind_types('web_edit_base'));
    if (!$manager_uid) {
        # Пробуем найти главного менеджера
        $manager_uid = get_idm_primary_manager_uid($self->subclient_client_id);
    }

    if ($manager_uid) {
        $self->stash->{manager_uid} = $manager_uid;
    }

    return;
}

=head2 _enable_relevant_keywords_for_text_camps_by_default

    DIRECT-68257 - включаем ДРФ по умолчанию

=cut

sub _enable_relevant_keywords_for_text_camps_by_default {
    my ($self, $rs) = @_;

    for my $item ( $rs->list ) {
        if (! exists $item->object->{TextCampaign}) {
            next;
        }

        my $text_campaign = $item->object->{TextCampaign};
        if (exists $text_campaign->{RelevantKeywords}) {
            next;
        }

        if (   exists $text_campaign->{Settings}
            && any { $_->{Option} eq 'ENABLE_RELATED_KEYWORDS' && $_->{Value} eq 'YES' } @{ $text_campaign->{Settings} }
        ) {
            $text_campaign->{RelevantKeywords} = {
                BudgetPercent  => Campaign::get_broad_match_default(),
                OptimizeGoalId => undef,
            };
        }
    }

    return;
}

=head2 _create_in_rbac($campaign)

    Создаем кампанию в RBAC, в случае успеха новый cid сохранятеся в $campaign и
    возвращается undef иначе объект ошибки Direct::Defect

    $campaign - хэш с данными кампании, c полями fio, email, type

=cut

sub _create_in_rbac {
    my ($self, $campaign) = @_;

    my $campaign_model = Direct::Model::Campaign->new(
        client_fio => $campaign->{fio},
        email => $campaign->{email},
        currency   => $campaign->{currency},
        campaign_type => $campaign->{type}
    );

    if (my $cid = $self->campaign_creator->create($campaign_model)) {
        $campaign->{cid} = $cid;
        return;
    } else {
        # что-то пошло не так, но ошибку маскируем под "нет прав"
        return $self->campaign_creator->error;
    }
}

=head2 _enable_wallet

    Если клиент при помощи метода add создает свою первую кампанию,
    то вместе с ней создается и подключается общий счет DIRECT-47025

=cut

sub _enable_wallet {
    my ($self, $ctx, $campaign) = @_;

    my $agency_uid;
    if ($self->campaign_creator->create_by_role eq 'agency') {
        $agency_uid = $self->campaign_creator->create_by_agency_uid;
    }
    if (Wallet::need_enable_wallet(cid => $campaign->{cid}, client_id => $self->subclient_client_id)) {
        $ctx->client_client_id($self->subclient_client_id);
        $ctx->client_chief_uid($self->subclient_uid);
        my $result = Wallet::enable_wallet($ctx, $campaign->{currency}, $agency_uid
            , dont_check_onoff_time => 1
            , allow_wallet_before_first_camp => 1
            , first_camp_cid => $campaign->{cid});
        if (exists $result->{error}) {
            croak $result->{error};
        }
        return 1;
    }
    return 0;
}

=head2 _get_wallet_day_budget

    Получение настроек ОС

=cut

sub _get_wallet_day_budget {
    my ($self, $currency) = @_;

    my $ctx = $self->get_direct_context();

    my $agency_client_id;
    if ($self->is_operator_agency()) {
        $agency_client_id = $self->operator_client_id;
    } else {
        $agency_client_id = 0; # TODO: если создает менеджер - то чей ОС нужно брать?
    }

    my $camps = Wallet::get_wallets_by_uids(
        [{ c => $ctx, agency_client_id => $agency_client_id, client_currency => $currency }],
    );

    my $wallet_day_budget;
    if (@$camps) {
        my $wallet = $camps->[0]{wallet_camp};
        if ($wallet && $wallet->{enabled}) {
            $wallet_day_budget = {
                day_budget => $wallet->{day_budget}{sum},
                day_budget_show_mode => $wallet->{day_budget}{show_mode},
            };
        }
    }

    return $wallet_day_budget;
}

sub _save {
    my ($self, $model, $is_new_camp, %opts) = @_;

    my $ctx = $self->get_direct_context();

    my $package_strategy_is_changed = $opts{package_strategy_is_changed};
    my $need_to_create_new_package = $opts{need_to_create_new_package};

    my $old_campaign = save_camp($ctx, $model, $self->subclient_uid,
        is_new_camp => $is_new_camp,
        dont_sort_disabled_ips => 1,
        ignore_hierarchical_multipliers => 1,
        package_strategy_is_changed => $package_strategy_is_changed,
        need_to_create_new_package => $need_to_create_new_package
    );

    my $camp_in_base = hash_cut($old_campaign, qw/cid currency mediaType ClientID/);
    $camp_in_base->{strategy} = Campaign::define_strategy($old_campaign);
    $camp_in_base->{timeTarget} = TimeTarget::pack_timetarget($model);

    my $is_strategy_set = Campaign::camp_set_strategy($camp_in_base, $model->{strategy},
        {
            uid => $self->subclient_uid,
            send_notifications => 1,
            is_attribution_model_changed => get_attribution_model_or_default_by_type($model) ne $old_campaign->{attribution_model},
            need_to_create_new_package => $need_to_create_new_package,
            package_strategy_is_changed => $package_strategy_is_changed
        });

    if ($is_new_camp && !$is_strategy_set) {
            Campaign::mark_strategy_change($camp_in_base->{cid},
                $self->subclient_uid,
                $model->{strategy},
                $camp_in_base->{strategy});
    }

    # Отправляем на сервисируемость если выполняются все условия:
    # - уже есть сервисируемые кампании (удалось определить uid менеджера)
    # - у оператора есть право для отправки на сервисируемость (NB: под менеджером автоматически создается сервисируемая кампания)
    # - явно просят отправить на сервисируемость или создается новая кампания без явного указания настройки "Отправить на сервисируемость"
    my $manager_uid = $self->stash->{manager_uid};
    if ( $manager_uid
         && rbac_can_create_scamp_by_role( undef, $self->operator_role )
         && ( ($is_new_camp && ! exists $model->{is_servicing}) || $model->{is_servicing} )
    ) {
        send_camp_to_service(undef, $model->{cid}, $self->subclient_uid, $manager_uid);
    }

    camp_save_metrika_counters($model->{cid}, $model->{strategy_id}, $model->{metrika_counters},
        package_strategy_is_changed => $package_strategy_is_changed,
        need_to_create_new_package => $need_to_create_new_package);

    if ( $model->{is_favorite} ) {
        favorite_camp($self->subclient_uid, $model->{cid}, 1);
    } else {
        favorite_camp($self->subclient_uid, $model->{cid}, 0);
    }

    update_camp_auto_optimization(
        $model->{cid}, $self->operator_chief_rep_uid, $model->{autoOptimization}, dont_send_notification => 1
    );

    if ($is_new_camp) {
        _enable_wallet($self, $ctx, $model);
        BillingAggregateTools::on_save_new_camp($self->subclient_client_id, $model->{cid}, $self->operator_uid);
    }

    set_status_empty_to_no($model->{cid});

    if ($save_available_counters_goals_on_campaign_saving_property->get(300) &&
        $model->{metrika_counters}) {
        my @counter_ids = split(' ', $model->{metrika_counters});
        my @old_counter_ids = split(',', $old_campaign->{metrika_counters});

        my %old_counter_ids_hash = map { $_ => 1 } @old_counter_ids;

        if (any { !$old_counter_ids_hash{$_} } @counter_ids) {
            my $counters_goals = MetrikaCounters::get_client_counters_goals(
                $self->subclient_client_id, \@counter_ids,
                no_cache => 1, get_steps => 1) // [];

            my @goals = grep { $_ } map { @$_ } grep { $_ } values %$counters_goals;

            MetrikaCounters::add_or_update_goals_for_cid($model->{cid}, \@goals);
        }
    }

    return;
}

sub _mix_user_data {
    my ($user_data, $model) = @_;
    $model->{fio} //= $user_data->{fio};
    $model->{email} //= $user_data->{email};
    return;
}

sub _mix_currency {
    my ($work_currency, $model) = @_;
    $model->{currency} = $work_currency;
    return;
}

sub _validate_add_request_items {
    my ($self, $rs, $currency) = @_;

    for my $item ($rs->list) {
        next if $self->_validate_campaign_type($item, 1);
        $self->_validate_strategy_consistent($item);
        $self->_validate_context_strategy($item);
        $self->_validate_empty_array_items($item);
        $self->_validate_duplicate_array_items($item);
        $self->_validate_time_targeting($item);
        $self->_validate_excluded_domains_for_camp_types($item);
        $self->_validate_negative_keywords($item);
        $self->_validate_WeeklyPacketOfClicks_params($item);
        $self->_validate_daily_budged_fill_default($item, $currency);
        $self->_validate_require_servicing($item);
        $self->_validate_campaign_settings($item);
        $self->_validate_frequency_cap($item);

        if (defined $item->object->{TimeZone}) {
            $item->add_error(error_BadParams(iget('Неверно указана временная зона')))
                unless DateTime::TimeZone->is_valid_name($item->object->{TimeZone});
        }
    }

    return;
}

sub _validate_require_servicing {
    my ($self, $item) = @_;

    my $type = get_type_structure_name($item->object);
    if ($type && $item->object->{$type}{Settings}) {
        my $settings = $item->object->{$type}{Settings};
        if (any {$_->{Option} eq 'REQUIRE_SERVICING' and $_->{Value} eq 'YES'} @$settings) {
            # если нет ни одной сервисируемой кампании или у оператора нет прав для отправки на сервисируемость
            if ( ! ( $self->stash->{manager_uid} && rbac_can_create_scamp_by_role( undef, $self->operator_role ) ) ) {
                $item->add_warning(warning_SettingNotChanged_RequireServicing());
            }
        }
    }

    return; # ok
}

# не заданный дневной бюджет == 0.0
# в API сброс значения поля в значение по умолчанию - передача nil
# поэтому запрещаем передавать 0 как не валидное значение,
# но меняем сами nil на 0.0 для того, чтобы "убрать" бюджет
sub _validate_daily_budged_fill_default {
    my ($self, $item, $currency) = @_;

    return unless exists $item->object->{DailyBudget};

    if ($self->is_nil($item->object->{DailyBudget})) {
        $item->object->{DailyBudget} = {
            Amount => 0,
            Mode => 'STANDARD'
        };
    } elsif ($item->object->{DailyBudget}) {
        if ( $item->object->{DailyBudget}{Amount} == 0 ) {
             $item->add_error( error_InvalidField_NotPositive(undef, field => 'DailyBudget Amount') );
        } elsif ( convert_to_money($item->object->{DailyBudget}{Amount}) == 0 ) {
            # в ядровую валидацию попадет уже поделенное на 1_000_000 и округленное до 2х знаков после запятой значение
            # т.о. значение 1000 становится 0, но это не корректно, поэтому нужна валидация до преобразования из микрофишек в деньги.
            $item->add_error( error_BadParams(iget('Минимальная сумма дневного бюджета %s', format_const($currency, 'MIN_DAY_BUDGET'))) );
        }
    }

    return; # ok
}

sub _daily_budget_default_to_nil {
    my ($self, $campaign) = @_;
    if ($campaign->{DailyBudget} && $campaign->{DailyBudget}{Amount} == 0 and $campaign->{DailyBudget}{Mode} = 'DEFAULT') {
        $campaign->{DailyBudget} = $self->nil;
    }
}

sub _validate_WeeklyPacketOfClicks_params {
    my ($self, $item) = @_;
    my $type = get_type_structure_name($item->object);
    if ($type && $item->object->{$type}{BiddingStrategy}) {
        my $strategies = $item->object->{$type}{BiddingStrategy};
        for my $strategy_type (qw/Search Network/) {
            next unless $strategies->{$strategy_type};
            my $strategy = $strategies->{$strategy_type};
            if ( $strategy->{BiddingStrategyType} eq "WEEKLY_CLICK_PACKAGE" ) {
                my $structure_name = get_structure_name_by_strategy($strategy->{BiddingStrategyType});
                my $params = $strategy->{$structure_name};

                $item->add_error(
                    error_PossibleOnlyOneParameter(undef, fields => join(', ', qw/AverageCpc BidCeiling/))
                ) if defined $params && $params->{AverageCpc} && !$self->is_nil($params->{AverageCpc})
                                     && $params->{BidCeiling} && !$self->is_nil($params->{BidCeiling})
            }
        }
    }
    return;
}

sub _validate_WbMaximumConversionRate_for_update {
    my ($self, $item) = @_;
    my $type = get_type_structure_name($item->object);
    if ($type && $item->object->{$type}{BiddingStrategy}) {
        my $strategies = $item->object->{$type}{BiddingStrategy};
        for my $strategy_type (qw/Search Network/) {
            next unless $strategies->{$strategy_type};
            my $strategy = $item->object->{$type}{BiddingStrategy}{$strategy_type};
            if ( $strategy->{BiddingStrategyType} eq "WB_MAXIMUM_CONVERSION_RATE" ) {
                my $structure_name = get_structure_name_by_strategy($strategy->{BiddingStrategyType});
                $item->add_error( error_BadParams(iget('Цель не задана')) )
                    if defined $strategy->{$structure_name} && !defined $strategy->{$structure_name}{GoalId}
            }
        }
    }
    return;
}

sub _validate_empty_array_items {
    my ($self, $item) = @_;
    for my $field (qw/ExcludedSites BlockedIps/){
        if (exists $item->object->{$field}) {
            for my $value (@{$item->object->{$field}{Items}}) {
                $item->add_error(
                    error_InvalidField_EmptyArrayItem(undef, field => $field, value => $value)
                ) unless $value && $value =~ /\S/;
            }
        }
    }
    return;
}

sub _validate_excluded_domains_for_camp_types {
    my ($self, $item, $type) = @_;
    my $camp_type_internal = $type // convert_type(get_type_by_structure($item->object));
    my $field_to_validate = 'ExcludedSites';
    if ($TYPES_WITH_NO_EXCLUDED_DOMAIN_ITEMS->{$camp_type_internal} && exists $item->object->{$field_to_validate} ) {
        $item->add_error(error_BadParams(iget('Поле %s не поддерживается в кампании заданного типа', $field_to_validate)));
    }
    return;
}

sub _validate_campaign_settings {
    my ($self, $item) = @_;

    # DIRECT-63109 - ENABLE_RELATED_KEYWORDS
    # DIRECT-68434 - ENABLE_AUTOFOCUS
    # DIRECT-69528 - ENABLE_BEHAVIORAL_TARGETING
    my %deprecated_settings_by_camp_type = (
        TextCampaign        => [qw/ ENABLE_AUTOFOCUS ENABLE_RELATED_KEYWORDS ENABLE_BEHAVIORAL_TARGETING /],
        MobileAppCampaign   => [qw/ ENABLE_AUTOFOCUS ENABLE_BEHAVIORAL_TARGETING /],
        DynamicTextCampaign => [qw/ ENABLE_BEHAVIORAL_TARGETING /],
    );

    my $camp_type = get_type_structure_name($item->object);
    if ($camp_type && exists $deprecated_settings_by_camp_type{ $camp_type } && $item->object->{ $camp_type }{Settings}) {
        my $camp_settings = $item->object->{ $camp_type }{Settings};
        my $deprecated_settings = $deprecated_settings_by_camp_type{ $camp_type };
        for my $setting_name ( @$deprecated_settings ) {
            if ( any { $_->{Option} eq $setting_name } @$camp_settings ) {
                $item->add_warning(warning_SettingNotChanged(iget('Настройка %s больше не поддерживается', $setting_name)));
            }
        }

        if (any { $_->{Option} eq 'MAINTAIN_NETWORK_CPC' } @$camp_settings) {
            $item->mark($HAS_MAINTAIN_NETWORK_CPC_SETTING_FLAG_NAME);
        }
    }
}

sub _validate_add_request {
    my ($self, $camp_object) = @_;

    # Возвращаем CantWrite ошибки, т.к. в случае отсутствия прав на чтение
    # оператором данных сабклиента мы бы отвалились еще в хэндлере
    if(my $error = $self->campaign_creator->why_cant_create_campaign) {
        return $error;
    }

    if ($camp_object->count_ok > $ADD_IDS_LIMIT) {
        return error_RequestLimitExceeded(
            iget('Разрешено создавать не более %s кампаний в одном запросе', $ADD_IDS_LIMIT)
        );
    }
    my $e = check_add_client_campaigns_limits(
                ClientID => $self->subclient_client_id,
                unarchived_count => $camp_object->count_ok
    );
    return error_ReachLimit($e) if $e;
}

sub _validate_add_campaign {
    my ($self, $item, $model, $wallet_day_budget, $meaningful_goals_data, $prefetched_goals) = @_;

    $self->_prevalidate_placement_types($item, $model);
    $self->_prevalidate_broad_match_on_add($item, $model);

    my $state = $self->_build_state_for_model( $model, new => 1 );

    my $model_obj;

    if (!$self->get_business_logic_class_by_type($model->{type})) {
        die "Not error_NotSupported_CampaignType for type $model->{type}";
    } elsif ( $model->{type} eq 'text' ) {
        $model_obj = Direct::Model::CampaignText->new( $state );
    } elsif ( $model->{type} eq 'mobile_content' ) {
        $model_obj = Direct::Model::CampaignMobileContent->new( $state );
    } elsif ( $model->{type} eq 'dynamic' ) {
        $model_obj = Direct::Model::CampaignDynamic->new( $state );
    } elsif ( $model->{type} eq 'performance' ) {
        $model_obj = Direct::Model::CampaignPerformance->new( $state );
    } elsif ( $model->{type} eq 'cpm_banner' ) {
        $model_obj = Direct::Model::CampaignCpmBanner->new( $state );
    } elsif ( $model->{type} eq 'content_promotion' ) {
        $model_obj = Direct::Model::CampaignContentPromotion->new( $state );
    } else {
        die "Unexpected camp type in validation $model->{type}";
    }

    my $is_allowed_to_use_value_from_metrika = Campaign::is_allowed_to_use_value_from_metrika($model->{strategy});

    my $vres = validate_add_campaign( $model_obj,
        translocal_tree => 'api',
        prefetched_meaningful_goals_data => $meaningful_goals_data,
        uid => $self->subclient_uid,
        unavailable_auto_goals_allowed => $API::Settings::UNAVAILABLE_AUTO_GOALS_ALLOWED_APP_ID->{$self->application_id},
        is_allowed_to_use_value_from_metrika => $is_allowed_to_use_value_from_metrika
    );

    $vres->process_descriptions(%DEFECT_DESCRIPTION_FIELD_MAP);

    $item->add_error( $_ ) foreach @{ $vres->get_errors };
    $item->add_warning( $_ ) foreach @{ $vres->get_warnings };

    my $model_data = {cid => $model->{cid}, type => $model->{type}, currency => $model->{currency}};
    $model_data->{start_date} = $model->{start_date};
    $model_data->{finish_date} = $model->{finish_date};

    $model_data->{metrika_counters} = $model->{metrika_counters};

    my $turbo_apps_allowed = Client::ClientFeatures::has_turbo_app_allowed($self->subclient_client_id);
    if ($model->{opts}{has_turbo_app} && !$turbo_apps_allowed) {
        $item->add_error(error_LimitAccess(iget('Продвижение турбо-аппов недоступно')));
    }

    if ($model->{type} eq 'performance' && defined $model_data->{metrika_counters}) {
        my @errors = Models::Campaign::validate_campaign_metrika_counters($model_data->{metrika_counters},
                $model->{type}, $self->subclient_uid, $self->operator_uid());
            $item->add_error(error_BadParams( iget($errors[0]))) if @errors;
    }

    my $cpa_pay_for_conversion_extended_mode_allowed = Client::ClientFeatures::has_cpa_pay_for_conversions_extended_mode_allowed($self->subclient_client_id);
    if ($model->{strategy}->{search}->{name} eq 'no_premium') {
        $item->add_error(
                error_NotSupported(iget('Стратегия IMPRESSIONS_BELOW_SEARCH не поддерживается'))
        );
    } elsif(_is_strategy_pay_for_conversion_prohibited($model, $cpa_pay_for_conversion_extended_mode_allowed)) {
        $item->add_error(error_LimitAccess_StrategyPayForConversion());
    } else {
        my $strategy_error = Campaign::validate_camp_strategy($model_data, $model->{strategy},
            {
                is_api => 1,
                new_camp => 1,
                client_id => $self->subclient_client_id,
                has_cpa_pay_for_conversions_extended_mode_allowed => $cpa_pay_for_conversion_extended_mode_allowed,
                has_cpa_pay_for_conversions_mobile_apps_allowed => Client::ClientFeatures::has_cpa_pay_for_conversions_mobile_apps_allowed($self->subclient_client_id),
                request_meaningful_goals => $model->{meaningful_goals},
                defect_description_field_map => \%STRATEGY_DEFECT_DESCRIPTION_FIELD_MAP,
                prefetched_goals => $prefetched_goals,
                has_edit_avg_cpm_without_restart_enabled => Client::ClientFeatures::has_edit_avg_cpm_without_restart_feature($self->subclient_client_id),
                has_disable_all_goals_optimization_for_dna_enabled => Client::ClientFeatures::has_disable_all_goals_optimization_for_dna_feature($self->subclient_client_id),
                has_increased_cpa_limit_for_pay_for_conversion => Client::ClientFeatures::has_increased_cpa_limit_for_pay_for_conversion($self->subclient_client_id),
                has_disable_autobudget_week_bundle_feature => Client::ClientFeatures::has_disable_autobudget_week_bundle_in_api_feature($self->subclient_client_id),
                has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed => Client::ClientFeatures::has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed($self->subclient_client_id),
                has_flat_cpc_disabled => Client::ClientFeatures::has_flat_cpc_disabled($self->subclient_client_id),
                has_flat_cpc_adding_disabled => Client::ClientFeatures::has_flat_cpc_adding_disabled($self->subclient_client_id),
            }
        );
        if ( $strategy_error ) {
            $item->add_error( error_BadParams( iget( $strategy_error ) ) );
        }
    }

    if ( exists $model->{day_budget} || exists $model->{day_budget_show_mode} ) {
        my $vr = validate_camp_day_budget(
            strategy => $model->{strategy}{name},
            is_autobudget => $model->{strategy}{is_autobudget},
            new_day_budget_data => {
                day_budget => $model->{day_budget},
                day_budget_show_mode => $model->{day_budget_show_mode},
            },
            wallet_day_budget => $wallet_day_budget,
            currency => $model->{currency},
            new_camp => 1
        );

        $item->add_error( $_ ) foreach @{ $vr->get_errors };
        $item->add_warning( $_ ) foreach @{ $vr->get_warnings };
    }

    return;
}

=head2 _is_strategy_pay_for_conversion_prohibited

    Проверить разрешено ли клиенту использовать стратегию "Оплата за конверсии"

    Параметры:
      - model - внутренняя модель кампании, в которой проверяется параметр стартегии.
      - cpa_pay_for_conversions_extended_mode_allowed - фича для расширенного режима Оплаты за конверсии.
            https://wiki.yandex-team.ru/users/tudalova/strategii-v-direkte/oplata-za-konversii-v-direkte/#fichi

    возвращает:
     1 - если стратегия недоступна пользователю
     0 - если доступна для использования

=cut

sub _is_strategy_pay_for_conversion_prohibited {
    my ($model, $cpa_pay_for_conversions_extended_mode_allowed) = @_;
    for my $type (qw/search net/) {
        if ($model->{strategy}{$type}{pay_for_conversion} && !$cpa_pay_for_conversions_extended_mode_allowed) {
            return 1;
        }
    }
    return 0;
}

sub _preprocess_add_update {
    my ($model, $old_model) = @_;

    if ($model->{placement_types}) {
        $model->{placement_types} = convert_placement_types($model->{placement_types}, ($old_model // {})->{placement_types});
    }

    if ($model->{strategy} && $model->{strategy}{net} &&
        $model->{strategy}{net}{name} eq 'default'
    ) {
        $model->{ContextLimit} =
            delete $model->{strategy}{net}{ContextLimit};
    }

        if ($model->{strategy} && $model->{strategy}{net} &&
        ($model->{strategy}{net}{name} eq 'autobudget_avg_cpv' || $model->{strategy}{net}{name} eq 'autobudget_avg_cpv_custom_period')
    ) {
        $model->{strategy}->{net}->{pay_for_conversion} = 1;
    }

    if ($model->{type} && $model->{type} eq 'cpm_banner') {
        $model->{platform} = 'context';
        $model->{rf} //= 0 if exists $model->{rf};
        $model->{rfReset} //= 0 if exists $model->{rfReset};
    }

    # 'cause column is named 'start_time' in DB,
    # but 'start_date' in Direct::Model::Campaign::*
    # and API::Service::Campaigns::ConvertMap
    $model->{start_date}  = $model->{start_time};
    $model->{finish_date} = $model->{finish_time};

    return $model;
}

=head2 update

    обновление полей кампаний

=cut

sub update {
    my ($self, $request) = @_;

    my $rs = API::Service::ResultSet::Update->new(@{$request->{Campaigns}});

    my $e = $self->_validate_update_request($rs);
    return $e if $e;

    # проверим есть ли у клиента сервисируемые кампании,
    # если есть, то запомним менеджера
    $self->_guess_and_remember_manager();

    my $client_currencies = get_client_currencies($self->subclient_client_id);
    my $work_currency = $client_currencies->{work_currency};
    $self->_validate_update_request_items($rs, $work_currency);

    $self->_enable_relevant_keywords_for_text_camps_by_default($rs);

    my $converter = $self->converter('internal');

    my $old_campaigns = $self->_get_old_campaigns($rs);
    my %old_campaigns_hash = map { $_->{cid} => $_ } @$old_campaigns;
    my @converted_campaigns;
    for my $item ($rs->list_ok) {
        my $old_campaign = $old_campaigns_hash{$item->object->{Id}};

        if (camp_kind_in(type => $old_campaign->{type}, $self->_get_supported_camp_kind())) {
            if (has_type_structure($item->object)) {
                my $request_type = get_type_by_structure($item->object);
                my $type = convert_type_to_external($old_campaign->{type});
                if ($request_type eq $type) {
                    $item->object->{Type} = $type;
                    disclose_type_structure($item->object);
                } else {
                    $item->add_error(error_BadParams(iget('Тип кампании %s не совпадает с указанным в запросе %s', $type, $request_type)));
                }
            }
        } else {
            $item->add_error(error_NotSupported_CampaignType());
        }
        $self->_validate_time_targeting($item, $old_campaign->{type});
        $self->_validate_excluded_domains_for_camp_types($item, $old_campaign->{type});
        $self->_validate_negative_keywords($item, $old_campaign->{type});
        $self->_preprocess_nillable_fields($item);
        $self->_validate_update_relevant_keywords($item, $old_campaign);
        $self->_validate_update_strategy_id($item, $old_campaign);

        my $converted = $converter->convert(dclone($item->object));
        $self->_update_prepare_model_default_values($item, $converted, $old_campaign);

        push @converted_campaigns, _preprocess_add_update($converted, $old_campaign);
    }

    my $campaigns = $self->_add_old_campaigns_fields(\@converted_campaigns, $old_campaigns);
    my %campaigns_changes_hash = map { $_->{cid} => $_ } @converted_campaigns;
    my %campaigns_hash = map { $_->{cid} => $_ } @$campaigns;

    _detect_strategy_changes($rs, \%campaigns_changes_hash, \%campaigns_hash, \%old_campaigns_hash);

    _validate_update_strategy($rs, \%campaigns_changes_hash, \%campaigns_hash, \%old_campaigns_hash);

    $self->_validate_update_campaigns($rs, $campaigns, $old_campaigns, $work_currency);

    my @result;
    for my $item ($rs->list) {
        my %result;
        if (! $item->has_errors) {
            my $cid = $item->object->{Id};
            my $campaign = $campaigns_hash{$cid};
            my $old_camp = $old_campaigns_hash{$cid};
            $self->_preprocess_network_default_strategy_params($item, $campaign, $old_camp);
            my $old_strategy_id = $old_camp->{strategy_id};
            my $strategy_id = $campaign->{strategy_id};
            my $need_to_create_new_package = ($campaign->{strategy_is_public} eq 'Yes')
                && (!defined $strategy_id || (($strategy_id == $old_strategy_id) && $campaign->{has_strategy_changes})) ? 1 : 0;
            my $package_strategy_is_changed = defined $strategy_id && ($strategy_id != $old_strategy_id) ? 1 : 0;
            $self->_save($campaign, 0, package_strategy_is_changed => $package_strategy_is_changed, need_to_create_new_package => $need_to_create_new_package);
            $result{Id} = $campaign->{cid};
        } else {
            $result{Errors} = action_errors($item->list_errors)
                if $item->has_errors;
        }
        $result{Warnings} = action_errors($item->list_warnings)
            if $item->has_warnings;
        push @result, \%result;
    }

    $self->units_withdraw_for_results( campaign => $rs );

    return {
        UpdateResults => \@result
    };
}

sub _update_prepare_model_default_values {
    my ($self, $item, $campaign, $old_campaign) = @_;

    if ($campaign->{type} && $campaign->{type} eq 'cpm_banner') {
        my $is_net_strategy_changed = $campaign->{strategy} && $campaign->{strategy}{net};
        my $video_type = $campaign->{eshows_video_type};
        my $old_strategy_type = $old_campaign->{strategy_name};
        my $is_old_strategy_type_avg_cpv = $old_strategy_type eq "autobudget_avg_cpv" ||
            $old_strategy_type eq "autobudget_avg_cpv_custom_period";

        if ($is_net_strategy_changed) {
            my $strategy_type = $campaign->{strategy}{net}{name};
            my $is_strategy_type_avg_cpv = $strategy_type eq "autobudget_avg_cpv" ||
                $strategy_type eq "autobudget_avg_cpv_custom_period";

            if ($is_strategy_type_avg_cpv) {
                $campaign->{eshows_video_type} = undef;
                if (defined $video_type) {
                    $item->add_warning(
                        warning_ParamNotUseful(
                            iget('Параметр %s для заданной стратегии не поддерживается', 'VideoTarget')
                        )
                    );
                }
            } elsif ($is_old_strategy_type_avg_cpv && !defined $video_type) {
                $campaign->{eshows_video_type} = "completes";
            }
        } elsif (defined $video_type) {
            if ($is_old_strategy_type_avg_cpv) {
                $campaign->{eshows_video_type} = undef;
                if (defined $video_type) {
                    $item->add_warning(
                        warning_ParamNotUseful(
                            iget('Параметр %s для заданной стратегии не поддерживается', 'VideoTarget')
                        )
                    );
                }
            }
        }
    }

    return $campaign;
}

sub _detect_strategy_changes {
    my ($rs, $campaigns_changes_hash, $campaigns_hash, $old_campaigns_hash) = @_;
    for my $item ($rs->list_ok) {
        my $cid = $item->object->{Id};
        my $model_change = $campaigns_changes_hash->{$cid};
        my $model = $campaigns_hash->{$cid};
        my $old_model = $old_campaigns_hash->{$cid};

        $model->{has_strategy_changes} = 0;

        if ($model_change->{strategy} && is_strategy_change_ignore_platform($model->{strategy}, $old_model->{strategy})){
            $model->{has_strategy_changes} = 1;
            last;
        }

        for my $field (qw/attribution_model strategy_name day_budget_show_mode/) {
            next unless exists $model_change->{$field};
            if ($model->{$field} ne $old_model->{$field}) {
                $model->{has_strategy_changes} = 1;
                last;
            }
        }

        for my $field (qw/meaningful_goals/) {
            next unless exists $model_change->{$field};
            if (_is_meaningful_goals_changed($model_change->{$field}, $old_model->{$field})) {
                $model->{has_strategy_changes} = 1;
                last;
            }
        }

        for my $field (qw/day_budget/) {
            next unless exists $model_change->{$field};
            if ($model->{$field} != $old_model->{$field}) {
                $model->{has_strategy_changes} = 1;
                last;
            }
        }

        if ($model_change->{opts}){
            if (!$model->{opts}->{enable_cpc_hold} != !$old_model->{opts}->{enable_cpc_hold}) {
                $model->{has_strategy_changes} = 1;
                last;
            }
        }


        if ($model_change->{metrika_counters}){
            my @counter_ids = split /[\s,]+/, ($model->{metrika_counters} =~ s/(^[\s,]+|[\s,]+$)//gr);
            my @old_counter_ids = split /[\s,]+/, ($old_model->{metrika_counters} =~ s/(^[\s,]+|[\s,]+$)//gr);
            if (@{ xdiff(\@counter_ids, \@old_counter_ids) }){
                $model->{has_strategy_changes} = 1;
                last;
            }
        }
    }
    return;
}

sub _is_meaningful_goals_changed {
    my ($new, $old) = @_;

    my %new_hash = map { $_->{goal_id} => $_ } @$new;
    my %old_hash = map { $_->{goal_id} => $_ } @$old;

    if (scalar keys %new_hash != scalar keys %old_hash){
        return 1;
    }

    foreach my $goal_id (keys %new_hash) {
        if (!exists $old_hash{$goal_id}) {
            return 1;
        }

        my $new_item = $new_hash{$goal_id};
        my $old_item = $old_hash{$goal_id};

        if (!exists $old_item->{is_metrika_source_of_value}){
            $old_item->{is_metrika_source_of_value} = 0;
        }

        if (!exists $new_item->{is_metrika_source_of_value}){
            $new_item->{is_metrika_source_of_value} = 0;
        }

        foreach my $field (keys %$new_item){
            if (!exists $old_item->{$field}) {
                return 1;
            }
            my $new_value = defined $new_item->{$field} ? $new_item->{$field} : '';
            my $old_value = defined $old_item->{$field} ? $old_item->{$field} : '';
            if (($new_value ne $old_value) && ($new_value != $old_value)) {
                return 1;
            }
        }
    }

    return 0;
}

sub _validate_update_strategy {
    my ($rs, $campaigns_changes_hash, $campaigns_hash, $old_campaigns_hash) = @_;
    for my $item ($rs->list_ok) {
        my $cid = $item->object->{Id};
        my $model_change = $campaigns_changes_hash->{$cid};
        my $model = $campaigns_hash->{$cid};
        my $old_model = $old_campaigns_hash->{$cid};
        my $strategies_change = $model_change->{strategy};
        my $strategies_model = $model->{strategy};
        my $old_strategy = $old_campaigns_hash->{$cid}->{strategy};

        for my $strategy_type (qw/search net/) {
            next unless $strategies_change->{$strategy_type};
            if ( $strategies_change->{$strategy_type}->{name} eq "autobudget_week_bundle") {
                if ( ((!$strategies_change->{$strategy_type}->{bid} || !$strategies_change->{$strategy_type}->{avg_bid}) &&
                        $strategies_model->{$strategy_type}->{bid} &&  $strategies_model->{$strategy_type}->{avg_bid})
                ) {
                    $item->add_error(
                    error_InconsistentState(
                        'Параметры AverageCpc и BidCeiling являются взаимоисключающими. Один из них был задан ранее. Чтобы задать другой параметр, нужно вместе с ним передать null для ранее заданного параметра.'));
                }
            }
        }
        if (is_network_default_strategy($strategies_model) && $strategies_change->{search}
            && !is_autobudget($old_strategy) && is_autobudget($strategies_model)) {
            if (!$item->has_flag($HAS_NETWORK_DEFAULT_LIMIT_PERCENT_FLAG_NAME)) {
                $item->add_warning( warning_ParamNotUseful( iget('Параметр %s для заданной стратегии не поддерживается', 'LimitPercent') ));
            }
            if (!$item->has_flag($HAS_MAINTAIN_NETWORK_CPC_SETTING_FLAG_NAME) && $model->{opts} && $model->{opts}{enable_cpc_hold}) {
                $item->add_warning(
                    warning_SettingNotChanged( iget('Настройка %s для заданной стратегии не поддерживается', 'MAINTAIN_NETWORK_CPC') )
                );
            }
        }

        if (exists $model_change->{strategy_id} && defined $model->{strategy_id} && !$model->{strategy_id}){
            $item->add_error(error_NotFound_Strategy());
        }

        if ($model->{has_strategy_changes} && $model->{strategy_id}){
            if ($model->{strategy_id} == $old_model->{strategy_id}){
                if ($old_model->{strategy_is_public} eq 'Yes'){
                    $item->add_warning(
                        warning_NewStrategyHasBeenCreated()
                    );
                }
            } else {
                $item->add_warning(
                    warning_ParamNotUseful()
                );
            }
        }
    }
    return;
}

=head2 _preprocess_network_default_strategy_params

    Для сочетания стратегии на поиске и network_default в сети, убираем из данных кампании ContextLimit и enable_cpc_hold

=cut

sub _preprocess_network_default_strategy_params {
    my ($self, $rs, $campaign, $old_campaign) = @_;

    if (is_network_default_strategy($campaign->{strategy})) {
        if (is_autobudget($campaign->{strategy})) {
            if ($old_campaign) { # сохраняем текущие настройки
                $campaign->{opts}{enable_cpc_hold} = $old_campaign->{opts}{enable_cpc_hold};

            } else { # проставляем дефолтные значения
                $campaign->{ContextLimit} = $DEFAULT_LIMIT_PERCENT;
                $campaign->{opts}{enable_cpc_hold} = 1;
            }

            if ($rs->has_flag($HAS_NETWORK_DEFAULT_LIMIT_PERCENT_FLAG_NAME)) {
                $rs->add_warning( warning_ParamNotUseful( iget('Параметр %s для заданной стратегии не поддерживается', 'LimitPercent') ));
            }

            if ($rs->has_flag($HAS_MAINTAIN_NETWORK_CPC_SETTING_FLAG_NAME)) {
                $rs->add_warning(
                    warning_SettingNotChanged( iget('Настройка %s для заданной стратегии не поддерживается', 'MAINTAIN_NETWORK_CPC') )
                );
            }
        } else {
            if ($old_campaign && is_network_default_strategy($old_campaign->{strategy})) {
                $campaign->{ContextLimit} = $old_campaign->{ContextLimit} if !$campaign->{ContextLimit};
            }

            $campaign->{ContextLimit} ||= $DEFAULT_LIMIT_PERCENT;
        }
    }

    return;
}

sub _preprocess_nillable_fields {
    my ($self, $campaign) = @_;

    my $campaign_data = $campaign->object;
    for my $field (
        qw/RelevantKeywords EndDate NegativeKeywords BlockedIps ExcludedSites FrequencyCap PriorityGoals StrategyId/,
        [RelevantKeywords => 'OptimizeGoalId'],
        [TimeTargeting => 'HolidaysSchedule'],
        [FrequencyCap => 'PeriodDays'],
    ) {
        my $structure = $campaign_data;
        if (ref $field) {
            next unless exists $structure->{$field->[0]};
            $structure = $structure->{$field->[0]};
            $field = $field->[1];
        }
        if (exists $structure->{$field}){
            $structure->{$field} = undef
                if $self->is_nil($structure->{$field});
        }
    }
    if (exists $campaign_data->{BiddingStrategy}) {
        my $strategies = $campaign_data->{BiddingStrategy};
        for my $type (qw/Search Network/) {
            next unless $strategies->{$type};
            my $strategy = $strategies->{$type};
            my $settings_structure_name = get_structure_name_by_strategy($strategy->{BiddingStrategyType});
            my $settings_structure = $strategy->{$settings_structure_name};
            for my $setting (keys %$settings_structure) {
                if ($self->is_nil($settings_structure->{$setting})) {
                    $settings_structure->{$setting} = undef;
                }
            }
        }
    }
    return;
}

=head2 _validate_update_relevant_keywords($item, $old_campaign)

    Валидируем наличие полей BudgetPercent и Mode в структуре RelevantKeywords в случае, если
    в предыдущем состоянии RelevantKeywords не задана и добавляем соответствующие ошибки

=cut

sub _validate_update_relevant_keywords {
    my ($self, $item, $old_campaign) = @_;

    my $new_campaign = $item->object;
    if ($old_campaign->{broad_match_flag} eq 'No'
            && exists $new_campaign->{RelevantKeywords}
            && defined $new_campaign->{RelevantKeywords}
            && ! exists $new_campaign->{RelevantKeywords}{BudgetPercent}
    ) {
        $item->add_error(error_ReqField(undef, field => 'BudgetPercent'));
    }

    return;
}

=head2 _validate_update_strategy_id($item, $old_campaign)

    Валидируем публичность пакета в случае strategy_id = null

=cut

sub _validate_update_strategy_id {
    my ($self, $item, $old_campaign) = @_;

    my $new_campaign = $item->object;
    if ($old_campaign->{strategy_is_public} eq 'No'
        && exists $new_campaign->{StrategyId}
        && !defined $new_campaign->{StrategyId}
    ) {
        $item->add_error(error_InconsistentState(
            iget("Невозможно отвязать кампанию от непубличного пакета")
        ));
    }

    return;
}

sub _get_old_campaigns {
    my ($self, $rs) = @_;
    my @cids = map {$_->object->{Id}} $rs->list_ok;
    my $old_campaigns = $self->_get_campaigns_from_db({'c.cid' => \@cids});
    return $old_campaigns;
}

sub _add_old_campaigns_fields {
    my ($self, $models, $old_models) = @_;
    my @cids = map {$_->{cid}} @$models;
    my $favorite_camps = mass_get_favorite_camps($self->subclient_uid);
    my $metrika_counters = camp_metrika_counters_multi(\@cids);
    my %old_models_hash = map { $_->{cid} => $_ } @$old_models;
    my @result_models;
    for my $model (@$models) {
        my $id = $model->{cid};
        my $old_model = $old_models_hash{$id};
        croak 'old campaign is not found' unless $old_model;
        $old_model->{fio} = $old_model->{FIO};
        $old_model->{broad_match_flag} = $old_model->{broad_match_flag} eq 'No' ? 0 : 1;

        my @sms_flags = split(','=> delete $old_model->{sms_flags});
        $old_model->{$_} = 1 for @sms_flags;
        my $sms_settings_empty_fields = xminus \@SMS_SETTINGS_FIELDS, [keys %$model];
        my $sms_settings_old_values = hash_cut $old_model, $sms_settings_empty_fields;
        hash_merge ($model, $sms_settings_old_values);

        # в save_camp проверка на undef
        $old_model->{fairAuction} = undef if $old_model->{fairAuction} == 0;
        foreach my $field (qw/sendWarn sendAccNews/) {
            # в save_camp проверка на undef
            $old_model->{$field} = undef if $old_model->{$field} eq 'No';
        }
        my $time_target = TimeTarget::parse_timetarget($old_model->{timeTarget});
        hash_merge($old_model, $time_target);

        my $time_target_empty_fields = xminus \@TIME_TARGET_FIELDS, [keys %$model];
        my $time_target_old_values = hash_cut $old_model, $time_target_empty_fields;
        hash_merge ($model, $time_target_old_values);

        hash_merge($old_model, convert_sms_time_string_to_struct($old_model->{sms_time})) if $old_model->{sms_time};
        $old_model->{metrika_counters} = join(' ' => @{$metrika_counters->{$id}})
            if defined $metrika_counters->{$id};
        $old_model->{is_favorite} = $favorite_camps->{$id} ? 1 : 0;
        $old_model->{statusMetricaControl} = $old_model->{statusMetricaControl} eq 'Yes' ? 1 : 0;
        $old_model->{campaign_minus_words} = $old_model->{minus_words};
        $old_model->{strategy} = Campaign::define_strategy($old_model);

        if ( defined $old_model->{finish_time} && $old_model->{finish_time} eq '0000-00-00' ) {
            $old_model->{finish_time} = undef;
        }

        my $result_strategy = merge_strategies($model->{strategy} || {}, $old_model->{strategy});
        my $result_model = {};
        hash_merge($result_model, $old_model, $model);

        $result_model->{opts} = hash_merge({}, $old_model->{opts}, $model->{opts});

        $result_model->{strategy} = $result_strategy;

        push @result_models, $result_model;
    }

    return \@result_models;
}

sub _validate_update_request {
    my ($self, $camp_object) = @_;
    return error_RequestLimitExceeded(
        iget('Разрешено изменять не более %s кампаний в одном запросе', $UPDATE_IDS_LIMIT)
    ) if $camp_object->count_ok > $UPDATE_IDS_LIMIT;
}

sub _validate_update_request_items {
    my ($self, $rs, $currency) = @_;

    for my $item ($rs->list) {
        next if $self->_validate_campaign_type($item);
        $self->_validate_strategy_consistent($item);
        $self->_validate_context_strategy($item);
        $self->_validate_empty_array_items($item);
        $self->_validate_duplicate_array_items($item);
        $self->_validate_WeeklyPacketOfClicks_params($item);
        $self->_validate_WbMaximumConversionRate_for_update($item);
        $self->_validate_daily_budged_fill_default($item, $currency);
        $self->_validate_require_servicing($item);
        $self->_validate_campaign_settings($item);
        $self->_validate_frequency_cap($item);
        $self->_validate_meaningful_goals($item);

        if (defined $item->object->{TimeZone}) {
            $item->add_error(error_BadParams(iget('Неверно указана временная зона')))
                unless DateTime::TimeZone->is_valid_name($item->object->{TimeZone});
        }
    }

    $rs->add_error_for_id_dups(
        error_Duplicated(
            iget('Id кампании присутствует в запросе более одного раза')
        )
    );

    $rs->add_error_when(
        error_InvalidField_NotPositive(undef, field => 'Id'),
        sub { not is_valid_id( shift ) ? 1 : 0 }
    );

    $self->_validate_update_campaign_availability($rs);
    $self->_validate_update_campaign_rights($rs);

    return;
}

sub _validate_strategy_consistent {
    my ($self, $item) = @_;
    my $type = get_type_structure_name($item->object);

    return unless defined $type &&
                  defined $item->object->{$type} &&
                  defined $item->object->{$type}{BiddingStrategy};

    my $strategies = $item->object->{$type}{BiddingStrategy};
    my ($search_strategy_has_error, $network_strategy_has_error);
    my $consistency = is_strategy_has_no_more_than_one_structures($strategies);
    if (exists $consistency->{Search} && !$consistency->{Search}) {
        $item->add_error(error_BadParams(iget('Стратегия на поиске может содержать не больше одной структуры с настройками')));
        $search_strategy_has_error = 1;
    }
    if (exists $consistency->{Network} && !$consistency->{Network}) {
        $item->add_error(error_BadParams(iget('Стратегия в Рекламной сети может содержать не больше одной структуры с настройками')));
        $network_strategy_has_error = 1;
    }
    $consistency = is_strategy_consistent($strategies);
    if (exists $consistency->{Search} && !$consistency->{Search} && !$search_strategy_has_error) {
        $item->add_error(error_BadParams(iget('Cтруктура с параметрами для стратегии на поиске не соответствует названию стратегии')));
        $search_strategy_has_error = 1;
    }
    if (exists $consistency->{Network} && !$consistency->{Network} && !$network_strategy_has_error) {
        $item->add_error(error_BadParams(iget('Cтруктура с параметрами для стратегии в Рекламной сети не соответствует названию стратегии')));
        $network_strategy_has_error = 1;
    }
    $consistency = is_strategy_has_needed_structure($strategies, $type);
    if (exists $consistency->{Search} && !$consistency->{Search} && !$search_strategy_has_error) {
        $item->add_error(error_BadParams(iget('Cтратегия %s на поиске должна содержать структуру с настройками', $strategies->{Search}{BiddingStrategyType})));
        $search_strategy_has_error = 1;
    }
    if (exists $consistency->{Network} && !$consistency->{Network} && !$network_strategy_has_error) {
        $item->add_error(error_BadParams(iget('Стратегия %s в Рекламной сети должна содержать структуру с настройками', $strategies->{Network}{BiddingStrategyType})));
        $network_strategy_has_error = 1;
    }

    return;
}

sub _validate_context_strategy {
    my ($self, $item) = @_;
    my $type = get_type_structure_name($item->object);
    return unless defined $type &&
                  defined $item->object->{$type} &&
                  defined $item->object->{$type}{BiddingStrategy};
    my $strategy = $item->object->{$type}{BiddingStrategy};

    if(defined $strategy->{Network}) {
        my $multiplicity = 10;
        my $name = $strategy->{Network}{BiddingStrategyType};
        my $params_field = get_structure_name_by_strategy($name);
        my $limit = $strategy->{Network}{$params_field}{LimitPercent};

        if (defined $limit && !$self->is_nil($limit)) {
            $item->mark($HAS_NETWORK_DEFAULT_LIMIT_PERCENT_FLAG_NAME);
            if ($limit < 10 || $limit > 100) {
                $item->add_error(error_InvalidField(
                    iget("Максимальный процент бюджета в Рекламной сети должен быть от 10 до 100 %")
                ));
            } elsif ($limit % $multiplicity != 0) {
                $item->add_error(error_InvalidField(
                    iget("Максимальный процент бюджета в Рекламной сети должен быть кратен %s", $multiplicity)
                ));
            }
        }
    }

    return;
}

my @platforms = qw/_invalid_ search context both/;

sub _build_state_for_model {
    my ( $self, $model, %opts ) = @_;

    my $state = {
        campaign_type           => $model->{type},
        campaign_name           => $model->{name},
        client_fio              => $model->{fio},
        client_id               => $self->subclient_client_id,
        email                   => $model->{email},
        start_date              => $model->{start_time},
        timezone_id             => $model->{timezone_id} || 0,
        time_target             => $model->{timeTarget},
        money_warning_threshold => $model->{money_warning_value} // 20,
        position_check_interval => $model->{warnPlaceInterval}   // 60,
        currency                => $model->{currency},
        strategy_id             => $model->{strategy_id},

        ( $model->{strategy} ? (
                                _autobudget    => $model->{strategy}{is_autobudget} ? 'Yes' : 'No',
                               )
                             : () ),


        ( exists $model->{finish_time}          ? ( finish_date           => $model->{finish_time}                     ) : () ),
        ( exists $model->{DontShow}             ? ( _disabled_domains     => $model->{DontShow}                        ) : () ),
        ( exists $model->{disabledIps}          ? ( _disabled_ips         => $model->{disabledIps}                     ) : () ),
        ( exists $model->{placement_types}      ? ( placement_types       => $model->{placement_types}                 ) : () ),
        ( exists $model->{campaign_minus_words} ? ( minus_words           => $model->{campaign_minus_words} // ''      ) : () ),
        ( exists $model->{broad_match_flag}     ? ( broad_match_flag      => $model->{broad_match_flag} ? 'Yes' : 'No' ) : () ),
        ( exists $model->{broad_match_limit}    ? ( broad_match_limit     => $model->{broad_match_limit}               ) : () ),
        ( exists $model->{broad_match_goal_id}  ? ( broad_match_goal_id   => $model->{broad_match_goal_id}             ) : () ),
        ( exists $model->{metrika_counters}     ? ( _metrika_counters     => $model->{metrika_counters}                ) : () ),
        ( exists $model->{meaningful_goals}     ? ( meaningful_goals      => $model->{meaningful_goals} || []          ) : () ),
        ( exists $model->{sms_time_hour_from}   ? ( sms_time_from_hours   => $model->{sms_time_hour_from} // ''        ) : () ),
        ( exists $model->{sms_time_min_from}    ? ( sms_time_from_minutes => $model->{sms_time_min_from}  // ''        ) : () ),
        ( exists $model->{sms_time_hour_to}     ? ( sms_time_to_hours     => $model->{sms_time_hour_to}   // ''        ) : () ),
        ( exists $model->{sms_time_min_to}      ? ( sms_time_to_minutes   => $model->{sms_time_min_to}    // ''        ) : () ),
        ( exists $model->{attribution_model}    ? ( attribution_model     => $model->{attribution_model}  // ''        ) : () ),
    };

    my $is_search = $model->{strategy}->{is_search_stop} ? 0 : 1;
    my $is_context = $model->{strategy}->{is_net_stop} ? 0 : 2;
    my $platform = $platforms[$is_search + $is_context];
    if ($platform ne '_invalid_') {
        $state->{platform} = $platform;
    } else {
        $state->{platform} = $model->{platform};
    }

    if (exists $model->{opts}) {
        my $camp_opts = $model->{opts} // {};
        $state->{opts} = join(',', grep { $camp_opts->{$_} } keys %$camp_opts);
    }

    if ( $opts{new} ) {
        $state->{time_target} = TimeTarget::pack_timetarget( $model );
    } else {
        $state->{id}          = $model->{cid};
        $state->{time_target} = $model->{timeTarget};
    }

    return $state;
}

sub _validate_update_campaigns {
    my ($self, $rs, $models, $old_models, $work_currency) = @_;

    my $wallet_day_budget = $self->_get_wallet_day_budget($work_currency);

    my %models_hash     = map { $_->{cid} => $_ } @$models;
    my %old_models_hash = map { $_->{cid} => $_ } @$old_models;

    my $meaningful_goals_data = $self->_prefetch_meaningful_goals_data_on_update($rs);

    my $prefetched_goals = $self->_prefetch_goals($rs);

    for my $item ( $rs->list_ok ) {
        my $model     = $models_hash{ $item->object->{Id} };
        my $old_model = $old_models_hash{ $item->object->{Id} };

        $self->_prevalidate_placement_types($item, $model);
        $self->_prevalidate_broad_match_on_update($item, $model, $old_model);

        my $state     = $self->_build_state_for_model( $model );
        my $old_state = $self->_build_state_for_model( $old_model );

        my $model_obj;

        if (!$self->get_business_logic_class_by_type($model->{type})) {
            die "Not error_NotSupported_CampaignType for type $model->{type}";
        } elsif ( $model->{type} eq 'text' ) {
            $model_obj = Direct::Model::CampaignText->new( %$old_state );
            $model_obj->merge_with( Direct::Model::Campaign->new( %$state ) );
        } elsif ( $model->{type} eq 'mobile_content' ) {
            $model_obj = Direct::Model::CampaignMobileContent->new( %$old_state );
            $model_obj->merge_with( Direct::Model::CampaignMobileContent->new( %$state ) );
        } elsif ( $model->{type} eq 'dynamic' ) {
            $model_obj = Direct::Model::CampaignDynamic->new( %$old_state );
            $model_obj->merge_with( Direct::Model::CampaignDynamic->new( %$state ) );
        } elsif ( $model->{type} eq 'performance' ) {
            $model_obj = Direct::Model::CampaignPerformance->new( %$old_state );
            $model_obj->merge_with( Direct::Model::CampaignPerformance->new( %$state ) );
        } elsif ( $model->{type} eq 'cpm_banner' ) {
            $model_obj = Direct::Model::CampaignCpmBanner->new( %$old_state );
            $model_obj->merge_with( Direct::Model::CampaignCpmBanner->new( %$state ) );
        } elsif ( $model->{type} eq 'content_promotion' ) {
            $model_obj = Direct::Model::CampaignContentPromotion->new( %$old_state );
            $model_obj->merge_with( Direct::Model::CampaignContentPromotion->new( %$state ) );
        } else {
            die "Unexpected camp type in validation $model->{type}";
        }

        if ($model->{type} eq 'performance' && $model->{metrika_counters}) {
            my @errors = Models::Campaign::validate_campaign_metrika_counters($model->{metrika_counters},
                $model->{type}, $self->subclient_uid, $self->operator_uid());
            $item->add_error(error_BadParams( iget($errors[0]))) if @errors;
        }

        my $turbo_apps_allowed = Client::ClientFeatures::has_turbo_app_allowed($self->subclient_client_id);
        if (!$old_model->{opts}{has_turbo_app} && $model->{opts}{has_turbo_app} && !$turbo_apps_allowed) {
            $item->add_error(error_LimitAccess(iget('Продвижение турбо-аппов недоступно')));
        }

        my $old_strategy_id = $old_model->{strategy_id};
        my $strategy_id = $model->{strategy_id};
        my $package_strategy_is_changed = defined $strategy_id && ($strategy_id != $old_strategy_id) ? 1 : 0;
        my $is_allowed_to_use_value_from_metrika = Campaign::is_allowed_to_use_value_from_metrika($model->{strategy});

        my $vres = validate_update_campaign($model_obj,
            skip_not_changed_meaningful_goals => 1,
            prefetched_meaningful_goals_data  => $meaningful_goals_data,
            strategy_use_meaningful_goals     => is_campaign_strategy_use_meaningful_goals_optimization($model->{strategy}),
            translocal_tree                   => 'api',
            uid                               => $self->subclient_uid,
            unavailable_auto_goals_allowed    => $API::Settings::UNAVAILABLE_AUTO_GOALS_ALLOWED_APP_ID->{$self->application_id},
            package_strategy_is_changed                => $package_strategy_is_changed,
            wallet_cid                        => $model->{wallet_cid},
            is_allowed_to_use_value_from_metrika => $is_allowed_to_use_value_from_metrika
        );

        $vres->process_descriptions(%DEFECT_DESCRIPTION_FIELD_MAP);

        $item->add_error( $_ ) foreach @{ $vres->get_errors };
        $item->add_warning( $_ ) foreach @{ $vres->get_warnings };

        $old_model->{start_date} = $model->{start_date};
        $old_model->{finish_date} = $model->{finish_date};

        my $cpa_pay_for_conversion_extended_mode_allowed = Client::ClientFeatures::has_cpa_pay_for_conversions_extended_mode_allowed($self->subclient_client_id);
        if ($model->{strategy}->{search}->{name} eq 'no_premium') {
            $item->add_error(
                error_NotSupported(iget('Стратегия IMPRESSIONS_BELOW_SEARCH не поддерживается'))
            );
        } elsif(_is_strategy_pay_for_conversion_prohibited($model, $cpa_pay_for_conversion_extended_mode_allowed)) {
            $item->add_error(error_LimitAccess_StrategyPayForConversion());
        } else {
            my $has_flat_cpc_adding_disabled = Client::ClientFeatures::has_flat_cpc_adding_disabled($self->subclient_client_id);
            my $strategy_error = Campaign::validate_camp_strategy($old_model, $model->{strategy},
                {
                    is_api => 1,
                    has_cpa_pay_for_conversions_extended_mode_allowed => $cpa_pay_for_conversion_extended_mode_allowed,
                    has_cpa_pay_for_conversions_mobile_apps_allowed => Client::ClientFeatures::has_cpa_pay_for_conversions_mobile_apps_allowed($self->subclient_client_id),
                    request_meaningful_goals => $model_obj->{meaningful_goals},
                    defect_description_field_map => \%STRATEGY_DEFECT_DESCRIPTION_FIELD_MAP,
                    prefetched_goals => $prefetched_goals,
                    has_edit_avg_cpm_without_restart_enabled => Client::ClientFeatures::has_edit_avg_cpm_without_restart_feature($self->subclient_client_id),
                    has_disable_all_goals_optimization_for_dna_enabled => Client::ClientFeatures::has_disable_all_goals_optimization_for_dna_feature($self->subclient_client_id),
                    has_increased_cpa_limit_for_pay_for_conversion => Client::ClientFeatures::has_increased_cpa_limit_for_pay_for_conversion($self->subclient_client_id),
                    has_disable_autobudget_week_bundle_feature => Client::ClientFeatures::has_disable_autobudget_week_bundle_in_api_feature($self->subclient_client_id),
                    has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed => Client::ClientFeatures::has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed($self->subclient_client_id),
                    has_flat_cpc_disabled => Client::ClientFeatures::has_flat_cpc_disabled($self->subclient_client_id),
                    has_flat_cpc_adding_disabled => $has_flat_cpc_adding_disabled,
                }
            );

            if ( $strategy_error ) {
                chomp $strategy_error;
                $item->add_error(error_BadParams(iget($strategy_error)));
            } elsif($has_flat_cpc_adding_disabled && ($old_model->{type} eq 'text' || $old_model->{type} eq 'dynamic' || $old_model->{type} eq 'mobile_content')
                && !$model->{strategy}->{is_net_stop} && !$model->{strategy}->{is_search_stop} && $model->{strategy}->{name} eq 'default') {
                $item->add_warning(warning_BadUsage(iget('Стратегия совместого управления ставками не поддерживается')));
            }
        }

        if ( exists $item->object->{BiddingStrategy}
            && !exists $item->object->{DailyBudget}
            && is_autobudget($model->{strategy})
        ) {
            if ($model->{day_budget} != 0) {
                $model->{day_budget} = 0;
                $model->{day_budget_show_mode} = 'default';
                $item->add_warning(warning_DailyBudgetReset());
            }
        }

        if (exists $item->object->{DailyBudget} && !$self->is_nil($item->object->{DailyBudget})) {
            my $vr = validate_camp_day_budget(
                strategy => $model->{strategy}{name},
                is_autobudget => $model->{strategy}{is_autobudget},
                new_day_budget_data => {
                    day_budget => $model->{day_budget},
                    day_budget_show_mode => $model->{day_budget_show_mode},
                },
                old_day_budget_data => {
                    day_budget => $old_model->{day_budget},
                    day_budget_daily_change_count => $old_model->{day_budget_daily_change_count},
                },
                wallet_day_budget => $wallet_day_budget,
                currency => $model->{currency},
            );

            $item->add_error( $_ ) foreach @{ $vr->get_errors };
            $item->add_warning( $_ ) foreach @{ $vr->get_warnings };
        }
    }

    return;
}

sub _validate_update_campaign_availability {
    my ($self, $rs) = @_;
    my @cids = map {$_->object->{Id}} $rs->list_ok;
    my $checker = $self->get_campaigns_availability_checker(\@cids,
        supported_camp_kind => $self->_get_supported_camp_kind(),
        error_not_found => error_NotFound_Campaign(),
        error_not_supported => error_NotSupported_CampaignType(),
        error_is_archived => error_BadStatus_ArchivedCampaign(),
    );

    foreach my $item ( $rs->list_ok ) {
        if (my $error = $checker->get_error($item->object->{Id})) {
            $item->add_error($error);
        }
    }

    return;
}

sub _validate_update_campaign_rights {
    my ($self, $camp_object) = @_;

    my @cids = map {$_->object->{Id}} $camp_object->list_ok;
    my %write_details = $self->check_campaigns_write_detail(@cids);
    my $camp_no_write = { map { $_ => 1 } $self->grep_no_write_rights(%write_details) };
    my $camp_not_found = { map { $_ => 1 } $self->grep_not_found(%write_details) };
    my $cids_ok = [
        uniq
        grep { not (exists $camp_no_write->{$_} || $camp_not_found->{$_} ) }
        @cids
    ];
    foreach my $campaign ($camp_object->list_ok) {
        my $cid = $campaign->object->{Id};
        $campaign->add_error( error_NotFound_Campaign() ) if $camp_not_found->{$cid};
        $campaign->add_error( error_NoRights_CantWrite() ) if $camp_no_write->{$cid};
    }
    return;
}

sub _validate_duplicate_array_items {
    my ($self, $item) = @_;
    my @fields_with_uniq_items = qw/ExcludedSites BlockedIps PlacementTypes CounterIds NegativeKeywords/;
    my $type = get_type_structure_name($item->object);
    for my $field (@fields_with_uniq_items) {
        if (exists $item->object->{$field}) {
            if($self->is_nil($item->object->{$field})) {
                # хак, выпилить после отказа от XML::Compile
                $item->object->{$field} = undef;
            } else {
                my @items = @{ $item->object->{$field}{Items} };
                if ( $field eq 'ExcludedSites' ) {
                    @items = map { lc $_ } @items;
                }
                $self->_item_check_array_field_for_dups($item, $field, \@items)
            }
        } elsif ($type && exists $item->object->{$type}{$field}) {
            if($self->is_nil( $item->object->{$type}{$field} )) {
                # хак, выпилить после отказа от XML::Compile
                $item->object->{$type}{$field} = undef;
            } else {
                my $array = $item->object->{$type}{$field};
                my @items = $field ne 'PlacementTypes' ? @{ $array->{Items} } : map { $_->{Type} } @$array;
                $self->_item_check_array_field_for_dups($item, $field, \@items)
            }
        }
    }
    return;
}

sub _item_check_array_field_for_dups {
    my ($self, $item, $field, $items) = @_;
    my $error = get_duplicate_error($items||[], $field);
    $item->add_error($error) if $error;

    return;
}

=head2 get_duplicate_error

    Проверяет что в переданном массиве элемент присутствует не более одного раза
    Принимает: массив со значениями и название поля(для сообщения об ошибке)
    Возвращает undef(если все хорошо) иначе объект ошибки Direct::Defect

=cut

sub get_duplicate_error {
    my ($items, $field) = @_;
    my %items_hash;
    $items_hash{$_}++ foreach (@$items);
    my @duplicate = grep {$items_hash{$_} > 1} sort keys %items_hash;
    if (scalar @duplicate) {
        my $duplicate_str = join (',' => @duplicate);
        if (scalar @duplicate == 1) {
            return error_DuplicatedArrayItem(
                iget('Элемент %s присутствует в списке %s более одного раза', $duplicate_str, $field)
            );
        } else {
            return error_DuplicatedArrayItem(
                iget('Элементы %s присутствуют в списке %s более одного раза', $duplicate_str, $field)
            );
        }
    }
    return;
}

=head2 is_valid_day

    Проверяет что передан правильный номер дня недели расписания временного таргетинга
    Принимает: номер дня недели
    Возвращает 1/0

=cut

sub is_valid_day {
    my $day_number = shift;
    if ( is_valid_int($day_number) and $day_number <= 7 and $day_number >= 1 ) {
        return 1;
    }
    return 0;
}

sub _validate_time_targeting {
    my ($self, $item, $type) = @_;
    my $c = $item->object;

    if (exists $c->{TimeTargeting}) {
        my $tt = $c->{TimeTargeting};
        my $on_off_only = exists $c->{CpmBannerCampaign} || $type && $type eq 'cpm_banner' ? 1 : 0;
        if (exists $tt->{Schedule}) {
            $self->_validate_tt_schedule($item, $tt->{Schedule}, on_off_only => $on_off_only);
        }

        if (exists $tt->{HolidaysSchedule} and !$self->is_nil($tt->{HolidaysSchedule})) {
            $self->_validate_holidays_schedule($item, $tt->{HolidaysSchedule}, on_off_only => $on_off_only);
        }
    }
    return;
}

sub _validate_negative_keywords {
    my ($self, $item, $type) = @_;
    my $c = $item->object;

    if (exists $c->{NegativeKeywords} && (exists $c->{CpmBannerCampaign} || $type && $type eq 'cpm_banner')) {
        $item->add_error(error_InconsistentState(iget('Медийная кампания не может содержать минус-фразы')));
    }

    return;
}

sub _validate_tt_schedule {
    my ($self, $item, $schedule, %options) = @_;
    my $str_items = $schedule->{Items};
    my @items = map {[split(',' => $_)]} @$str_items;
    my @days = map { $_->[0] } @items;
    if ( !all { is_valid_day($_) } @days) {
        $item->add_error(error_InvalidField(
            iget('Временной таргетинг должен содержать номера дней недели от 1 до 7')));
    }
    my @correct_days = grep { is_valid_day($_) } @days;
    my @uniq = uniq @correct_days;
    if (scalar @uniq < scalar @correct_days) {
        $item->add_error(error_DuplicatedArrayItem(
            iget('Временной таргетинг содержит повторные дни недели')
        ));
    }
    for my $day (@items){
        my @coefs = @$day[1..$#$day];
        if (scalar @coefs != 24 ) {
            $item->add_error(error_InvalidField(
                iget('Временной таргетинг должен содержать коэффициенты ставок для каждого из 24 часов')
            ));
            last;
        }
    }

    for my $day (@items){
        for my $coef (@$day[1..$#$day]) {
            if (is_valid_int($coef)) {
                if($coef < 0) {
                    $item->add_error(error_InvalidField(
                        iget('Временной таргетинг не может содержать отрицательные коэффициенты ставок')
                    ));
                    last;
                } elsif ($options{on_off_only}) {
                    if ($coef != 0 && $coef != 100) {
                        $item->add_error(error_InvalidField(
                            iget('Коэффициенты ставок временного таргетинга в медийной кампании должны содержать значения 0 или 100')
                        ));
                        last;
                    }
                } elsif ($coef%10 != 0) {
                    $item->add_error(error_InvalidField(
                        iget('Временной таргетинг должен содержать коэффициенты ставок кратные 10')
                    ));
                    last;
                } elsif ($coef > 200) {
                    $item->add_error(error_InvalidField(
                        iget('Коэффициенты ставок временного таргетинга должны содержать значения от %s до %s включительно', 0, 200)
                    ));
                    last;
                }
            } else {
                $item->add_error(error_InvalidField(
                    iget('Временной таргетинг содержит не целочисленные коэффициенты ставок')
                ));
                last;
            }
        }
    }

    return;
}

sub _validate_holidays_schedule {
    my ($self, $item, $schedule, %options) = @_;

    if(exists $schedule->{SuspendOnHolidays} && $schedule->{SuspendOnHolidays} eq "YES") {
        $item->add_error(error_BadParams(iget(
            'HolidaysSchedule не может содержать %s при SuspendOnHolidays в значении YES', 'StartHour'
        ))) if exists $schedule->{StartHour};

        $item->add_error(error_BadParams(iget(
            'HolidaysSchedule не может содержать %s при SuspendOnHolidays в значении YES', 'EndHour'
        ))) if exists $schedule->{EndHour};

        $item->add_error(error_BadParams(iget(
            'HolidaysSchedule не может содержать %s при SuspendOnHolidays в значении YES', 'BidPercent'
        ))) if exists $schedule->{BidPercent};
    } else {
        if(defined $schedule->{BidPercent}) {
            if ($options{on_off_only} && $schedule->{BidPercent} != 100) {
                $item->add_error(error_BadParams(iget('Процент от ставок в выходные дни в медийной кампании должен быть равен 100%')))
            } else {
                my $multiplicity = 10;
                $item->add_error(error_BadParams(iget('Процент от ставок в выходные дни должен быть от 10 до 200 %')))
                    if $schedule->{BidPercent} < 10 || $schedule->{BidPercent} > 200;
                $item->add_error(error_BadParams(iget('Процент от ставок в выходные дни должен быть кратен %s', $multiplicity)))
                    if  $schedule->{BidPercent} % $multiplicity != 0;
            }
        }
        if(!defined $schedule->{StartHour} || !defined $schedule->{EndHour}) {
            $item->add_error(error_ReqField(iget(
                'StartHour и EndHour должны быть заданы, если не задано SuspendOnHolidays = "YES"'
            )));
        } else {
            $item->add_error(error_InvalidField(iget(
                'Значение поля %s должно быть меньше значения поля %s', 'StartHour', 'EndHour'
            ))) if $schedule->{StartHour} >= $schedule->{EndHour};
        }
        $self->_validate_hour_range($item, 'StartHour', $schedule->{StartHour}, 0, 23) if defined $schedule->{StartHour};
        $self->_validate_hour_range($item, 'EndHour', $schedule->{EndHour}, 1, 24) if defined $schedule->{EndHour};
    }
    return;
}

sub _validate_hour_range {
    my ($self, $item, $name, $value, $min, $max) = @_;

    $item->add_error(error_InvalidField(iget(
        '%s должен содержать значения от %s до %s включительно', $name, $min, $max
    ))) if $value < $min || $value > $max;

    return;
}

=head2 _prevalidate_broad_match_goal_id($item, $campaign_data)

    Проверяем идентификатор цели для ДРФ - если значение задано, то оно должно быть
    положительное. Для отрицательного значения добавляем соответствующую ошибку в $item и
    удаляем поле из данных.

=cut

sub _prevalidate_broad_match_goal_id {
    my ($self, $item, $campaign) = @_;

    if (exists $campaign->{broad_match_goal_id} && ($campaign->{broad_match_goal_id} // 0) < 0) {
        delete $campaign->{broad_match_goal_id};
        $item->add_error(error_InvalidField_NotPositive(undef, %{$DEFECT_DESCRIPTION_FIELD_MAP{broad_match_goal_id}}));
    }

    return;
}

=head2 _prevalidate_broad_match_on_add($item, $campaign)

    Проверяем соответствие настроек ДРФ и стратегии при добавлении новой кампании - если включается
    ДРФ, а указанная стратегия предполагает отключение показов на поиске, то добавляем соответствующее
    предупреждение в $item и выключаем ДРФ.

    Так же проверяем идентификатор цели для ДРФ.

=cut

sub _prevalidate_broad_match_on_add {
    my ($self, $item, $campaign) = @_;

    $self->_prevalidate_broad_match_goal_id($item, $campaign);

    my $strategy = $campaign->{strategy};
    if ( defined($strategy) && $strategy->{is_search_stop} && $campaign->{broad_match_flag} ) {
        $campaign->{broad_match_flag} = 0;
        $item->add_warning(warning_BroadMatchReset());
    }

    return;
}

=head2 _prevalidate_broad_match_on_update($item, $new_campaign, $old_campaign)

    Проверяем соответствие настроек ДРФ и стратегии при обновлении кампании - если включается ДРФ,
    а стратегия предполагает отключение показов на поиске, или включаемая стратегия предполагает
    отключение показов, а ДРФ включены, то добавляем соответствующее предупреждение в $item и
    выключаем ДРФ.

    Так же проверяем идентификатор цели для ДРФ.

=cut

sub _prevalidate_broad_match_on_update {
    my ($self, $item, $new_campaign, $old_campaign) = @_;

    $self->_prevalidate_broad_match_goal_id($item, $new_campaign);

    if ($new_campaign->{strategy}->{is_search_stop} && $new_campaign->{broad_match_flag}) {
        $new_campaign->{broad_match_flag} = 0;
        $item->add_warning(warning_BroadMatchReset());
    }

    return;
}

sub _prevalidate_placement_types {
    my ($self, $item, $campaign) = @_;

    if (@{ $item->object->{PlacementTypes} // [] } && !@{ $campaign->{placement_types} // [] }) {
        $item->add_error(error_InvalidField(iget('Запрещено выключать все виды размещения')));
    }

    return;
}

=head2 suspend

    остановка кампаний

=cut

sub suspend {
    my ($self, $request) = @_;

    my $req = API::Service::Campaigns::SuspendRequest->new( $request );

    # check ids per request limit
    my $limits_err = $req->validate( limits => { Ids => $SUSPEND_IDS_LIMIT } );
    return $limits_err if defined $limits_err;

    my $rs = API::Service::Campaigns::ResultSet->new(
        map { +{ Id => $_ } } @{ $req->_selection('Ids') } # $req->selection_ids('Ids')
    );

    # populate $rs with kind of InvalidField error
    $rs->add_error_when(
        error_InvalidField_NotPositive(undef, field => 'Ids'),
        sub { is_valid_id( shift() ) ? 0 : 1 }
    );

    # check duplicates
    $rs->add_warning_for_id_dups(
        warning_Duplicated(
            iget('Id кампании присутствует в запросе более одного раза')
        )
    );

    # check access rights
    my $perm_checks = API::Service::Campaigns::PermissionChecks->new({ can_read => 1, can_write => 1 });
    $self->_populate_rs_with_access_errors( $rs, $perm_checks );

    $self->_fill_rs_with_models(
        $rs,
        fields => {
            campaigns    => [qw/ id campaign_type status_empty status_show status_bs_synced /], # uid
            camp_options => [qw/ stop_time /],
        }
    );

    # check buisness rules
    my $validation_rs = validate_suspend_campaigns([ map { $_->object } $rs->list_ok ]);

    my $generic_errors = $validation_rs->get_generic_errors;
    return $generic_errors if @$generic_errors;

    $self->_populate_rs_with_validation_results( $rs, $validation_rs );

    for my $buisness_logic ( $self->_split_camps_by_buisness_logic( $rs ) ) {
        $buisness_logic->suspend( $self->subclient_uid );
    }

    $self->units_withdraw_for_results( campaign => $rs );

    return { SuspendResults => $rs->prepare_for_xml };
}

=head2 resume

    запуск кампаний

=cut

sub resume {
    my ($self, $request) = @_;

    my $req = API::Service::Campaigns::ResumeRequest->new( $request );
    my $is_cpm_banner_campaign_disabled = Client::ClientFeatures::is_feature_cpm_banner_campaign_disabled_enabled($self->subclient_client_id);

    # check ids per request limit
    my $limits_err = $req->validate( limits => { Ids => $RESUME_IDS_LIMIT } );
    return $limits_err if defined $limits_err;

    my $rs = API::Service::Campaigns::ResultSet->new(
        map { +{ Id => $_ } } @{ $req->_selection('Ids') } # $req->selection_ids('Ids')
    );

    # populate $rs with kind of InvalidField error
    $rs->add_error_when(
        error_InvalidField_NotPositive(undef, field => 'Ids'),
        sub { is_valid_id( shift() ) ? 0 : 1 }
    );

    # check duplicates
    $rs->add_warning_for_id_dups(
        warning_Duplicated(
            iget('Id кампании присутствует в запросе более одного раза')
        )
    );

    # check access rights
    my $perm_checks = API::Service::Campaigns::PermissionChecks->new({ can_read => 1, can_write => 1 });
    $self->_populate_rs_with_access_errors( $rs, $perm_checks );

    $self->_fill_rs_with_models(
        $rs,
        fields => {
            campaigns => [qw/ id campaign_type status_empty status_show status_archived status_bs_synced /], # uid
        }
    );

    # check buisness rules
    my $validation_rs = validate_resume_campaigns([ map { $_->object } $rs->list_ok ], $is_cpm_banner_campaign_disabled);

    my $generic_errors = $validation_rs->get_generic_errors;
    return $generic_errors if @$generic_errors;

    $self->_populate_rs_with_validation_results( $rs, $validation_rs );

    for my $buisness_logic ( $self->_split_camps_by_buisness_logic( $rs ) ) {
        $buisness_logic->resume( $self->subclient_uid );
    }

    $self->units_withdraw_for_results( campaign => $rs );

    return { ResumeResults => $rs->prepare_for_xml };
}

=head2 archive

    архивирование кампаний

    TODO: переписать с использованием новых моделей, методов валидации и бизнес-логики

=cut

sub archive {
    my ($self, $request) = @_;

    my $req = API::Service::Request::Archive->new( $request );

    # check ids per request limit
    my $limits_err = $req->validate( limits => { Ids => $ARCHIVE_IDS_LIMIT } );
    return $limits_err if defined $limits_err;

    my $rs = API::Service::ResultSet::Archive->new( @{ $req->_selection('Ids') } );

    # populate $rs with kind of InvalidField error
    $rs->add_error_when(
        error_InvalidField_NotPositive(undef, field => 'Ids'),
        sub { not is_valid_id( shift ) ? 1 : 0 }
    );

    # check duplicates
    $rs->add_warning_for_id_dups(
        warning_Duplicated(
            iget('Id кампании присутствует в запросе более одного раза')
        )
    );

    # check permissions & bussines rules
    if ( $rs->count_ok ) {
        my @cids = uniq $rs->list_ok_ids;

        my $checker = $self->get_campaigns_availability_checker(\@cids,
            supported_camp_kind => $self->_get_supported_camp_kind(),
            pass_archived => 1,
            error_not_found => error_NotFound_Campaign(),
            error_not_supported => error_NotSupported_CampaignType(),
        );
        my %write_details = $self->check_campaigns_write_detail($checker->get_available_cids());
        my %cant_read     = map { $_ => 1 } $self->grep_not_found( %write_details );
        my %cant_write    = map { $_ => 1 } $self->grep_no_write_rights( %write_details );

        foreach my $item ( $rs->list_ok ) {
            my $cid = $item->object;

            if ( exists $cant_read{ $cid } ) {
                $item->add_error( error_NotFound_Campaign() );
                next;
            }

            if ( exists $cant_write{ $cid } ) {
                $item->add_error( error_NoRights_CantWrite() );
                next;
            }

            if (my $error = $checker->get_error($cid)) {
                $item->add_error($error);
                next;
            }

            if ($checker->get_result($cid)->is_archived) {
                $item->add_warning( warning_AlreadyArchived_Campaign() );
            }
        }
    }

    if ( $rs->count_ok ) {
        my @cids = uniq $rs->list_ok_ids;

        my $arc_results = Wallet::arc_camp(
            undef,
            $self->operator_uid,
            $self->subclient_uid,
            \@cids,
            archived_is_error  => 0,
            dont_stop_on_error => 1,
        );

        # если были ошибки, то $arc_results->{result} = 1,
        # а $arc_results->{errors} будет содержать ошибки
        # с детализацией по кампании
        if ( $arc_results->{result} and exists $arc_results->{errors} ) {

            my $error_by_cid = $arc_results->{errors};

            foreach my $item ( $rs->list_ok ) {
                my $cid = $item->object;

                next unless exists $error_by_cid->{ $cid };

                $item->add_error( error_CantArchive( $error_by_cid->{ $cid } ) );
            }
        }
    }

    $self->units_withdraw_for_results( campaign => $rs );

    return { ArchiveResults => $rs->prepare_for_xml };
}

=head2 unarchive

    разархивирование кампаний

    TODO: переписать с использованием новых моделей, методов валидации и бизнес-логики

=cut

sub unarchive {
    my ($self, $request) = @_;

    my $req = API::Service::Request::Unarchive->new( $request );

    # check ids per request limit
    my $req_limits_err = $req->validate( limits => { Ids => $UNARCHIVE_IDS_LIMIT } );
    return $req_limits_err if defined $req_limits_err;

    my $rs = API::Service::ResultSet::Unarchive->new( @{ $req->_selection('Ids') } );

    # populate $rs with kind of InvalidField error
    $rs->add_error_when(
        error_InvalidField_NotPositive(undef, field => 'Ids'),
        sub { not is_valid_id( shift ) ? 1 : 0 }
    );

    # check duplicates
    $rs->add_warning_for_id_dups(
        warning_Duplicated(
            iget('Id кампании присутствует в запросе более одного раза')
        )
    );

    # check permissions & busines rules
    if ( $rs->count_ok ) {
        my @cids = $rs->list_ok_ids;

        # check permissions
        my $checker = $self->get_campaigns_availability_checker(\@cids,
            supported_camp_kind => $self->_get_supported_camp_kind(),
            check_currency => 1,
            pass_archived => 1,
            error_not_found => error_NotFound_Campaign(),
            error_not_supported => error_NotSupported_CampaignType(),
        );
        my %write_details = $self->check_campaigns_write_detail($checker->get_available_cids());
        my %cant_read     = map { $_ => 1 } $self->grep_not_found( %write_details );
        my %cant_write    = map { $_ => 1 } $self->grep_no_write_rights( %write_details );

        # определим сколько можно разархивировать кампаний до достижения лимита незаархивированных кампаний
        my $cur_camp_stat = Client::get_client_campaign_count( $self->subclient_client_id ); # Client.pm не экспортирует get_client_campaign_count
        my $client_limits = get_client_limits( $self->subclient_client_id );
        my $unarc_camps_left_to_limit_count = $client_limits->{unarc_camp_count_limit} - $cur_camp_stat->{unarc_count};
        $unarc_camps_left_to_limit_count = $unarc_camps_left_to_limit_count > 0 ? $unarc_camps_left_to_limit_count : 0;

        # не даем разархивировать геопродуктовую кампанию без фичи cpm_geoproduct_enabled
        my $geoproduct_campaigns = Campaign::get_geoproduct_campaigns(\@cids);
        my %is_geoproduct_campaign = map { $_ => 1 } @$geoproduct_campaigns;
        my $has_cpm_geoproduct_enabled = Client::ClientFeatures::has_cpm_geoproduct_enabled($self->subclient_client_id);

        # не даем разархивировать охватные продукты и продвижение контента без соответствующих фич
        @cids = grep { !exists $cant_write{$_} } $checker->get_available_cids;
        my $campaign_type_by_cid = get_camp_type_multi(cid => \@cids);
        my @content_promotion_cids = grep { $campaign_type_by_cid->{$_} eq 'content_promotion' } @cids;
        my $content_promotion_types = CampaignTools::mass_get_content_promotion_content_type(\@content_promotion_cids);
        my $has_cpm_deals_allowed = Client::ClientFeatures::has_cpm_deals_allowed_feature($self->subclient_client_id);
        my $has_content_promotion_video_allowed_feature = Client::ClientFeatures::has_content_promotion_video_allowed_feature($self->subclient_client_id);
        my $has_content_promotion_collection_allowed_feature = Client::ClientFeatures::is_feature_content_promotion_collection_enabled($self->subclient_client_id);
        my $is_cpm_banner_campaign_disabled = Client::ClientFeatures::is_feature_cpm_banner_campaign_disabled_enabled($self->subclient_client_id);

        foreach my $item ( $rs->list_ok ) {
            my $cid  = $item->object;

            if ( exists $cant_read{ $cid } ) {
                $item->add_error( error_NotFound_Campaign() );
                next;
            }

            if ( exists $cant_write{ $cid } ) {
                $item->add_error( error_NoRights_CantWrite() );
                next;
            }

            if (my $error = $checker->get_error($cid)) {
                $item->add_error($error);
                next;
            }

            my $camp_check = $checker->get_result($cid);
            if (!$camp_check->is_archived) {
                $item->add_warning( warning_NotArchived_Campaign() );
            }

            # не даём разархивировать кампании, пережившие переход клиента на мультивалютность,
            # оставшиеся при этом в у.е. (они либо были сконвертированы, либо оставлены в архиве)
            if (!$camp_check->is_currency_matched) {
                $item->add_error( error_CantUnarchive(iget('Кампания %d не может быть разархивирована, так как находится в специальном архиве.', $cid)) );
                next;
            }

            # не даем разархивировать геопродуктовую кампанию без фичи cpm_geoproduct_enabled
            # не даем разархивировать охватные продукты и продвижение контента без соответствующих фич
            if ((!$has_cpm_geoproduct_enabled && $is_geoproduct_campaign{$cid})
                || (!$has_cpm_deals_allowed && $campaign_type_by_cid->{$cid} eq 'cpm_deals')
                || ($campaign_type_by_cid->{$cid} eq 'content_promotion' && exists $content_promotion_types->{$cid}
                    && (($content_promotion_types->{$cid} eq 'video' && !$has_content_promotion_video_allowed_feature)
                        || ($content_promotion_types->{$cid} eq 'collection' && !$has_content_promotion_collection_allowed_feature)))) {
                $item->add_error( error_CantUnarchive(iget('Кампания %d не может быть разархивирована, так как находится в специальном архиве.', $cid)) );
                next;
            }

            if (Campaign::is_cpm_campaign($campaign_type_by_cid->{$cid}) && $is_cpm_banner_campaign_disabled) {
                $item->add_error( error_CantUnarchive(iget('Медийная кампания %d не может быть разархивирована.', $cid)) );
            }
            # общее количество кампаний не изменяется, так что можно эту проверку не делать
            if ( $unarc_camps_left_to_limit_count - 1 < 0 ) {
                $item->add_error( error_CantUnarchive( iget('Превышено максимальное количество незаархивированных кампаний — %d', $client_limits->{unarc_camp_count_limit} ) ) );
                next;
            }
            $unarc_camps_left_to_limit_count--;
        }
    }

    foreach my $item ( $rs->list_ok ) {
        my $cid  = $item->object;
        unarc_camp( $self->subclient_uid, $cid );
    }

    $self->units_withdraw_for_results( campaign => $rs );

    return { UnarchiveResults => $rs->prepare_for_xml };
}

=head2 delete

    удаление кампаний

=cut

sub delete {
    my ($self, $request) = @_;

    my $req = API::Service::Campaigns::DeleteRequest->new( $request );

    # check ids per request limit
    my $limits_err = $req->validate( limits => { Ids => $DELETE_IDS_LIMIT } );
    return $limits_err if defined $limits_err;

    my $rs = API::Service::Campaigns::ResultSet->new(
        map { +{ Id => $_ } } @{ $req->_selection('Ids') } # $req->selection_ids('Ids')
    );

    # populate $rs with kind of InvalidField error
    $rs->add_error_when(
        error_InvalidField_NotPositive(undef, field => 'Ids'),
        sub { is_valid_id( shift() ) ? 0 : 1 }
    );

    # check duplicates
    $rs->add_error_for_id_dups(
        error_Duplicated(
            iget('Id кампании присутствует в запросе более одного раза')
        )
    );

    # check access rights
    my $perm_checks = API::Service::Campaigns::PermissionChecks->new({ can_read => 1, can_write => 1, can_delete => 1 });
    $self->_populate_rs_with_access_errors( $rs, $perm_checks );

    $self->_fill_rs_with_models(
        $rs,
        fields => {
            campaigns => [qw/ id campaign_type status_empty converted currency sum sum_to_pay sum_last bs_order_id /], # uid
        },
        roles => 'BsQueue'
    );

    # check buisness rules
    my $validation_rs = validate_delete_campaigns([ map { $_->object } $rs->list_ok ]);

    my $generic_errors = $validation_rs->get_generic_errors;
    return $generic_errors if @$generic_errors;

    $self->_populate_rs_with_validation_results( $rs, $validation_rs );

    my %deleted_cids;
    for my $buisness_logic ( $self->_split_camps_by_buisness_logic( $rs ) ) {
        my $deleted_cids = $buisness_logic->delete( undef, $self->subclient_uid );
        for my $cid ( @$deleted_cids ) {
            $deleted_cids{ $cid } = undef;
        }
    }

    my $count_failed = $rs->count_failed;

    foreach my $item ( $rs->list_ok ) {
        my $camp = $item->object;
        next if exists $deleted_cids{ $camp->id };
        $item->add_error( error_OperationFailed() );
    }

    $self->units_withdraw_for_objects( campaign => $rs->count_ok );

    # here excluded objects with OperationFailed errors
    $self->units_withdraw_for_objects_with_errors( $count_failed );

    return { DeleteResults => $rs->prepare_for_xml };
}

sub _split_camps_by_buisness_logic {
    my ( $self, $rs ) = @_;

    my %seen_camps;
    my %campaings_by_type;
    for my $item ( $rs->list_ok ) {
        my $camp = $item->object;
        next if exists $seen_camps{ $camp->id };
        $seen_camps{ $camp->id }++;
        push @{ $campaings_by_type{ $camp->campaign_type } }, $camp;
    }

    my @blogics;
    for my $camp_type ( keys %campaings_by_type ) {
        push @blogics,
                $self->get_business_logic_class_by_type($camp_type)
                    ->new( items => $campaings_by_type{ $camp_type } );
    }

    return @blogics;
}

sub _populate_rs_with_access_errors {
    my ( $self, $rs, $perm_checks ) = @_;

    # TODO: нужно ли проверять что оператор имеет права на владельцев кампаний?
    # my @uids = uniq map { $_->object->uid } $rs->list_ok;
    # my $is_owner = rbac_mass_is_owner( undef, $self->uid, $uids );

    my @cids = uniq map { $_->object->{Id} } $rs->list_ok;

    my $checker = $self->get_campaigns_availability_checker(\@cids,
        supported_camp_kind => $self->_get_supported_camp_kind(),
        error_not_found => error_NotFound_Campaign(),
        error_not_supported => error_NotSupported_CampaignType(),
        error_is_archived => error_BadStatus_ArchivedCampaign(),
    );
    @cids = $checker->get_available_cids;

    my ( %cant_read, %cant_write, %cant_delete );
    if ( $perm_checks->can_read || $perm_checks->can_write ) {

        my %write_details = $self->check_campaigns_write_detail( @cids );
        %cant_read = map { $_ => 1 } $self->grep_not_found( %write_details );

        if ( $perm_checks->can_write ) {
            %cant_write = map { $_ => 1 } $self->grep_no_write_rights( %write_details );
        }
    }

    if ( $perm_checks->can_delete ) {
        %cant_delete = $self->check_campaigns_delete( @cids );
    }

    foreach my $item ( $rs->list_ok ) {

        my $camp = $item->object;

        if ( exists $cant_read{ $camp->{Id} } ) {
            $item->add_error( error_NotFound_Campaign() );
            next;
        }

        if ( exists $cant_write{ $camp->{Id} } ) {
            $item->add_error( error_NoRights_CantWrite() );
            next;
        }

        if ( exists $cant_delete{ $camp->{Id} } ) {
            $item->add_error( error_NoRights_CantDelete() );
            next;
        }

        if (my $error = $checker->get_error($camp->{Id})) {
            $item->add_error($error);
        }

    }

    return;
}

sub _fill_rs_with_models {
    my ( $self, $rs, %opts ) = @_;

    my ( $fields, $roles ) = delete @opts{qw/ fields roles /};
    croak 'unsupported options given' if %opts;

    # fetch camps
    my $camps = Direct::Campaigns->get(
        [ uniq $rs->list_ok_ids ],
        campaign_type => $self->get_api_allowed_camp_types(),
        ( defined $fields ? ( fields     => $fields ) : () ),
        ( defined $roles  ? ( with_roles => $roles  ) : () ),
    );

    my %camps_by_id = map { $_->id => $_ } @{$camps->items};

    for my $item ( $rs->list_ok ) {
        if ( exists( $camps_by_id{ $rs->_id_from_object( $item->object ) } ) ) {
            $item->object( $camps_by_id{ $rs->_id_from_object( $item->object ) } );
        }
    }

    return $rs;
}

sub _populate_rs_with_validation_results {
    my ( $self, $rs, $validation_rs ) = @_;

    my @items = $rs->list_ok();
    my $validation_results = $validation_rs->get_objects_results;

    # NB: should assertion on arrays sizes equality be added here?
    # carp "WTF - arrays have different sizes??!" unless scalar( $rs->list_ok ) == scalar( @$validation_results )

    my $iterator = each_array( @items, @$validation_results );
    while ( my ( $item, $validation_result ) = $iterator->() ) {
        $item->add_error( $_ ) foreach @{ $validation_result->get_errors };
        $item->add_warning( $_ ) foreach @{ $validation_result->get_warnings }
    }

    return;
}

=head2 get_api_allowed_camp_types

    Получение списка разрешённых для редактирования кампаний

=cut

sub get_api_allowed_camp_types {
    my ($self) = @_;

    # # NB: кажется, что управлять гео кампаниями через апи имеют право менеджеры из Я.Справочника
    # my $camps_kind = get_camp_kind_types(
    #                         $self->_authorization->operator_user->api_geo_allowed eq 'Yes'
    #                             ? 'api5_edit_geo'
    #                             : 'api5_edit'
    return get_camp_kind_types($self->_get_supported_camp_kind());
}

sub _get_supported_camp_kind {
    return SUPPORTED_CAMP_KIND;
}

1;
