package API::Service::Campaigns::ConvertMap;

use strict;
use warnings;
use utf8;

use Mouse;

use API::Service::Campaigns::Strategies qw/
    is_autobudget
    strategy_sets_to_internal
    strategy_sets_to_external/;

use API::Converter::ConvertSubs qw/
    convert_to_long
    convert_to_money
    convert_string_comma_to_array
    convert_array_to_string_comma
    convert_string_space_to_array
    convert_array_to_string_space
    convert_from_yesno_to_bool
    convert_nillable_array
    convert_items_element_to_array
    uppercase/;

use API::Service::Campaigns::Types qw/
    convert_type convert_types
    convert_type_to_external
    convert_attribution_model_to_external
    convert_video_target_to_external
/;

use base qw/API::Converter::ConvertMap/;

my %to_internal_common = (
    Id => 'cid',
    Name => 'name',
    Type => [
        type => \&convert_type
    ],
    Notification => {
        SmsSettings => \&API::Service::Campaigns::ConvertSubs::convert_sms_notification_to_internal,
        EmailSettings => {
            Email => 'email',
            CheckPositionInterval => 'warnPlaceInterval',
            WarningBalance => 'money_warning_value',
            SendAccountNews => ['sendAccNews' => sub {shift eq "YES" ? 1 : undef}],
            SendWarnings => ['sendWarn' => sub {shift eq "YES" ? 1 : undef}],
        }
    },
    TimeTargeting => \&API::Service::Campaigns::ConvertSubs::convert_time_targetings,
    DailyBudget => {
        Amount => [ day_budget => \&convert_to_money ],
        Mode => [ day_budget_show_mode =>
                    \&API::Service::Campaigns::ConvertSubs::convert_show_mode
                ],
    },
    TimeZone => [ 'timezone_id' => \&API::Service::Campaigns::ConvertSubs::convert_time_zone ],
    NegativeKeywords => [
        campaign_minus_words => \&convert_items_element_to_array
    ],
    BlockedIps => [
        disabledIps => \&convert_array_to_string_comma
    ],
    ExcludedSites => [
        DontShow => \&convert_array_to_string_comma
    ],
    PlacementTypes => 'placement_types', # preprocessed
    Settings => \&API::Service::Campaigns::ConvertSubs::convert_settings,
    BiddingStrategy => [ strategy => \&API::Service::Campaigns::Strategies::strategy_sets_to_internal ],
    StrategyId => 'strategy_id',
    RelevantKeywords => \&API::Service::Campaigns::ConvertSubs::convert_broad_match,
    FrequencyCap => \&API::Service::Campaigns::ConvertSubs::convert_frequency_cap_to_internal,
    AttributionModel => [
        attribution_model => \&API::Service::Campaigns::Types::convert_attribution_model
    ],
    VideoTarget => [
        eshows_video_type => \&API::Service::Campaigns::Types::convert_video_target
    ]
);

my %to_internal_add_update_common = (
    %to_internal_common,
    ClientInfo => 'fio',
    StartDate => 'start_time',
    EndDate => 'finish_time',
    CounterIds => [
        metrika_counters => \&convert_array_to_string_space
    ],
    CounterId => 'metrika_counters',
);

=head2 build_to_internal_map

    Набор правил для преобразования полей из внешнего представления во внутреннее

=cut

sub build_to_internal_map {
    my $self = shift;
    my $map = {
        get => {
            SelectionCriteria => [ SelectionCriteria => {
                    Ids => 'c.cid',
                    Types => [ 'c.type' => \&convert_types ],
                }
            ],
            %to_internal_common,
            StartDate => 'start_date',
            EndDate => 'finish_date',
            SourceId => 'source_id',
            Currency => 'currency',
            ClientInfo => 'user_fio',
            RepresentedBy => {
                Manager => 'agency_name',
                Agency => 'manager_fio',
            },
            Statistics => {
                Impressions => 'shows',
                Clicks => 'clicks',
            },
            CounterIds => 'metrika_counters',
            CounterId => 'metrika_counters', # performance
            Status => 'status',
            State => 'state',
            PriorityGoals => 'meaningful_goals',
        },
        add => {
            %to_internal_add_update_common,
            PriorityGoals => \&API::Service::Campaigns::ConvertSubs::convert_meaningful_goals_to_internal,
        },
        update => {
            %to_internal_add_update_common,
            # для PriorityGoals пока без учёта операции
            PriorityGoals => \&API::Service::Campaigns::ConvertSubs::convert_meaningful_goals_to_internal,
        },
    };
    return $map->{$self->method};
}

=head2 build_to_external_map

    Набор правил для преобразования полей из внутреннего представления во внешнее

=cut

sub build_to_external_map {
    my $self = shift;
    my $map = {
        get => {
            cid => 'Id',
            name => 'Name',
            type => [
                Type => \&convert_type_to_external
            ],
            start_date => 'StartDate',
            finish_date => 'EndDate',
            source_id => 'SourceId',
            currency => 'Currency',
            FIO => 'ClientInfo',
            agency_name => ['RepresentedBy' => 'Agency'],
            manager_fio => ['RepresentedBy' => 'Manager'],
            time_zone => {
                timezone => "TimeZone",
            },
            timeTarget => [
                TimeTargeting => \&API::Service::Campaigns::ConvertSubs::convert_time_targetings_to_external
            ],
            shows => [ Statistics => 'Impressions'],
            clicks => [ Statistics => 'Clicks'],
            disabledIps => [
                BlockedIps => \&convert_string_comma_to_array
            ],
            DontShow => [
                ExcludedSites => \&convert_string_comma_to_array
            ],
            metrika_counter => 'CounterId',
            metrika_counters => [
                CounterIds => \&convert_nillable_array
            ],
            minus_words => [
                NegativeKeywords => \&convert_nillable_array
            ],
            day_budget => [ 'DailyBudget', 'Amount' => \&convert_to_long ],
            day_budget_show_mode => [
                'DailyBudget',
                'Mode' => \&API::Service::Campaigns::ConvertSubs::convert_show_mode_to_external
            ],
            strategy => [
                BiddingStrategy => \&API::Service::Campaigns::Strategies::strategy_sets_to_external
            ],
            strategy_id => 'StrategyId',
            status => 'Status',
            state => 'State',
            status_payment => 'StatusPayment',
            email => ['Notification', 'EmailSettings', 'Email'],
            warnPlaceInterval => ['Notification', 'EmailSettings', 'CheckPositionInterval'],
            money_warning_value => ['Notification', 'EmailSettings', 'WarningBalance'],
            sendAccNews => [
                'Notification', 'EmailSettings',
                'SendAccountNews' => \&uppercase
            ],
            sendWarn => [
                'Notification', 'EmailSettings',
                'SendWarnings' => \&uppercase
            ],
            sms_flags => [
                'Notification', 'SmsSettings',
                'Events' => \&API::Service::Campaigns::ConvertSubs::convert_events_to_external
            ],
            sms_time => [
                'Notification',
                'SmsSettings' => \&API::Service::Campaigns::ConvertSubs::convert_sms_time_to_external
            ],
            finance => 'Funds',
            settings => 'Settings',
            status_clarification => 'StatusClarification',
            placement_types => 'PlacementTypes',
            broad_match => [RelevantKeywords =>
                \&API::Service::Campaigns::ConvertSubs::convert_broad_match_to_external
            ],
            frequency_cap => [
                FrequencyCap => \&API::Service::Campaigns::ConvertSubs::convert_frequency_cap_to_external
            ],
            meaningful_goals => [
                PriorityGoals => \&API::Service::Campaigns::ConvertSubs::convert_meaningful_goals_to_external,
            ],
            attribution_model => [
                'AttributionModel' => \&API::Service::Campaigns::Types::convert_attribution_model_to_external
            ],
            eshows_video_type => [
                VideoTarget => \&API::Service::Campaigns::Types::convert_video_target_to_external
            ]
        },
        add => {

        },
        update => {

        },
    };
    return $map->{$self->method};
}

__PACKAGE__->meta->make_immutable();

1;

__END__
