package API::Service::Campaigns::ConvertSubs;

use strict;
use warnings;
use utf8;

use List::MoreUtils qw/pairwise uniq any/;
use Carp;

use TimeTarget;
use Tools;

use API::Converter::ConvertSubs qw/convert_to_long convert_from_yesno_to_bool convert_to_money/;

use Exporter qw/import/;
our @EXPORT_OK = qw/convert_finance_to_external
                    convert_show_mode_to_external
                    convert_show_mode
                    convert_time_targetings
                    convert_time_targetings_to_external
                    convert_events_to_external
                    convert_sms_time_to_external
                    convert_sms_time_string_to_struct
                    convert_settings_to_external
                    convert_placement_types
                    convert_placement_types_to_external
                    where_from_model
/;

my %show_mode_map = (
    default => 'STANDARD',
    stretched => 'DISTRIBUTED'
);

=head2 convert_show_mode_to_external

    Преобразует day_budget_show_mode к внешнему виду API

=cut

sub convert_show_mode_to_external {
    my $value = shift;
    return $show_mode_map{$value};
}

=head2 convert_show_mode

    Преобразует DailyBudget->Mode к внутреннему виду

=cut

sub convert_show_mode {
    my %mapper = reverse %show_mode_map;
    return $mapper{$_[0]};
}

my %events_map = (
    active_orders_money_warning_sms => 'MONEY_OUT',
    active_orders_money_out_sms => 'MONEY_OUT',
    notify_order_money_in_sms => 'MONEY_IN',
    moderate_result_sms => 'MODERATION',
    notify_metrica_control_sms => 'MONITORING',
    camp_finished_sms => 'FINISHED'
);

=head2 convert_events_to_external

    Преобразует Notification->SmsSettings->Events к внешнему виду API
    Принимает: строку вида "notify_metrica_control_sms,moderate_result_sms..."
    Возвращает: массив [ (MONITORING|MODERATION|MONEY_IN|MONEY_OUT\FINISHED) ] 

=cut

sub convert_events_to_external {
    my $events = shift;
    my @events = split(',', $events);
    my @result_events;
    for my $event (@events) {
        push @result_events, $events_map{$event}
            if exists $events_map{$event};
    }
    return [uniq @result_events];
}

=head2 convert_sms_time_to_external

    Преобразует sms_time к внешнему виду API
    Принимает: строку sms_time (HH:MM:HH:MM)
    Возвращает: структуру {
        TimeFrom: 'HH:MM',
        TimeTo: 'HH:MM'
    }

=cut

sub convert_sms_time_to_external {
    my $time = shift;
    my %result;
    if($time =~ /^(\d+\:\d+)\:(\d+\:\d+)$/){
        $result{TimeFrom} = $1;
        $result{TimeTo} = $2;
    }
    return \%result;
}

=head2 convert_time_zone

    Преобразует название временной зоны в её id

=cut

sub convert_time_zone {
    my $value = shift;

    ## Функция get_timezone вытаскивает нужный timezone_id из таблицы geo_timezones,
    ## где соответствие timezone => timezone_id неоднозначное (для timezone может быть
    ## несколько разных timezone_id). Важно, чтобы для таких timezone всегда выбирался
    ## один и тот же timezone_id. Для большинства таких случаев get_timezone выбирает
    ## timezone_id из России или наименьший или наименьший из тех, кто в России,
    ## это досттаочно предсказуемо. В этой таблице случаи, когда принципиально важно,
    ## чтобы в нашей стране присутствия страна по часовому поясу выбиралась конкретная.
    my $timezone_name = {
        'Europe/Moscow' => 'Москва',
        'Europe/Kiev'   => 'Украина',
    }->{$value};

    if ($timezone_name) {
        if ( my $tz = TimeTarget::get_timezone( timezone => $value, name => $timezone_name ) ) {
            return $tz->{timezone_id};
        }

        ## Но если не свезло, не падаем, есть путь отхода: выбрать строчку по timezone,
        ## у которой может быть другой name.
        ## Но пусть кто-нибудь это починит, пожалуйста.
        ## Если в логах такое сообщение, скорее всего, что-то поменялось в таблице geo_timezones.
        ## Надо поправить либо данные в таблице, либо код в этой процедуре, чтобы они друг другу
        ## соответствовали.
        warn "Couldn't find timezone=$value, name=$timezone_name";
    }

    my $tz = TimeTarget::get_timezone( timezone => $value );
    return $tz->{timezone_id};
}

=head2 convert_sms_time_string_to_struct

    Преобразует sms_time из строки (HH:MM:HH:MM) в структуру

=cut

sub convert_sms_time_string_to_struct {
    my $sms_time_string = shift;
    my ($sms_hour_from, $sms_min_from, $sms_hour_to, $sms_min_to) = Tools::string2sms_time($sms_time_string);
    return {
        sms_time_hour_from => $sms_hour_from + 0,
        sms_time_hour_to => $sms_hour_to + 0,
        sms_time_min_from => $sms_min_from + 0,
        sms_time_min_to => $sms_min_to + 0
    };
}

=head2 convert_sms_notification_to_internal

    Преобразует Notification->SmsSettings к внутреннему виду

=cut

sub convert_sms_notification_to_internal {
    my $sms_settings = shift;
    my %result;
    # Events если передано обновляем целиком,
    # если не передано получаем из БД
    my %events = map {$_ => 1} @{ $sms_settings->{Events} // [] };
    if (%events) {
        for my $inner_event (keys %events_map) {
            my $outer_event = $events_map{$inner_event};
            $result{$inner_event} = $events{$outer_event} ? 1 : undef; # явно задаем undef, чтобы затереть не переданные значения
        }
    }
    my $time_from = $sms_settings->{TimeFrom} ? $sms_settings->{TimeFrom} : '00:00';
    my $time_to = $sms_settings->{TimeTo} ? $sms_settings->{TimeTo} : '00:00';

    ($result{sms_time_hour_from}, $result{sms_time_min_from}) = split(':', $time_from);
    ($result{sms_time_hour_to}, $result{sms_time_min_to}) = split(':', $time_to);

    return \%result;
}

=head2 convert_time_targetings_to_external

    Преобразует TimeTargeting к внешнему виду API

=cut

sub convert_time_targetings_to_external {
    my $value = shift;
    my $tt = TimeTarget::parse_timetarget($value);
    my @schedule;
    my $hours = TimeTarget::hours_hash($tt->{timeTarget});
    for my $day_of_week (1..7){
        my @day_line;
        push @day_line, $day_of_week;
        for my $hour (0..23){
            my $persent = $hours->{$day_of_week}{$hour};
            $day_line[$hour+1] = $persent ? $persent : 0;
        }
        push @schedule, join(',' => @day_line);
    }
    my %holidays_schedule;
    if ($tt->{time_target_holiday}) {
        $holidays_schedule{SuspendOnHolidays} =
            defined $tt->{time_target_holiday_dont_show} &&
            $tt->{time_target_holiday_dont_show} == 1 ? "YES" : "NO";
        if (!$tt->{time_target_holiday_dont_show}) {
            $holidays_schedule{BidPercent} = $tt->{time_target_holiday_coef} // 100;
            $holidays_schedule{StartHour} = $tt->{time_target_holiday_from};
            $holidays_schedule{EndHour} = $tt->{time_target_holiday_to};
        }
    }
    my %result;
    $result{ConsiderWorkingWeekends} = defined $tt->{time_target_working_holiday} && $tt->{time_target_working_holiday} ? 'YES' : 'NO';
    $result{Schedule} = { Items => \@schedule };
    if (%holidays_schedule) {
        $result{HolidaysSchedule} = \%holidays_schedule;
    } else {
        $result{HolidaysSchedule} = undef;
    }
    return \%result;
}

=head2 convert_time_targetings

    Преобразует TimeTargeting к внутреннему виду

=cut

sub convert_time_targetings {
    my $tt = shift;
    my %result;
    $result{time_target_preset} = 'other';
    if (exists $tt->{HolidaysSchedule}) { # так можно делать только потому, что мы сконвертированли NIL в undef в контроллере
        my $hs = $tt->{HolidaysSchedule};
        if (defined $tt->{HolidaysSchedule}) {
            $result{time_target_holiday} = 1;
            if ($hs->{SuspendOnHolidays} eq 'YES') {
                $result{time_target_holiday_dont_show} = 1;
            } else {
                $result{time_target_holiday_dont_show} = 0;
                $result{time_target_holiday_coef} = $hs->{BidPercent};
                $result{time_target_holiday_from} = $hs->{StartHour};
                $result{time_target_holiday_to} = $hs->{EndHour};
            }
        } else {
            $result{time_target_holiday} = 0;
        }
    } 
    $result{time_target_working_holiday} = convert_from_yesno_to_bool($tt->{ConsiderWorkingWeekends}) if defined $tt->{ConsiderWorkingWeekends};

    $result{timeTarget} = '';

    # значение самого расписания находится начиная с третьего символа, т.к. номер дня отделён запятой
    my %shedule_by_day = map { substr($_, 0, 1) => substr($_, 2) } @{ $tt->{Schedule}{Items} || [] };
    for my $day (1..7) {
        my $item = $shedule_by_day{ $day };

        my @values;
        if ($item) {
            @values = split(',' => $item);
        } else {
            @values = (100) x 24;
        }
        my @hours = (0..$#values);

        ## DollarAB не позволяет добавить свои функции в %sorters, pairwise там нет
        ## https://github.com/Grinnz/Perl-Critic-Freenode/issues/21
        ## no critic (Freenode::DollarAB)
        my @lettes = pairwise { our ($a, $b); $b > 0 ? TimeTarget::hour2letter($a)._bidcoef_letter($b) : '' } @hours, @values;
        $result{timeTarget} .= $day.join("", sort @lettes);
    }
    return \%result;
}

sub _bidcoef_letter {
    my $coef = shift;
    return '' if ($coef == 100);
    return TimeTarget::coef2letter($coef);
}

=head2 convert_finance_to_external

    Преобразует  sums_uni в структуру FundsParam

=cut

sub convert_finance_to_external {
    my ($model, $client_discount, $spend_with_nds) = @_;
    my $money = $model->{sums_uni};
    my %result;
    $result{Mode} = convert_mode_to_external($model->{wallet_is_enabled});
    if ($model->{wallet_is_enabled}) {
        my $total = $model->{sum} - $model->{sum_spent};
        # по задумке, положительный $total на кампании под ОС может быть
        # только в случае возврата по антифроду
        $total = $total > 0 ? $total : 0;
        my %wallet_fund = (
            Spend => convert_to_long(($spend_with_nds && exists $money->{sum_spent_include_nds}) ? $money->{sum_spent_include_nds} : $money->{sum_spent}),
            Refund => convert_to_long($total),
        );
        $result{SharedAccountFunds} = \%wallet_fund;
    } else {
        my %campaign_fund = (
            Sum => convert_to_long(($spend_with_nds && exists $money->{sum_include_nds}) ? $money->{sum_include_nds} : $money->{sum}),
            Balance => convert_to_long($money->{total}),
            BalanceBonus => convert_to_long($money->{bonus}),
            SumAvailableForTransfer => convert_to_long($money->{available_for_transfer}),
        );
        $result{CampaignFunds} = \%campaign_fund;
    }
    return \%result;
}

=head2 convert_mode_to_external

    Получает на основе данных о подключенности общего счета FundsParam->Mode

=cut

sub convert_mode_to_external {
    my $wallet_is_enabled = shift;
    if ($wallet_is_enabled) {
        return 'SHARED_ACCOUNT_FUNDS';
    } else {
        return 'CAMPAIGN_FUNDS';
    }
    
}

my %settings_map = (
    ADD_OPENSTAT_TAG => 'statusOpenStat',
    ADD_METRICA_TAG => 'status_click_track',
    ADD_TO_FAVORITES => 'is_favorite',
    ENABLE_SITE_MONITORING => 'statusMetricaControl',
    EXCLUDE_PAUSED_COMPETING_ADS => 'fairAuction',
    REQUIRE_SERVICING => 'is_servicing',
    SHARED_ACCOUNT_ENABLED => 'wallet_is_enabled',
    ENABLE_EXTENDED_AD_TITLE => 'no_title_substitute',
    ENABLE_AREA_OF_INTEREST_TARGETING => 'no_extended_geotargeting',
    MAINTAIN_NETWORK_CPC => 'enable_cpc_hold',
    ENABLE_COMPANY_INFO => 'hide_permalink_info',
    TURBO_APPS_ENABLED => 'has_turbo_app',
    CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED => 'is_order_phrase_length_precedence_enabled'
);

my @BOOL_FIELDS = qw/
    ADD_METRICA_TAG ADD_TO_FAVORITES ENABLE_SITE_MONITORING
    REQUIRE_SERVICING MAINTAIN_NETWORK_CPC TURBO_APPS_ENABLED CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED
/;

my @INVERSED_FIELDS = qw/
    ENABLE_AREA_OF_INTEREST_TARGETING ENABLE_EXTENDED_AD_TITLE ENABLE_COMPANY_INFO
/;

my @OPTS_FIELDS = qw/
    ENABLE_AREA_OF_INTEREST_TARGETING ENABLE_EXTENDED_AD_TITLE MAINTAIN_NETWORK_CPC ENABLE_COMPANY_INFO
    TURBO_APPS_ENABLED CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED
/;

=head2 convert_settings

    Преобразует настройки кампании к внутреннему виду

=cut

sub convert_settings {
    my $settings = shift;

    my %result;
    for my $setting (@$settings) {

        my $value  = $setting->{Value};
        my $option = $setting->{Option};

        next if $option eq 'ENABLE_RELATED_KEYWORDS'; # DIRECT-63109
        next if $option eq 'ENABLE_AUTOFOCUS'; # DIRECT-68434
        next if $option eq 'ENABLE_BEHAVIORAL_TARGETING'; # DIRECT-69528

        my $converted_val;
        if (any { $_ eq $option } @BOOL_FIELDS) {
            $converted_val = convert_from_yesno_to_bool($value);
        } elsif (any { $_ eq $option } @INVERSED_FIELDS) {
            $converted_val = ! convert_from_yesno_to_bool($value);
        } elsif ($option eq 'EXCLUDE_PAUSED_COMPETING_ADS') {
            $converted_val = convert_from_yesno_to_bool($value) ? 1 : undef;
        } else {
            $converted_val = ucfirst lc $value;
        }

        my $internal_option = $settings_map{ $option };
        if (any { $_ eq $option } @OPTS_FIELDS) {
            $result{opts}->{ $internal_option } = $converted_val;
        } else {
            $result{ $internal_option } = $converted_val;
        }
    }

    return \%result;
}

=head2 convert_settings_to_external

    Преобразует настройки кампании к внешнему виду API

=cut

sub convert_settings_to_external {
    my ($model) = @_;
    my @result;
    push @result, { Option => 'ADD_TO_FAVORITES',
                    Value => $model->{is_favorite} ? 'YES' : 'NO' };
    push @result, { Option => 'REQUIRE_SERVICING',
                    Value => $model->{is_servicing} ? 'YES' : 'NO' };
    push @result, { Option => 'SHARED_ACCOUNT_ENABLED',
                    Value => $model->{wallet_is_enabled} ? 'YES' : 'NO' };
    push @result, { Option => 'DAILY_BUDGET_ALLOWED', Value => 'YES' };
    if (!$model->{is_network_default_with_search_conversion_strategy}
        && ($model->{type} eq 'text' || $model->{type} eq 'mobile_content')) {
        push @result, { Option => 'MAINTAIN_NETWORK_CPC',
                        Value => $model->{opts}{enable_cpc_hold} ? 'YES' : 'NO' };
    }
    if ($model->{type} eq 'text' || $model->{type} eq 'dynamic' || $model->{type} eq 'cpm_banner') {
        push @result, { Option => 'ENABLE_SITE_MONITORING', Value => uc $model->{statusMetricaControl} };
        push @result, { Option => 'ADD_METRICA_TAG',
                        Value => $model->{status_click_track} ? 'YES' : 'NO' };
        push @result, { Option => 'ADD_OPENSTAT_TAG', Value => uc $model->{statusOpenStat} };
    }
    if ($model->{type} eq 'text' || $model->{type} eq 'dynamic') {
        push @result, { Option => 'ENABLE_EXTENDED_AD_TITLE',
                        Value => $model->{opts}{no_title_substitute} ? 'NO' : 'YES' };
        push @result, { Option => 'ENABLE_COMPANY_INFO',
                        Value => $model->{opts}{hide_permalink_info} ? 'NO' : 'YES' };
    }

    if ($model->{type} eq 'text' || $model->{type} eq 'dynamic' || $model->{type} eq 'mobile_content' ) {
        push @result, { Option => 'CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED',
            Value => $model->{opts}{is_order_phrase_length_precedence_enabled} ? 'YES' : 'NO' };
    }

    if ($model->{type} eq 'text') {
        push @result, { Option => 'EXCLUDE_PAUSED_COMPETING_ADS',
                        Value => $model->{fairAuction} ? 'YES' : 'NO' };
    }

    push @result, { Option => 'ENABLE_AREA_OF_INTEREST_TARGETING',
                    Value => $model->{opts}{no_extended_geotargeting} ? 'NO' : 'YES' };

    return \@result;
}

my %PLACEMENT_TYPE_TO_EXTERNAL = (
    adv_gallery => 'PRODUCT_GALLERY',
    search_page => 'SEARCH_RESULTS'
);

=head2 convert_placement_types_to_external

    Преобразует настройки видов размещения к внешнему виду API

=cut

sub convert_placement_types_to_external {
    my $placement_types = shift;
    if (!@{ $placement_types // [] }) {
        $placement_types = [@Direct::Model::Campaign::PLACEMENT_TYPES];
    }
    my %lookup = map { $_ => undef } @$placement_types;
    return {
        Items => [map { {
            Type => $PLACEMENT_TYPE_TO_EXTERNAL{$_} // (die "Unknown internal placement type: $_"),
            Value => exists $lookup{$_} ? 'YES' : 'NO'
        } } @Direct::Model::Campaign::PLACEMENT_TYPES]
    };
}

my %PLACEMENT_TYPE_FROM_EXTERNAL = reverse %PLACEMENT_TYPE_TO_EXTERNAL;

=head2 convert_placement_types

    Преобразует настройки видов размещения к внутреннему виду

=cut

sub convert_placement_types {
    my ($placement_types, $old_placement_types) = @_;
    if (!@{ $old_placement_types // [] }) {
        $old_placement_types = [@Direct::Model::Campaign::PLACEMENT_TYPES];
    }
    my %values = map { $_ => 1 } @$old_placement_types;
    foreach my $item (@{ $placement_types // [] }) {
        my $type = $PLACEMENT_TYPE_FROM_EXTERNAL{$item->{Type}} // die "Unknown external placement type: $item->{Type}";
        my $value = convert_from_yesno_to_bool($item->{Value});
        $values{$type} = $value;
    }
    return [grep { $values{$_} } keys(%values)];
}

=head2 convert_broad_match_to_external

    Преобразует настройки дополнительных релевантных фраз к внешнему виду API

=cut

sub convert_broad_match_to_external {
    my $broad_match = shift;
    if ($broad_match->{flag} eq 'Yes') {
        return {
            BudgetPercent => $broad_match->{limit},
            OptimizeGoalId => $broad_match->{goal_id}
        };
    }
    return;
}

=head2 convert_broad_match

    Преобразует настройки дополнительных релевантных фраз к внутреннему виду

=cut

sub convert_broad_match {
    my $broad_match = shift;
    if ($broad_match) {
        return {
            broad_match_flag => 1,
            exists $broad_match->{BudgetPercent} ? (broad_match_limit => $broad_match->{BudgetPercent}) : (),
            exists $broad_match->{OptimizeGoalId} ? (broad_match_goal_id => $broad_match->{OptimizeGoalId}) : ()
        };
    } else {
        return {
            broad_match_flag => 0,
            broad_match_limit => 0,
        };
    }
}

=head2 convert_frequency_cap_to_external

    Преобразует настройки ограничения частоты показов к внешнему виду API

=cut

sub convert_frequency_cap_to_external {
    my ($frequency_cap) = @_;

    if ($frequency_cap->{rf}) {
        return {
            Impressions => $frequency_cap->{rf},
            PeriodDays => $frequency_cap->{rfReset} || undef,
        };
    }

    return;
}


=head2 convert_frequency_cap_to_internal

    Преобразует настройки ограничения частоты показов во внутреннее представление

=cut

sub convert_frequency_cap_to_internal {
    my ($frequency_cap) = @_;

    return {
        rf => $frequency_cap->{Impressions},
        rfReset => $frequency_cap->{PeriodDays},
    };
}


=head2 convert_meaningful_goals_to_external

Преобразует ключевые цели кампании к внешнему виду API

=cut

sub convert_meaningful_goals_to_external {
    my ($meaningful_goals) = @_;
    return if !$meaningful_goals || !@$meaningful_goals;
    my @mf_external = map {_meaningful_goal_to_external($_)} @$meaningful_goals;
    return {
        Items => \@mf_external,
    };
}

sub _meaningful_goal_to_external {
    my ($mg) = @_;
    my $is_metrika_source_of_value = defined $mg->{is_metrika_source_of_value} && $mg->{is_metrika_source_of_value} ? 'YES' : 'NO';
    my %result = (
        GoalId => $mg->{goal_id},
        Value => convert_to_long($mg->{value}),
        IsMetrikaSourceOfValue => $is_metrika_source_of_value
    );
    return \%result;
}


=head2 convert_meaningful_goals_to_internal

Преобразует ключевые цели кампании во внутреннее представление

=cut

sub convert_meaningful_goals_to_internal {
    my ($meaningful_goals) = @_;
    return { meaningful_goals => undef }  if !$meaningful_goals;

    my @mg_internal = map {_meaningful_goal_to_internal($_)} @{$meaningful_goals->{Items}};
    return { meaningful_goals => \@mg_internal };
}
sub _meaningful_goal_to_internal {
    my ($mg) = @_;
    my %result = (
        goal_id                => $mg->{GoalId},
        value                  => convert_to_money($mg->{Value}),
    );

    if (defined $mg->{IsMetrikaSourceOfValue}) {
       $result{is_metrika_source_of_value} = $mg->{IsMetrikaSourceOfValue} eq 'YES' ? 1 : 0;
    }

    return \%result;
}


=head2 where_from_model

    Добавляет к структуре полученной из SelectionCriteria префиксы таблиц,
    чтоб получилась полноценная структура where для Yandex::DBTools

=cut

sub where_from_model($) {
    my $model = shift;
    my %where;
    for my $field (keys %$model) {
        if ($field =~ /finish_time/ && !defined $model->{$field}) {
            $model->{$field} = '0000-00-00'
        }
        
        $where{"c.$field"} = $model->{$field};
    }
   
    return \%where;
}


1;

