package API::Service::Campaigns::Statuses;

use strict;
use warnings;
use utf8;

=pod

    $Id$

=head1 NAME

    API::Service::Campaigns::Statuses

=head1 SYNOPSIS

=head1 DESCRIPTION
    Модуль содержит методы преднозначенные для преобразования статусов и состояний
    из внешнего формата API в where базы данных

=cut

use Exporter qw/import/;
our @EXPORT_OK = qw/statuses_where
                    states_where
                    status_payment_where
                    status_from_model
                    state_from_model
                    status_payment_from_model
                    merge_statuses_where
                    calculate_status_clarification/;

use Yandex::I18n qw/iget/;
use Yandex::Clone qw/yclone/;

use Currencies;

use API::Service::Statuses qw/get_field_value_depends_on_rules/;
use API::Service::Campaigns::ConvertSubs qw/where_from_model/;

=head2 $RULES

    Ссылка на массив описывающий статусы кампании,
    вида [ [status1 => $conditions1], .., [statusN => $conditionsN] ]

    Формат условий $conditions => { field => $values, .. } где field =
    moderation_status|post_moderation_status, $values значение поля, либо ссылка на массив
    допустимых для статуса значений

=cut


my $STATUS_RULES = [
    [ DRAFT       => { statusModerate => 'New' } ],
    [ MODERATION  => { statusModerate => ['Sent', 'Ready'] } ],
    [ REJECTED    => { statusModerate => 'No'} ],
    [ ACCEPTED    => { statusModerate => 'Yes' } ],
];

my $STATE_RULES = [
    [ CONVERTED => {
        currency => 'YND_FIXED',
        currencyConverted => 'Yes',
        archived => 'Yes',
    } ],
    [ ARCHIVED => {
        archived => 'Yes',
        currencyConverted => "No"
    }, {
        archived => 'Yes',
        currencyConverted => 'Yes',
        currency__ne => 'YND_FIXED',
    } ],
    [ ENDED => {
        finish_time__ne => undef,
        finish_time__ne => '0000-00-00',
        finish_time__lt__dont_quote => 'CURDATE()',
        archived => 'No',
        currencyConverted => "No"
    }, {
        finish_time__ne => undef,
        finish_time__ne => '0000-00-00',
        finish_time__lt__dont_quote => 'CURDATE()',
        archived => 'No',
        currencyConverted => 'Yes',
        currency__ne => 'YND_FIXED'
    } ],
    [ SUSPENDED => {
        statusShow => 'No',
        archived => 'No',
        currencyConverted => 'No',
        finish_time => [ undef, '0000-00-00' ],
    }, {
        statusShow => 'No',
        archived => 'No',
        currencyConverted => 'No',
        finish_time__ge__dont_quote => 'CURDATE()',
    }, {
        statusShow => 'No',
        archived => 'No',
        currencyConverted => 'Yes',
        currency__ne => 'YND_FIXED',
        finish_time => [ undef, '0000-00-00' ],
    }, {
        statusShow => 'No',
        archived => 'No',
        currencyConverted => 'Yes',
        currency__ne => 'YND_FIXED',
        finish_time__ge__dont_quote => 'CURDATE()',
    } ],
    [ OFF => {
        statusActive => 'No',
        statusShow => 'Yes',
        archived => 'No',
        currencyConverted => 'No',
        finish_time => [ undef, '0000-00-00' ],
    }, {
        statusActive => 'No',
        statusShow => 'Yes',
        archived => 'No',
        currencyConverted => 'No',
        finish_time__ge__dont_quote => 'CURDATE()',
    }, {
        statusActive => 'No',
        statusShow => 'Yes',
        archived => 'No',
        currencyConverted => 'Yes',
        currency__ne => 'YND_FIXED',
        finish_time => [ undef, '0000-00-00' ],
    }, {
        statusActive => 'No',
        statusShow => 'Yes',
        archived => 'No',
        currencyConverted => 'Yes',
        currency__ne => 'YND_FIXED',
        finish_time__ge__dont_quote => 'CURDATE()',
    } ],
    [ ON  => { 
        statusActive => 'Yes',
        archived => 'No',
        statusShow => 'Yes',
        finish_time => [ undef, '0000-00-00' ],
    }, { 
        statusActive => 'Yes',
        archived => 'No',
        statusShow => 'Yes',
        finish_time__ge__dont_quote => 'CURDATE()',
    } ]
];


my $STATUS_PAYMENT_RULES = [
    [ DISALLOWED => { statusModerate => ['Sent', 'Ready', 'No', 'New'],
                      wallet_cid => 0
                    } ],
    [ ALLOWED  => { statusModerate => 'Yes' }, { wallet_cid__ne => 0 } ],
];

=head2 statuses_where($statuses)

    При помощи правил из $STATUS_RULES
    преобразует названия статусов из SelectionCriteria в where
    Принимает: массив вида ['MODERATION', ...]
    Возвращает: структуру вида
    {
        statusModerate ["Sent", "Sending", "Ready"],
    }
=cut

sub statuses_where($) {
    my $statuses = shift;
    my @cond;
    foreach my $status ( @$statuses ) {
        foreach my $rule ( @$STATUS_RULES ) {
            my ($rule_status, $conditions) = @$rule;
            if ($status eq $rule_status) {
                push @cond, '_AND' , where_from_model(yclone($conditions));
            }
        }
    }
    return { _OR => \@cond };
}


=head2 states_where($states)

    При помощи правил из $STATE_RULES
    преобразует названия состояний из SelectionCriteria в where

    Принимает: массив вида ['OFF' ...]
    Возвращает: структуру вида
    {
        statusActive        "No",
        statusShow          "Yes"
    }

=cut

sub states_where($$) {
    my ($states, $work_currency) = @_;
    my @cond;
    foreach my $state ( @$states ) {
        foreach my $rule ( @$STATE_RULES ) {
            my ($rule_state, @conditions) = @$rule;
            if ($state eq $rule_state) {
                foreach my $condition ( @conditions ) {
                    push @cond, '_AND' , where_from_model(yclone($condition));
                }
            }
        }
    }
    return { _OR => \@cond };
}

=head2 status_payment_where($)

    При помощи правил из $STATUS_PAYMENT_RULES
    преобразует названия статусов оплаты из SelectionCriteria в sql_condition where

=cut

sub status_payment_where($) {
    my $statuses = shift;
    my @cond;
    my $epsilon = $Currencies::EPSILON;
    foreach my $status ( @$statuses ) {
        foreach my $rule ( @$STATUS_PAYMENT_RULES ) {
            my ($rule_status, @conditions) = @$rule;
            if ($status eq $rule_status) {
                foreach my $condition ( @conditions ) {
                    push @cond, '_AND' , where_from_model(yclone($condition));
                }
            }
        }
    }
    return { _OR => \@cond };
}


=head2 status_from_model($model)

    На основе полей модельки и правил получает статус объявления 

=cut

sub status_from_model($) {
    my $model = shift;
    my $status = get_field_value_depends_on_rules($model, $STATUS_RULES);
    return $status // 'UNKNOWN';
}

=head2 state_from_model($model)

    На основе полей модельки и правил получает состояние объявления 

=cut

sub state_from_model($$) {
    my ($model, $work_currency) = @_;
    my $state = get_field_value_depends_on_rules($model, $STATE_RULES);
    return $state // 'UNKNOWN';
}

=head2 status_payment_from_model($model)

    На основе полей модельки и правил получает статус объявления

=cut

sub status_payment_from_model($) {
    my ($model) = @_;
    my $status_payment = get_field_value_depends_on_rules($model, $STATUS_PAYMENT_RULES);
    return $status_payment // 'UNKNOWN';
}

=head2 merge_statuses_where(@where_conditions )

    Where-хэши @where_conditions в формате DBTools sql_condition вида { _OR => [\@conditions] }
    мерджит с условием AND между собой. Т.к. речь идет о where структурах на
    стейты-статусы, то проверяем на наличие _OR внутри, иначе не мерджим

=cut

sub merge_statuses_where {
    my @where_conditions = @_;
    my $merged = {};
    for my $or_condition (@where_conditions) {
        next unless ref $or_condition eq 'HASH' && exists $or_condition->{_OR};
        push @{$merged->{_AND}}, (_OR => $or_condition->{_OR});
    }
    return $merged;
}

1;
