package API::Service::Campaigns::Strategies;

use strict;
use warnings;
use utf8;
use constant CPA => "Cpa";


=head2

    $Id$

=head1 NAME

    API::Service::Campaigns::Strategies

=head1 SYNOPSIS

=head1 DESCRIPTION

    Модуль для преобразования стратегий из формата,
    который принимает функция Campaign::define_strategy
    в формат API и наоборот


=head1 METHODS

    strategy_sets_to_external()
    strategy_sets_to_internal()

=cut

use List::MoreUtils qw/any/;
use Hash::Merge::Simple qw/merge/;
use Carp;

use Settings;

use Stat::Const qw/:enums/;

use API::Converter::ConvertSubs qw/convert_to_long convert_to_money SNAKE_CASE_to_CamelCase convert_to_float_2_decimal_places/;

require Exporter;

use base qw/Exporter/;

our @EXPORT_OK = qw(
        strategy_sets_to_internal
        strategy_sets_to_external
        get_structure_name_by_strategy
        is_strategy_consistent
        is_strategy_has_no_more_than_one_structures
        is_strategy_has_needed_structure
        is_autobudget
        merge_strategies
        is_network_default_strategy
);

my %autobudget_strategies = (
    WB_MAXIMUM_CLICKS => 1,
    WEEKLY_CLICK_PACKAGE => 1,
    AVERAGE_CPC => 1,
    WB_MAXIMUM_CONVERSION_RATE => 1,
    WB_MAXIMUM_APP_INSTALLS => 1,
    AVERAGE_CPA => 1,
    PAY_FOR_CONVERSION => 1,
    PAY_FOR_CONVERSION_PER_CAMPAIGN => 1,
    PAY_FOR_CONVERSION_PER_FILTER => 1,
    PAY_FOR_CONVERSION_CRR => 1,
    AVERAGE_CPI => 1,
    PAY_FOR_INSTALL => 1,
    AVERAGE_ROI => 1,
    AVERAGE_CRR => 1,
    AVERAGE_CPC_PER_CAMPAIGN => 1,
    AVERAGE_CPC_PER_FILTER => 1,
    AVERAGE_CPA_PER_CAMPAIGN => 1,
    AVERAGE_CPA_PER_FILTER => 1,
    WB_MAXIMUM_IMPRESSIONS => 1,
    CP_MAXIMUM_IMPRESSIONS => 1,
    WB_AVERAGE_CPV => 1,
    CP_AVERAGE_CPV => 1,
    WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS => 1,
    CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS => 1,
);


my %fields_inner_to_outer = (
    limit_clicks    => 'ClicksPerWeek',
    bid             => 'BidCeiling',
    avg_bid         => 'AverageCpc',
    avg_cpa         => 'AverageCpa',
    filter_avg_bid  => 'FilterAverageCpc',
    filter_avg_cpa  => 'FilterAverageCpa',
    avg_cpi         => 'AverageCpi',
    avg_cpv         => 'AverageCpv',
    sum             => 'WeeklySpendLimit',
    goal_id         => 'GoalId',
    roi_coef        => 'RoiCoef',
    crr             => 'Crr',
    reserve_return  => 'ReserveReturn',
    profitability   => 'Profitability',
    ContextLimit    => 'LimitPercent',
    auto_prolongation => 'AutoContinue',
    avg_cpm         => 'AverageCpm',
    start           => 'StartDate',
    finish          => 'EndDate',
    budget          => 'SpendLimit',
);
my %fields_outer_to_inner = reverse %fields_inner_to_outer;
$fields_outer_to_inner{CPA()} = "avg_cpa";

# названия стратегий в API - к названиям стратегий из функции detect_strategy
my %strategy_name_inner_to_outer = (
    default                 => 'HIGHEST_POSITION',
    autobudget              => 'WB_MAXIMUM_CLICKS',
    autobudget_week_bundle  => 'WEEKLY_CLICK_PACKAGE',
    autobudget_avg_click    => 'AVERAGE_CPC',
    autobudget_avg_cpa      => 'AVERAGE_CPA',
    autobudget_avg_cpi      => 'AVERAGE_CPI',
    autobudget_crr          => 'AVERAGE_CRR',
    autobudget_roi          => 'AVERAGE_ROI',
    autobudget_avg_cpc_per_camp => 'AVERAGE_CPC_PER_CAMPAIGN',
    autobudget_avg_cpa_per_camp => 'AVERAGE_CPA_PER_CAMPAIGN',
    autobudget_avg_cpc_per_filter => 'AVERAGE_CPC_PER_FILTER',
    autobudget_avg_cpa_per_filter => 'AVERAGE_CPA_PER_FILTER',
    no_premium_highest_place => 'IMPRESSIONS_BELOW_SEARCH',
    stop                    => 'SERVING_OFF',
    cpa_optimizer           => 'WB_MAXIMUM_CONVERSION_RATE',
    cpi_optimizer           => 'WB_MAXIMUM_APP_INSTALLS',
);
my %strategy_name_outer_to_inner = reverse %strategy_name_inner_to_outer;
$strategy_name_outer_to_inner{PAY_FOR_CONVERSION} = "autobudget_avg_cpa";
$strategy_name_outer_to_inner{PAY_FOR_CONVERSION_CRR} = "autobudget_crr";
$strategy_name_outer_to_inner{PAY_FOR_CONVERSION_PER_CAMPAIGN} = "autobudget_avg_cpa_per_camp";
$strategy_name_outer_to_inner{PAY_FOR_CONVERSION_PER_FILTER} = "autobudget_avg_cpa_per_filter";
$strategy_name_outer_to_inner{PAY_FOR_INSTALL} = "autobudget_avg_cpi";

# названия стратегий для сети в API - к названиям стратегий из функции detect_context_strategy
my %context_strategy_name_inner_to_outer = (
    default                 => 'NETWORK_DEFAULT',
    stop                    => 'SERVING_OFF',
    maximum_coverage        => 'MAXIMUM_COVERAGE',
    autobudget              => 'WB_MAXIMUM_CLICKS',
    autobudget_week_bundle  => 'WEEKLY_CLICK_PACKAGE',
    autobudget_avg_click    => 'AVERAGE_CPC',
    autobudget_avg_cpa      => 'AVERAGE_CPA',
    autobudget_avg_cpi      => 'AVERAGE_CPI',
    autobudget_crr          => 'AVERAGE_CRR',
    autobudget_roi          => 'AVERAGE_ROI',
    autobudget_avg_cpc_per_camp => 'AVERAGE_CPC_PER_CAMPAIGN',
    autobudget_avg_cpa_per_camp => 'AVERAGE_CPA_PER_CAMPAIGN',
    autobudget_avg_cpc_per_filter => 'AVERAGE_CPC_PER_FILTER',
    autobudget_avg_cpa_per_filter => 'AVERAGE_CPA_PER_FILTER',
    cpa_optimizer           => 'WB_MAXIMUM_CONVERSION_RATE',
    cpi_optimizer           => 'WB_MAXIMUM_APP_INSTALLS',
    cpm_default             => 'MANUAL_CPM',
    autobudget_max_impressions => 'WB_MAXIMUM_IMPRESSIONS',
    autobudget_max_impressions_custom_period => 'CP_MAXIMUM_IMPRESSIONS',
    autobudget_avg_cpv      => 'WB_AVERAGE_CPV',
    autobudget_avg_cpv_custom_period => 'CP_AVERAGE_CPV',
    autobudget_max_reach    => 'WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS',
    autobudget_max_reach_custom_period => 'CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS',
);

my %context_strategy_name_outer_to_inner = reverse %context_strategy_name_inner_to_outer;
$context_strategy_name_outer_to_inner{PAY_FOR_CONVERSION} = "autobudget_avg_cpa";
$context_strategy_name_outer_to_inner{PAY_FOR_CONVERSION_PER_CAMPAIGN} = "autobudget_avg_cpa_per_camp";
$context_strategy_name_outer_to_inner{PAY_FOR_CONVERSION_PER_FILTER} = "autobudget_avg_cpa_per_filter";
$context_strategy_name_outer_to_inner{PAY_FOR_CONVERSION_CRR} = "autobudget_crr";
$context_strategy_name_outer_to_inner{PAY_FOR_INSTALL} = "autobudget_avg_cpi";

# список конверсионных стратегий
my %conversion_strategies = map { $_ => 1 } qw/autobudget_roi autobudget_crr autobudget_avg_cpa cpa_optimizer autobudget_avg_cpi cpi_optimizer/;

my @fields_need_convert_to_money = qw/sum bid avg_bid filter_avg_bid avg_cpa filter_avg_cpa avg_cpi avg_cpm avg_cpv budget/;
my @fields_need_convert_to_float = qw/roi_coef profitability/;
my %strategy_names_with_pay_for_conversion = map {$_ => 1} qw/autobudget_avg_cpa autobudget_avg_cpa_per_camp autobudget_crr autobudget_avg_cpv autobudget_avg_cpv_custom_period/;

=head2 is_autobudget(strategy_name)

    Проверяет является ли стратегия автобюджетной
    strategy_name название стратегии в API!

=cut

sub is_autobudget($){
    my $strategy_name = shift;
    if ($autobudget_strategies{$strategy_name}) {
        return 1;
    } else {
        return 0;
    }
}

=head2 get_structure_name_by_strategy($strategy_name)

    По названию стратегии (BiddingStrategy) возвращает название элемента содержащего ее настройки

=cut
sub get_structure_name_by_strategy {
    my $strategy_name = shift;
    my $structure_name = SNAKE_CASE_to_CamelCase($strategy_name);
    return $structure_name;
}

=head2 is_strategy_consistent(strategies)

    Проверяет соответствует ли название стратегии названию структуры с параметрам
    Принимает: настройки стратегий формата API
    Возвращает: структуру {Search => 1|0, Network => 1|0}

=cut

sub is_strategy_consistent {
    my $strategies = shift;
    my %result;
    for my $type (qw/Search Network/) {
        if (exists $strategies->{$type}) {
            my $strategy = $strategies->{$type};
            my $name = $strategy->{BiddingStrategyType};
            my $structure_name = get_structure_name_by_strategy($name);
            my $has_structure = scalar keys %$strategy == 1 ? 0 : 1;
            if (exists $strategy->{$structure_name} || !$has_structure) {
                $result{$type} = 1;
            } else {
                $result{$type} = 0;
            }
        }
    }
    return \%result;
}

=head2 is_strategy_has_no_more_than_one_structures(strategies)

    Проверяет что структур с настройками стратегий не больше одной
    Принимает: настройки стратегий формата API
    Возвращает: структуру {Search => 1|0, Network => 1|0}

=cut

sub is_strategy_has_no_more_than_one_structures {
    my $strategies = shift;
    my %result;
    for my $type (qw/Search Network/) {
        if (exists $strategies->{$type}) {
            my %strategy = %{$strategies->{$type}};
            delete $strategy{BiddingStrategyType};
            my $structures_count = scalar keys %strategy;
            if ($structures_count > 1) {
                $result{$type} = 0;
            } else {
                $result{$type} = 1;
            }
        }
    }
    return \%result;
}

=head2 is_strategy_has_needed_structure(strategies)

    Проверяет что для переданной автобюджетной стратегии,
    также передана структура с настройками
    Принимает: настройки стратегий формата API
    Возвращает: структуру {Search => 1|0, Network => 1|0}

=cut

sub is_strategy_has_needed_structure {
    my ($strategies, $campaign_type) = @_;
    my %result;
    for my $type (qw/Search Network/) {
        if (exists $strategies->{$type}) {
            my %strategy = %{$strategies->{$type}};
            my $name = $strategy{BiddingStrategyType};
            my $structure_name = get_structure_name_by_strategy($name);
            if (is_autobudget($name)) {
                if (exists $strategy{$structure_name}){
                    $result{$type} = 1;
                } else {
                    $result{$type} = 0;
                }
            } elsif ($name eq 'NETWORK_DEFAULT') {
                if (exists $strategy{$structure_name}){
                    $result{$type} = 1;
                } else {
                    $result{$type} = 0;
                }
            } else {
                $result{$type} = 1;
            }
        }
    }
    return \%result;
}

=head2 is_need_convert_setting_to_money($field)

    По внутреннему названию поля из Settings проверяет нуждается ли его
    значение в конвертации в деньги (делением на 10e6)

=cut

sub is_need_convert_setting_to_money {
    my $inner_setting_name = shift;
    return any { $_ eq $inner_setting_name }
        map {$_} @fields_need_convert_to_money;
}

=head2 is_need_convert_setting_to_float($field)

    По внутреннему названию поля из Settings проверяет нуждается ли его
    значение в конвертации в десятичную дробь с двумя знаками после запятой
    (делением на 10e6, и округлением до двух последний знаков)

=cut

sub is_need_convert_setting_to_float {
    my $inner_setting_name = shift;
    return any { $_ eq $inner_setting_name }
        map {$_} @fields_need_convert_to_float;
}

=head2 is_need_convert_setting_to_long($field)

    По внутреннему названию поля из Settings проверяет нуждается ли его
    значение в конвертации в long (умножением на 10e6) (микро-рубли)

=cut

sub is_need_convert_setting_to_long {
    my $inner_setting_name = shift;
    return any { $_ eq $inner_setting_name }
        map {$_} (@fields_need_convert_to_money,@fields_need_convert_to_float);
}

=head2 strategy_sets_to_internal(strategies)

    Преобразует настройки стратегий из формата API
    в формат который возвращает метод define_strategy

=cut

sub strategy_sets_to_internal {
    my ($strategies) = @_;
    my %result;

    for my $type (qw/Search Network/) {
        if (exists $strategies->{$type}) {
            my $strategy = $strategies->{$type};
            my $is_stop = $strategy->{BiddingStrategyType} eq 'SERVING_OFF' ? 1 : 0;
            my $result_strategy;
            if ($type eq 'Search') {
                $result_strategy->{name} = $strategy_name_outer_to_inner{$strategy->{BiddingStrategyType}};
            } else {
                $result_strategy->{name} = $context_strategy_name_outer_to_inner{$strategy->{BiddingStrategyType}};
            }
            if ($result_strategy->{name} =~ /^(no_premium)_(\w+)/) {
                $result{name} = 'strategy_'.$1;
                $result_strategy->{name} = $1;
                $result_strategy->{place} = $2;
            } elsif ($result_strategy->{name} =~ /^(min_price)_(\w+)/) {
                $result{name} = $result_strategy->{name};
                $result_strategy->{name} = $1;
                $result_strategy->{place} = $2;
            } elsif ($result_strategy->{name} eq 'cpa_optimizer') {
                $result{name} = 'cpa_optimizer';
                $result_strategy->{name} = 'autobudget';
                $result_strategy->{goal_id} //= undef;
            } elsif ($result_strategy->{name} eq 'cpi_optimizer') {
                $result{name} = 'cpa_optimizer';
                $result_strategy->{name} = 'autobudget';
                $result_strategy->{goal_id} = $Settings::DEFAULT_CPI_GOAL_ID;
            } elsif ($result_strategy->{name} =~ /^autobudget_avg_cp[ac]_per_(camp|filter)$/) {
                # TODO maxlog: хак для того, чтобы прописать правильную стратегию в смарт-кампанию. Надо проверить с autobudget_roi
                $result{name} = $result_strategy->{name};
            }

            # в случае autobudget_avg_cpi не выставляем здесь дефолтный goal_id,
            # т.к. при add он заполнится в _add_prepare_model_default_values
            # а при update он возьмётся из стратегии кампании в БД

            my $settings_structure_name = get_structure_name_by_strategy($strategy->{BiddingStrategyType});
            my $settings_structure = $strategy->{$settings_structure_name};
            if (defined $settings_structure) {
                for my $setting (keys %$settings_structure) {
                    my $value = $settings_structure->{$setting};
                    if ($setting eq 'LimitPercent' && !defined $value) {
                        $value = 100;
                    }

                    my $inner_setting_name;
                    if ($setting eq 'SpendLimit' && ($result_strategy->{name} eq 'autobudget_max_impressions'
                            || $result_strategy->{name} eq 'autobudget_avg_cpv'
                            || $result_strategy->{name} eq 'autobudget_max_reach')) {
                        $inner_setting_name = 'sum';
                    } elsif ($setting eq CPA() && $result_strategy->{name} eq 'autobudget_avg_cpa_per_filter') {
                        $inner_setting_name = 'filter_avg_cpa';
                    } else {
                        $inner_setting_name = $fields_outer_to_inner{$setting};
                    }
                    my $inner_value;
                    if ($inner_setting_name eq 'auto_prolongation') {
                        $inner_value = $value eq 'YES' ? 1 : 0;
                    } elsif (is_need_convert_setting_to_money($inner_setting_name)) {
                        $inner_value = convert_to_money($value);
                    } elsif (is_need_convert_setting_to_float($inner_setting_name)) {
                        $inner_value = convert_to_float_2_decimal_places($value);
                    } else {
                        $inner_value = $value;
                    }
                    $result_strategy->{$inner_setting_name} = $inner_value;
                }
                if ($result_strategy->{name} eq 'autobudget_avg_cpa'
                    || $result_strategy->{name} eq 'autobudget_avg_cpa_per_camp'
                    || $result_strategy->{name} eq 'autobudget_avg_cpa_per_filter'
                    || $result_strategy->{name} eq 'autobudget_avg_cpi'
                    || $result_strategy->{name} eq 'autobudget_crr') {
                    $result_strategy->{pay_for_conversion} = ($settings_structure_name eq 'PayForConversion' ||
                        $settings_structure_name eq 'PayForConversionPerCampaign' ||
                        $settings_structure_name eq 'PayForConversionPerFilter' ||
                        $settings_structure_name eq 'PayForInstall' ||
                        $settings_structure_name eq 'PayForConversionCrr') ? 1 : 0;
                }
            }
            if ($type eq 'Search') {
                $result{is_search_stop} = $is_stop;
                $result{name} ||= $result_strategy->{name};
                $result{search} = $result_strategy;
            } else {
                $result{is_net_stop} = $is_stop;
                $result{net} = $result_strategy;
            }
        }
    }
    if ( (exists $result{search} && $result{search}{name} =~ /autobudget/) ||
        (exists $result{net} && $result{net}{name} =~ /autobudget/)
    ) {
        $result{is_autobudget} = 1;
    } else {
        $result{is_autobudget} = 0;
    }
    if (exists $result{net} && $result{net}{name} ne 'default' && $result{net}{name} ne 'stop') {
        # TODO maxlog тут можем записать неправильную стратегию
        # print STDERR "Campaigns#991001 set different_places ".Dumper({result_name=> $result{name}, result_net=>$result{net}, result_search=>$result{search}})."\n";
        $result{name} = 'different_places';
    }
    return \%result;
}

=head2 strategy_sets_to_external(strategy)

    Преобразует настройки стратегий из внутреннего формата в API формат
    $strategy настройки стратегий в виде который возвращает define_strategy

=cut

sub strategy_sets_to_external {
    my ($strategy) = @_;
    my %result;
    for my $type (qw/search net/) {
        my %names_mapper;
        if ($type eq 'search') {
            %names_mapper = %strategy_name_inner_to_outer;
        } else {
            %names_mapper = %context_strategy_name_inner_to_outer;
        }
        if (exists $strategy->{$type}){
            my %result_strategy;
            my ($external_name, $internal_name);
            if (exists $strategy->{$type}{'place'}) {
                $internal_name = $strategy->{$type}{name}.'_'.$strategy->{$type}{place};
            } else {
                $internal_name = $strategy->{$type}{name};
            }
            if ($internal_name eq 'autobudget' && defined $strategy->{$type}{goal_id}) {
                if (exists $MOBILE_APP_SPECIAL_GOALS{ $strategy->{$type}{goal_id} }) {
                    $internal_name = 'cpi_optimizer';
                } else {
                    $internal_name = 'cpa_optimizer';
                }
            }

            # В зависимости от значения параметра pay_for_conversion(0/1) внутренние стратегии autobudget_avg_cpa*,
            # autobudget_crr могут мапиться в одну из двух внешних:
            #  autobudget_avg_cpa (0) -> AVERAGE_CPA
            #  autobudget_avg_cpa (1) -> PAY_FOR_CONVERSION
            #  autobudget_avg_cpa_per_camp (0) -> AVERAGE_CPA_PER_CAMPAIGN
            #  autobudget_avg_cpa_per_camp (1) -> PAY_FOR_CONVERSION_PER_CAMPAIGN
            #  autobudget_avg_cpa_per_filter (0) -> AVERAGE_CPA_PER_FILTER
            #  autobudget_avg_cpa_per_filter (1) -> PAY_FOR_CONVERSION_PER_FILTER
            #  autobudget_crr (0) -> AVERAGE_CRR
            #  autobudget_crr (1) -> PAY_FOR_CONVERSION_CRR
            if ($internal_name eq 'autobudget_avg_cpa' &&
                defined $strategy->{$type}{pay_for_conversion} && $strategy->{$type}{pay_for_conversion}) {
                $external_name = "PAY_FOR_CONVERSION";
            } elsif($internal_name eq 'autobudget_avg_cpa_per_camp' &&
                defined $strategy->{$type}{pay_for_conversion} && $strategy->{$type}{pay_for_conversion}) {
                $external_name = "PAY_FOR_CONVERSION_PER_CAMPAIGN";
            } elsif($internal_name eq 'autobudget_avg_cpa_per_filter' &&
                defined $strategy->{$type}{pay_for_conversion} && $strategy->{$type}{pay_for_conversion}) {
                $external_name = "PAY_FOR_CONVERSION_PER_FILTER";
            } elsif($internal_name eq 'autobudget_avg_cpi' &&
                defined $strategy->{$type}{pay_for_conversion} && $strategy->{$type}{pay_for_conversion}) {
                $external_name = "PAY_FOR_INSTALL";
            } elsif($internal_name eq 'autobudget_crr' &&
                defined $strategy->{$type}{pay_for_conversion} && $strategy->{$type}{pay_for_conversion}) {
                $external_name = "PAY_FOR_CONVERSION_CRR";
            } else {
                $external_name = $names_mapper{$internal_name};
            }

            my %settings;
            for my $setting (keys %{$strategy->{$type}}){
                next if $setting eq 'name';
                next if $setting eq 'place';
                next if $setting eq 'pay_for_conversion';
                next if $setting eq 'last_bidder_restart_time';
                next if $setting eq 'last_update_time';
                next if $setting eq 'daily_change_count';
                next if ($setting eq 'goal_id') and ($external_name =~ /WB_MAXIMUM_CLICKS|WB_MAXIMUM_APP_INSTALLS|AVERAGE_CPI|PAY_FOR_INSTALL/);
                if ($setting eq 'ContextLimit') {
                    if ($strategy->{$type}{$setting} == 0 || $strategy->{$type}{$setting} == 254) {
                        next;
                    }
                }
                if ($external_name eq "PAY_FOR_CONVERSION"
                    || $external_name eq "PAY_FOR_CONVERSION_PER_CAMPAIGN"
                    || $external_name eq "PAY_FOR_CONVERSION_PER_FILTER"
                    || $external_name eq "PAY_FOR_INSTALL") {
                    next if $setting eq 'bid';
                    next if $setting eq 'avg_bid';
                }

                my $value;
                if ( is_need_convert_setting_to_long($setting)) {
                    $value = convert_to_long($strategy->{$type}{$setting});
                } elsif ($setting eq 'auto_prolongation') {
                    $value = $strategy->{$type}{$setting} ? 'YES' : 'NO';
                } else {
                    $value = $strategy->{$type}{$setting};
                }

                my $external_setting_name;
                if ($setting eq 'sum' && ($external_name eq 'WB_MAXIMUM_IMPRESSIONS'
                        || $external_name eq 'WB_AVERAGE_CPV'
                        || $external_name eq 'WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS')) {
                    $external_setting_name = 'SpendLimit';
                } elsif (($external_name eq "PAY_FOR_CONVERSION"
                    || $external_name eq "PAY_FOR_CONVERSION_PER_CAMPAIGN")
                    && $setting eq 'avg_cpa') {
                    $external_setting_name = CPA;
                } elsif ($external_name eq "PAY_FOR_CONVERSION_PER_FILTER" && $setting eq 'filter_avg_cpa') {
                    $external_setting_name = CPA;
                } else {
                    $external_setting_name = $fields_inner_to_outer{$setting};
                }
                $settings{$external_setting_name} = $value;
            }
            my $settings_structure_name = get_structure_name_by_strategy($external_name);
            $result_strategy{BiddingStrategyType} = $external_name;
            $result_strategy{$settings_structure_name} = \%settings if %settings;
            if ($type eq 'search') {
                $result{Search} = \%result_strategy;
            } else {
                $result{Network} = \%result_strategy;
            }
        }
    }
    return \%result;
}

=head2 merge_strategies(strategy, old_strategy)

    Если клиент обновляет поля стратегий частично,
    метод добавляет другие необходимые поля из базы и считает вычисляемые поля

=cut

sub merge_strategies {
    my ($strategy, $old_strategy) = @_;
    my $result_strategy;
    if (%$strategy) {
        for my $type (qw/search net/) {
            if ($strategy->{$type} ) {
                if (
                    ($strategy->{$type}{name} ne $old_strategy->{$type}{name}) ||
                    _is_WB_MAXIMUM_CONVERSION_RATE_and_WB_MAXIMUM_CLICKS_replacement($strategy->{$type}, $old_strategy->{$type}) ||
                    _is_PAY_FOR_CONVERSION_flag_changed($strategy->{$type}, $old_strategy->{$type})
                ){
                    $result_strategy->{$type} = $strategy->{$type};

                } else {
                    $result_strategy->{$type} = merge $old_strategy->{$type}, $strategy->{$type};
                }
            } else {
                $result_strategy->{$type} = $old_strategy->{$type};
            }
        }
    } else {
        $result_strategy = $old_strategy;
    }
    $result_strategy->{name} = _calculate_strategy_common_name($result_strategy);
    if ($result_strategy->{search}{name} =~ /autobudget/ ||
        $result_strategy->{net}{name} =~ /autobudget/
    ) {
        $result_strategy->{is_autobudget} = 1;
    } else {
        $result_strategy->{is_autobudget} = 0;
    }
    $result_strategy->{is_search_stop} = $result_strategy->{search}{name} eq 'stop' ? 1 : 0;
    $result_strategy->{is_net_stop} = $result_strategy->{net}{name} eq 'stop' ? 1 : 0;
    return $result_strategy;
}

sub _calculate_strategy_common_name {
    my ($result_strategy) = @_;

    my $common_name;
    if ($result_strategy->{net}{name} ne 'default' && $result_strategy->{net}{name} ne 'stop') {
        $common_name = 'different_places';
    } else {
        $common_name = _calculate_strategy_common_name_by_search_name($result_strategy->{search});
    }
    return $common_name;
}

sub _calculate_strategy_common_name_by_search_name {
    my ($strategy) = @_;
    my $common_name;
    if ($strategy->{name} eq 'default') {
        $common_name = 'default';
    } elsif ($strategy->{name} eq 'no_premium') {
        $common_name = 'strategy_no_premium';
    } elsif ($strategy->{name} eq 'min_price') {
        $common_name = 'min_price_'.$strategy->{place};
    } elsif ($strategy->{name} eq 'autobudget' && defined $strategy->{goal_id}) {
        $common_name = 'cpa_optimizer';
    } else {
        $common_name = $strategy->{name};
    }
    return $common_name;
}

=head2 _is_WB_MAXIMUM_CONVERSION_RATE_and_WB_MAXIMUM_CLICKS_replacement(strategy, old_strategy)

    Если клиент пытается установить WB_MAXIMUM_CONVERSION_RATE вместо WB_MAXIMUM_CLICKS или наоборот
    возвращает 1 в противном случае 0

=cut

sub _is_WB_MAXIMUM_CONVERSION_RATE_and_WB_MAXIMUM_CLICKS_replacement {
    my ($strategy, $old_strategy) = @_;
    if ( $strategy->{name} eq 'autobudget' && $old_strategy->{name} eq 'autobudget' &&
        (  (defined $strategy->{goal_id} && !defined $old_strategy->{goal_id}) ||
           (defined $old_strategy->{goal_id} && !defined $strategy->{goal_id})  )
    ) {
        return 1;
    } else {
        return 0;
    };
}

=head2 _is_PAY_FOR_CONVERSION_flag_changed(strategy, old_strategy)

    Определяет факт переключения флага оплаты за конверсии без изменения имени внутренней стратегии
    Набор стратегий с оплатой за конверсии: %strategy_names_with_pay_for_conversion

    Возвращает:
     1 - если изменился флаг Оплата за конверсии
     0 - если флаг не изменился

=cut

sub _is_PAY_FOR_CONVERSION_flag_changed {
    my ($strategy, $old_strategy) = @_;

    if( $strategy_names_with_pay_for_conversion{$strategy->{name}}
        && $strategy_names_with_pay_for_conversion{$old_strategy->{name}}
        && $strategy->{pay_for_conversion} != $old_strategy->{pay_for_conversion}) {

        return 1;
    } else {
        return 0;
    }
}

=head2 is_network_default_strategy($strategy)

    Функция возвращает признак, является ли стратегия в сети NETWORK_DEFAULT

=cut

sub is_network_default_strategy {
    my ($strategy) = @_;
    return $strategy->{net} && ($strategy->{net}{name} // '') eq 'default' ? 1 : 0;
}

1;
