package API::Service::Campaigns::Types;

use Direct::Modern;

use Yandex::HashUtils qw/hash_cut hash_merge/;
use Yandex::ListUtils qw/xisect/;
use API::Converter::ConvertSubs qw/CamelCase_to_SNAKE_CASE SNAKE_CASE_to_CamelCase/;
use API::Campaign::Const qw/%CAMPAIGN_TYPE_MAP/;

use Exporter qw/import/;
our @EXPORT_OK = qw/
    convert_type
    convert_types
    convert_type_to_external
    convert_types_to_external
    get_type_structures_names
    get_structure_name_by_type
    get_type_by_structure_name
    get_type_by_structure
    get_type_structure_name
    prepare_type_structure
    disclose_type_structure
    has_type_structure
    get_external_types
    convert_attribution_model
    convert_video_target
    convert_attribution_model_to_external
    convert_video_target_to_external
/;

my %attribution_model_map = (
    last_click => 'LC',
    first_click => 'FC',
    last_significant_click => 'LSC',
    last_yandex_direct_click => 'LYDC',
    first_click_cross_device => 'FCCD',
    last_significant_click_cross_device => 'LSCCD',
    last_yandex_direct_click_cross_device => 'LYDCCD'
);

my %attribution_model_reverse_map = reverse %attribution_model_map;

my %video_target_map = (
    long_clicks             => 'CLICKS',
    completes            => 'VIEWS'
);

my %video_target_reverse_map = reverse %video_target_map;

=head2 get_external_types

    Возвращает внешние названия типов кампаний(те которые видят пользователи API)

=cut

sub get_external_types {
    return [sort values %CAMPAIGN_TYPE_MAP];
}

=head2 get_type_structures_names

    Получает названия структур с полями зависящими от типа
    Может принимать массив внешних названий типов кампаний

=cut

sub get_type_structures_names {
    my $external_types = shift;
    if (!$external_types || ! scalar @$external_types) {
        $external_types = get_external_types();
    }
    my @structures_names;
    for my $type (@$external_types){
        push @structures_names, get_structure_name_by_type($type);
    }
    return \@structures_names;
}

=head2 get_structure_name_by_type
    Получает название структуры с полями зависящими от типа  по внешнему названию типа кампании
    TEXT_CAMPAIGN -> TextCampaign

=cut

sub get_structure_name_by_type {
    return SNAKE_CASE_to_CamelCase(shift);
}

=head2 get_type_by_structure_name($name)

    Получает название типа кампании по названию структуры с полями зависящими от типа
    TextCampaign -> TEXT_CAMPAIGN

=cut

sub get_type_by_structure_name {
    return CamelCase_to_SNAKE_CASE(shift);
}

=head2 get_type_by_structure
    Высчитывает тип кампании
    Принимает: API::Service::ResultSet::Item->object
    Возвращает: внешние название типа, например - "TEXT_CAMPAIGN"

=cut

sub get_type_by_structure {
    my $item = shift;
    my $name = get_type_structure_name($item);
    if ($name) {
        return get_type_by_structure_name($name);
    }
    return;
}

=head2 get_type_structure_name
    Высчитывает название структуры с полями зависящими от типа
    Принимает: API::Service::ResultSet::Item->object
    Возвращает: название структуры, например - "TextCampaign"

=cut

sub get_type_structure_name {
    my $item = shift;
    my $structures_names = get_type_structures_names();
    for my $name (@$structures_names) {
        if (exists $item->{$name}) {
            return $name;
        }
    }
    return;
}

=head2 prepare_type_structure
    Добавляет поля зависящие от типа в соответствующую структуру
    Принимает: API::Service::Campaigns::GetRequest и хеш кампании после конвертации полей во внешний формат
    Возвращает: хеш кампании где зависящие от типа поля объединены в соответствующую структуру

=cut

sub prepare_type_structure {
    my ($get_request, $model) = @_;
    die 'unknown type' unless $model->{Type};
    my $type = get_structure_name_by_type($model->{Type});
    my @fields = $get_request->field_names_by_type($type);
    my $type_structure = hash_cut($model, @fields);
    delete $model->{$_} for @fields;
    $type_structure = undef unless %$type_structure;
    return $type_structure;
}

=head2 disclose_type_structure
    Раскрывает структуру с полями зависящими от типа, эти поля домерживаются в основной хеш, структура удаляется
    Принимает: хеш кампании до конвертации полей во внутренний формат
    Возвращает: хеш кампании где поля зависящие от типа находятся на одном уровне со всеми остальными

=cut

sub disclose_type_structure {
    my $model = shift;
    my $structures_names = get_type_structures_names();
    my $isect = xisect($structures_names, [keys %{$model}]);

    my $structure_name = shift @$isect;
    if (defined $structure_name) {
        hash_merge $model, $model->{$structure_name};
        delete $model->{$structure_name};
    }

    return $model;
}

=head2 has_type_structure
    Проверяет есть ли в хеше с кампанией структура с полями зависящими от типа
    Принимает: хеш кампании(во внешнем формате)
    Возвращает: 1 или 0

=cut

sub has_type_structure {
    my $model = shift;
    my $structures_names = get_type_structures_names();
    my $isect = xisect($structures_names, [keys %{$model}]);

    if (scalar @$isect) {
        return 1;
    }
    return 0;
}


=head2 convert_type
    Преобразует тип кампании ко внутреннему формату
=cut

sub convert_type {
    state $map = { reverse %CAMPAIGN_TYPE_MAP };
    defined( $_[0] ) ? $map->{ $_[0] } : undef;
}

=head2 convert_types
    Преобразует массив типов кампаний ко внутреннему формату

=cut

sub convert_types {
    my $types = shift;
    my @result;
    for my $type (@$types){
        push @result, convert_type($type);
    }
    return \@result;
}


=head2 convert_type_to_external
    Преобразует тип кампании к внешнему формату
=cut

sub convert_type_to_external {
    my $value = shift;
    return $CAMPAIGN_TYPE_MAP{$value};
}

=head2 convert_types_to_external
    Преобразует массив типов кампаний к внешнему формату

=cut

sub convert_types_to_external {
    my $types = shift;
    return [@CAMPAIGN_TYPE_MAP{@$types}];
}

=head2 convert_attribution_model
    Преобразует модель атрибуции кампании к внутреннему формату
=cut

sub convert_attribution_model {
    return $attribution_model_reverse_map{$_[0]};
}

=head2 convert_attribution_model_to_external
    Преобразует модель атрибуции кампании к внешнему формату
=cut

sub convert_attribution_model_to_external {
    my $value = shift;
    return $attribution_model_map{$value};
}

=head2 convert_video_target
    Преобразует цель показов видеоролика кампании к внутреннему формату
=cut

sub convert_video_target {
    return $video_target_reverse_map{$_[0]};
}

=head2 convert_video_target_to_external
    Преобразует цель показов видеоролика кампании к внешнему формату
=cut

sub convert_video_target_to_external {
    my $value = shift;
    return if !$value;
    return $video_target_map{$value};
}

1;
