package API::Service::Changes;

=pod

    $Id$

=head1 NAME

    API::Service::Changes — методы для получения информации о наличии изменений в данных API v5

=head1 DESCRIPTION

   Сервис предоставляет методы для получения информации о наличии изменений в различных объектах клиента и справочниках после заданного времени

=cut

use Direct::Modern;

use Try::Tiny;

use Yandex::DateTime qw/now_utc mysql2iso8601/;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::I18n;
use Yandex::HashUtils qw/
    hash_cut
    hash_grep
    hash_merge
/;
use Yandex::ListUtils qw/
    chunks
    xminus
    xisect
    nsort
/;
use Yandex::TimeCommon qw/mysql2unix/;

use List::Util qw/min max/;
use Scalar::Util qw/blessed/;

use Property;
use Settings;

use Client;

use Direct::Errors::Messages;

use API::Service::Request::Changes;
use API::Service::Changes::CampaingChangeTypes;
use API::Settings;
use API::Version;

use base qw/API::Service::Base/;

use constant SQL_WHERE_IDS_CHUNK_SIZE => 2000; # размер чанка для идентификаторов в SQL-запросе, чтобы не было длинных запросов
use constant SUPPORTED_CAMP_KIND => 'api5_changes';

my $ADGROUP_IDS_COUNT_QUERY_LIMIT = 10001; # ограничение на количество идентификаторов групп выбираемых из БД
my $AD_IDS_COUNT_QUERY_LIMIT = 50001; # ограничение на количество идентификаторов объявлений выбираемых из БД

my $property = Property->new($API::Settings::USE_CAMP_AGGREGATED_LASTCHANGE_PROPERTY);
my $dc2_property = Property->new($API::Settings::DK2_LIMITATION_ENABLED_PROPERTY_NAME);

my $local_timezone = DateTime::TimeZone->new(name=>'local');

sub response_ids {
    my ($self, $response) = @_;
    return undef;
}

sub count_error_warning_objects_in_reponse {
    my ($self, $response) = @_;
    return (0,0);
}

sub include_response_in_logs {
    my ($self) = @_;
    return 0;
}

=head2 checkDictionaries($request)

    Метод позволяет получить информацию о наличии изменений в справочниках регионов и временных зон после заданной даты
    self - объект сервиса
    request - ссылка на хэш, с данными запроса
    в методе используется только необязательный ключ Timestamp, содержащий дату в формате ISO8601, начиная с которой надо проверить наличие изменений

    $changes->checkDictionaries( { Timestamp => '2015-02-10T12:27:35Z' } );

    метод возвращает ссылку на хэш
    {
        RegionsChanged   => 'No',
        TimeZonesChanged => 'Yes',
        Timestamp        => '2015-02-10T12:27:35Z'
    }
    для случая, когда в запросе задан Timestamp
    или
    {
        Timestamp => '2015-02-10T12:27:35Z'
    }
    если Timestamp не задан

=cut

sub checkDictionaries {
    my ($self, $request) = @_;

    if (my $error = $self->_check_application_access()) {
        return $error;
    }

    my $response = {};

    try {
        my $changes_request = API::Service::Request::Changes->new($request);

        $response->{Timestamp} = $self->_get_current_timestamp_iso8601(); # получаем текущее время по UTC в формате YYYY-MM-DDThh:mm:ssZ

        if ($changes_request->has_timestamp()) {
            my $updated_dictionaries = $self->_get_dictionaries_changes($changes_request->mysql_timestamp()); # получаем информацию об изменении справочников регионов и временных зон после заданной даты
            $response->{RegionsChanged}   = $updated_dictionaries->{geo_regions}   ? 'YES' : 'NO';
            $response->{TimeZonesChanged} = $updated_dictionaries->{geo_timezones} ? 'YES' : 'NO';
            $response->{InterestsChanged} = $updated_dictionaries->{rmp_interest}  ? 'YES' : 'NO';
        }
    } catch {
        if ($self->_is_direct_defect_error($_)) { # обрабатываем исключения с объектом Direct::Defect
            $response = $_;
        } else {
            die $_; # остальные исключения пропагируем выше
        }
    };

    return $response;
}


sub _check_application_access() {
    my ($self) = @_;

    if ($self->application_id eq $API::Settings::DC2_APP_ID # только для ДК2
        && !Client::ClientFeatures::has_enable_dk2_in_api_feature($self->operator_client_id)) { # только для оператора без фичи
        return error_AccessToApiDeniedForApp_DC2;
    }

    return;
}


=head2 _is_direct_defect_error($error)

    Проверяем, что ошибка является объектом Direct::Defect

    if ($self->_is_direct_defect_error($error)) {
        return $error;
    } else {
        die $error;
    }

=cut

sub _is_direct_defect_error {
    my ($self, $error) = @_;

    return blessed($error) && $error->isa('Direct::Defect') ? 1 : 0;
}

=head2 _get_current_timestamp_iso8601()

    Метод возвращает текущеe время по UTC в формате YYYY-MM-DDThh:mm:ssZ

=cut

sub _get_current_timestamp_iso8601 {
    my ($self) = @_;

    return now_utc()->strftime('%FT%TZ'); # отдаем текущее время по UTC в формат YYYY-MM-DDThh:mm:ssZ
}

=head2 _get_camp_aggregated_last_processed_timestamp_iso8601()

    Метод возвращает время последнего обновления данных в таблице camp_aggregated_lastchange по UTC в формате YYYY-MM-DDThh:mm:ssZ
    или undef в случае некорректного значения свойства или его отсутствия

=cut

sub _get_camp_aggregated_last_processed_timestamp_iso8601 {
    my ($self) = @_;

    my $property_name = $API::Settings::CAMP_AGGREGATED_LASTCHANGE_LAST_TIMESTAMP_PROPERTY_PREFIX . get_shard(ClientID => $self->subclient_client_id);
    my $timestamp = Property->new($property_name)->get();
    if ($timestamp && $timestamp =~ /^(\d{4}-\d{2}-\d{2})T(\d{2}:\d{2}:\d{2})/) {
        $timestamp = mysql2iso8601($1.' '.$2, $local_timezone);
    } else {
        warn "wrong timestamp in $property_name property: ", $timestamp // "undef";
        $timestamp = undef;
    }

    return $timestamp;
}


=head2 _get_dictionaries_changes($mysql_timestamp)

    Метод возвращает ссылку на хэш, с информацией о том, были ли изменения в справочниках регионов и временных зон начиная с указанного в $mysql_timestamp момента

    $updated_dictionaries = _get_dictionaries_changes($changes_request->mysql_timestamp());

=cut

sub _get_dictionaries_changes {
    my ($self, $mysql_timestamp) = @_;

    return {} if $self->_is_timestamp_in_future($mysql_timestamp);

    my $changed_dictionaries = get_one_column_sql(PPCDICT, [ 'SELECT name FROM ppc_properties', WHERE => { name => [ qw/geo_regions_update_time geo_timezones_update_time/ ], value__ge => $mysql_timestamp } ]);
    my $result = { map { s/_update_time$//; $_ => 1 } @$changed_dictionaries };
    $result->{rmp_interest} = 1 if $self->_is_rmp_interest_last_change_more_then($mysql_timestamp);
    
    return $result;
}

=head2 _is_rmp_interest_last_change_more_then

    Проверяет появились ли изменения в справочнике интересов РМП

=cut

sub _is_rmp_interest_last_change_more_then {
    my ($self, $mysql_timestamp) = @_;
    
    my $is_more = get_one_field_sql(
        PPCDICT, [
            'SELECT 1 FROM targeting_categories',
            where => [ targeting_type => 'rmp_interest', state => 'Submitted', last_change__ge => $mysql_timestamp ]
        ]
    );
    return $is_more;
}

=head2 check($request)

    Метод позволяет получить информацию о наличии изменений в кампаниях, группах и баннерах после указанной даты
    request - ссылка на хэш, с данными запроса, запросы могут быть трех видов
    по кампаниям
    {
        CampaignIds => [13765,13766,13767,13769],
        FieldNames => ['CampaingIds', 'CampaignsStat', 'AdGroupIds', 'AdIds'],
        Timestamp => '2015-02-02T10:08:22Z'
    }
    по группам
    {
        AdGroupIds => [23764],
        FieldNames => ['AdGroupIds', 'AdIds'],
        Timestamp => '2015-02-02T10:08:22Z'
    }
    и по баннерам
    {
        AdIds => [33651,33732,34028,34029],
        FieldNames => ['AdIds'],
        Timestamp => '2015-02-02T10:08:22Z'
    }

    Пример использования:
    $changes->check(
        {
            CampaignIds => [13765,13766,13767,13769],
            FieldNames => ['CampaingIds', 'CampaignsStat', 'AdGroupIds', 'AdIds'],
            Timestamp => '2015-02-10T12:27:35Z'
        }
    );

    метод возвращает ссылку на хэш с результатами
    {
        Modified => {
            CampaingIds => [263,264,270],
            AdGroupIds => [26202,26203],
            AdIds => [2620101,2620102,2620103],
            CampaignsStat => [
                {
                    CampaignId => 262,
                    BorderDate => '2015-01-28'
                },
                {
                    CampaignId => 264,
                    BorderDate => '2015-01-28'
                }
            ],
        },
        NotFound => {
            CampaingIds => [265,266,301]
        },
        Timestamp => '2015-02-02T12:25:19Z'
    }

=cut

sub check {
    my ($self, $request) = @_;

    if (my $error = $self->_check_application_access()) {
        return $error;
    }

    my $response = {};
    try {
        my $changes_request = API::Service::Request::Changes->new($request);
    
        my $timestamp = $self->_get_current_timestamp_iso8601(); # получаем текущее время по UTC в формате YYYY-MM-DDThh:mm:ssZ
    
        my $data;
        if ($changes_request->has_cids) { # в запросе задан список кампаний
            $data = $self->_get_changes_for_cids($changes_request);
        } elsif ($changes_request->has_adgids) { # в запросе задан список групп
            $data = $self->_get_changes_for_adgids($changes_request);
        } else { # в запросе задан список баннеров
            $data = $self->_get_changes_for_bids($changes_request);
        }
    
        $data->{timestamp} = $timestamp;
        $response = $self->_generate_response_for_check($changes_request, $data);
    } catch {
        if ($self->_is_direct_defect_error($_)) { # обрабатываем исключения с объектом Direct::Defect
            $response = $_;
        } else {
            die $_; # остальные исключения пропагируем выше
        }
    };

    return $response;
}

=head2 _check_changes_for_cids($changes_request)

    Получаем изменения в объектах, типы которых указаны в $needed_fields, ограничиваясь идентификаторами, указанными в $uniq_cids начиная с даты $mysql_timestamp.
    Метод возвращает ссылку на хэш, состав ключей зависит от входных данных и наличия изменений
    {
        accessible_cids => [cid,...], # доступные оператору кампании
        not_found_ids => [cid,...], # недоступные оператору и ненайденные кампании
        min_unprocessed_cid => cid, # идентификатор кампании, начиная с которого из БД получили не все данные
        changed_cids => { cid => 1,... },
        changed_stat => { cid => BorderDate,... },
        cids2changed_adgids => { cid => [adgid,...],... },
        cids2changed_bids => { cid => [bid,...],... }
    }

    my $data = $self->_check_changes_for_cids($changes_request);

=cut

sub _get_changes_for_cids {
    my ($self, $changes_request) = @_;

    my $data = {};

    my $mysql_timestamp = $changes_request->mysql_timestamp(); # валидируем и получаем значение обязательного поля Timestamp

    ($data->{accessible_cids}, $data->{not_found_ids}) = $self->_split_inaccessible_cids($changes_request->uniq_cids()); # разделяем список кампаний, на доступные оператору и ненайденные, недоступные из-за прав в ответе никак не выделяем, возвращаем как ненайденные
    if (@{$data->{accessible_cids}} && !$self->_is_timestamp_in_future($mysql_timestamp)) {
        my $processign_cids = [ @{$data->{accessible_cids}} ];
        if ($changes_request->need_cids) {
            $data->{changed_cids} = $self->_get_changed_cids($processign_cids, $mysql_timestamp);
        }

        if ($changes_request->need_stat) {
            $data->{changed_stat} = $self->_get_campaigns_stat_changes($processign_cids, $mysql_timestamp);
        }

        if ($changes_request->need_adgids) {
            ($data->{cid2changed_adgids}, $data->{min_unprocessed_cid}) = $self->_get_cid2changed_adgids_for_cids($processign_cids, $mysql_timestamp);
        }

        if ($changes_request->need_bids) {
            @$processign_cids = grep { $_ < $data->{min_unprocessed_cid} } @$processign_cids if $data->{min_unprocessed_cid}; # убираем из списка необработанные кампании
            ($data->{cid2changed_bids}, my $unprocessed_cid) = $self->_get_cid2changed_bids($processign_cids, $mysql_timestamp);
            $data->{min_unprocessed_cid} = $unprocessed_cid if $unprocessed_cid && (!$data->{min_unprocessed_cid} || $unprocessed_cid < $data->{min_unprocessed_cid}); # выбираем минимальный идентификатор необработанных кампаний
        }
    }

    return $data;
}

sub _get_cid2changed_adgids_for_cids {
    my ($self, $cids, $mysql_timestamp) = @_;

    my ($cid2changed_adgids, $min_unprocessed_cid);
    my $changed_adgids =  get_hash_sql($self->ppc_shard, [ "SELECT pid, cid FROM phrases", WHERE => { cid__int => $cids, LastChange__ge => $mysql_timestamp }, "ORDER BY cid LIMIT $ADGROUP_IDS_COUNT_QUERY_LIMIT" ]);
    if (%$changed_adgids) {
        ($cid2changed_adgids, $min_unprocessed_cid) = $self->_delete_incomplete_block($changed_adgids, $ADGROUP_IDS_COUNT_QUERY_LIMIT);
    }

    return $cid2changed_adgids // {}, $min_unprocessed_cid;
}

sub _get_cid2changed_bids {
    my ($self, $cids, $mysql_timestamp) = @_;

    my ($cid2changed_bids, $unprocessed_cid);
    my $changed_bids = get_hash_sql($self->ppc_shard, [ "SELECT b.bid, cid FROM banners AS b LEFT JOIN banner_images AS bi USING (bid)", WHERE => { cid__int => $cids, _OR => { LastChange__ge => $mysql_timestamp, date_added__ge => $mysql_timestamp } }, "ORDER BY cid LIMIT $AD_IDS_COUNT_QUERY_LIMIT" ]);
    if (%$changed_bids) {
        ($cid2changed_bids, $unprocessed_cid) = $self->_delete_incomplete_block($changed_bids, $AD_IDS_COUNT_QUERY_LIMIT);
    }

    return $cid2changed_bids // {}, $unprocessed_cid;
}

=head2 _check_changes_for_adgids($changes_request)

    Получаем изменения в объектах, типы которых указаны в $needed_fields, ограничиваясь идентификаторами, указанными в $uniq_adgids начиная с даты $mysql_timestamp.
    Метод возвращает ссылку на хэш, состав ключей зависит от входных данных и наличия изменений
    {
        accessible_cids => [cid,...], # доступные оператору кампании
        cids2adgids => { cid => [adgid,...],... }, # списки групп по кампаниям
        not_found_ids => [cid,...], # недоступные оператору и ненайденные кампании
        min_unprocessed_cid => cid, # идентификатор кампании, начиная с которого из БД получили не все данные
        changed_cids => { cid => 1,... }, # список измененных кампаний
        changed_stat => { cid => BorderDate,... }, # изменения статистики в кампаниях
        cids2changed_adgids => { cid => [adgid,...],... }, # списки измененных групп по кампаниями
        cids2changed_bids => { cid => [bid,...],... } # списки измененных объявлений по кампаниями
    }

    my $data = $self->_check_changes_for_adgids($changes_request);

=cut

sub _get_changes_for_adgids {
    my ($self, $changes_request) = @_;

    my $data = {};

    my $mysql_timestamp = $changes_request->mysql_timestamp(); # валидируем и получаем значение обязательного поля Timestamp

    my $uniq_adgids = $changes_request->uniq_adgids();
    my $adgid2cid = $self->_get_adgid2cid($uniq_adgids);
    my $cid2adgids = $self->_reverse_hash($adgid2cid);
    $data->{accessible_cids} = $self->_filter_inaccessible_cids([ keys %$cid2adgids ]);
    $data->{accessible_cid2adgids} = hash_cut($cid2adgids, $data->{accessible_cids});
    my $accessible_adgids = [ map { @{$cid2adgids->{$_}} } @{$data->{accessible_cids}} ];
    if (@{$data->{accessible_cids}} && !$self->_is_timestamp_in_future($mysql_timestamp)) {
        if ($changes_request->need_cids) {
            $data->{changed_cids} = $self->_get_changed_cids($data->{accessible_cids}, $mysql_timestamp);
        }

        if ($changes_request->need_stat) {
            $data->{changed_stat} = $self->_get_campaigns_stat_changes($data->{accessible_cids}, $mysql_timestamp);
        }

        if ($changes_request->need_adgids) {
            my $accessible_adgid2cid = hash_cut($adgid2cid, $accessible_adgids);
            $data->{cid2changed_adgids} = $self->_get_cid2changed_adgids_for_adgids($accessible_adgid2cid, $mysql_timestamp);
        }

        if ($changes_request->need_bids) {
            ($data->{cid2changed_bids}, $data->{min_unprocessed_cid}) = $self->_get_cid2complete_changed_bids_for_adgids($accessible_adgids, $mysql_timestamp); # получаем информацию об измененных объявлениях (если из-за ограничения на SQL-запрос выбираются не все измененные объявления из кампании, они не возвращаются, и устанавливается начальный идентификатор невыбранных кампаний)
        }
    }

    $data->{not_found_ids} = xminus($uniq_adgids, $accessible_adgids); # недоступные клиенту группы в ответе никак не выделяем, возвращаем как ненайденные

    return $data;
}

sub _get_adgid2cid {
    my ($self, $adgids) = @_;

    my $adgid2cid = {};
    foreach my $adgids_chunk (chunks($adgids, SQL_WHERE_IDS_CHUNK_SIZE)) {
        hash_merge(
            $adgid2cid,
            get_hash_sql(
                $self->ppc_shard,
                [ "SELECT pid, cid FROM phrases", WHERE => { pid => $adgids_chunk } ]
            )
        );
    }

    return $adgid2cid;
}

sub _get_cid2changed_adgids_for_adgids {
    my ($self, $adgid2cid, $mysql_timestamp) = @_;

    my $cid2changed_adgids;
    my $changed_adgids = $self->_get_changed_adgids([ keys %$adgid2cid ], $mysql_timestamp);
    if (@$changed_adgids) {
        $cid2changed_adgids = $self->_filter_and_reverse_hash($adgid2cid, $changed_adgids);
    }

    return $cid2changed_adgids;
}

sub _get_changed_adgids {
    my ($self, $adgids, $mysql_timestamp) = @_;

    my @changed_adgids;
    foreach my $adgids_chunk (chunks($adgids, SQL_WHERE_IDS_CHUNK_SIZE)) {
        push(
            @changed_adgids,
            @{
                get_one_column_sql(
                    $self->ppc_shard,
                    [
                        'SELECT pid FROM phrases',
                        WHERE => {
                            pid__int => $adgids_chunk,
                            LastChange__ge => $mysql_timestamp
                        }
                    ]
                )
            }
        );
    }

    return \@changed_adgids;
}

sub _get_cid2complete_changed_bids_for_adgids {
    my ($self, $adgids, $mysql_timestamp) = @_;

    my $changed_bids = {};
    foreach my $adgids_chunk (chunks($adgids, SQL_WHERE_IDS_CHUNK_SIZE)) {
        hash_merge(
            $changed_bids,
            get_hash_sql(
                $self->ppc_shard,
                [
                    'SELECT b.bid, cid FROM banners AS b LEFT JOIN banner_images AS bi USING (bid)',
                    WHERE => {
                        pid__int => $adgids_chunk,
                        _OR => {
                            LastChange__ge => $mysql_timestamp,
                            date_added__ge => $mysql_timestamp
                        }
                    },
                    "ORDER BY cid LIMIT $AD_IDS_COUNT_QUERY_LIMIT"
                ]
            )
        );
    }

    if (%$changed_bids) {
        return $self->_delete_incomplete_block($changed_bids, $AD_IDS_COUNT_QUERY_LIMIT);
    }

    return undef, undef;
}

=head2 _check_changes_for_bids($changes_request)

    Получаем изменения в объектах, типы которых указаны в $needed_fields, ограничиваясь идентификаторами, указанными в $uniq_bids начиная с даты $mysql_timestamp.
    Метод возвращает ссылку на хэш, состав ключей зависит от входных данных и наличия изменений
    {
        accessible_cids => [cid,...], # доступные оператору кампании
        not_found_ids => [cid,...], # недоступные оператору и ненайденные кампании
        min_unprocessed_cid => cid, # идентификатор кампании, начиная с которого из БД получили не все данные
        changed_cids => { cid => 1,... },
        changed_stat => { cid => BorderDate,... },
        changed_adgids_by_cid => { cid => [adgid,...],... },
        changed_bids_by_cid => { cid => [bid,...],... }
    }

    my $data = $self->_check_changes_for_bids($changes_request);

=cut

sub _get_changes_for_bids {
    my ($self, $changes_request) = @_;

    my $data = {};

    my $mysql_timestamp = $changes_request->mysql_timestamp(); # валидируем и получаем значение обязательного поля Timestamp

    my $uniq_bids = $changes_request->uniq_bids();
    my $banners_data = $self->_get_banners_data($uniq_bids) ;
    my (%bid2cid, %cid2bids, %adgid2cid, %cid2adgids);
    for my $banner_data (@$banners_data) {
        $bid2cid{$banner_data->{bid}} = $banner_data->{cid};
        $adgid2cid{$banner_data->{pid}} //= $banner_data->{cid};
        push @{$cid2bids{$banner_data->{cid}}}, $banner_data->{bid};
        push @{$cid2adgids{$banner_data->{cid}}}, $banner_data->{pid} if $changes_request->need_adgids;
    }

    $data->{accessible_cids} = $self->_filter_inaccessible_cids([ keys %cid2bids ]);
    $data->{accessible_cid2bids} = hash_cut(\%cid2bids, $data->{accessible_cids});
    my $accessible_bids = [ map { @{$cid2bids{$_}} } @{$data->{accessible_cids}} ];
    if (@{$data->{accessible_cids}} && !$self->_is_timestamp_in_future($mysql_timestamp)) {
        if ($changes_request->need_cids) {
            $data->{changed_cids} = $self->_get_changed_cids($data->{accessible_cids}, $mysql_timestamp);
        }

        if ($changes_request->need_stat) {
            $data->{changed_stat} = $self->_get_campaigns_stat_changes($data->{accessible_cids}, $mysql_timestamp);
        }

        if ($changes_request->need_adgids) {
            my $accessible_adgid2cid = hash_grep(sub { exists $data->{accessible_cid2bids}{shift()} }, \%adgid2cid);
            $data->{cid2changed_adgids} = $self->_get_cid2changed_adgids_for_adgids($accessible_adgid2cid, $mysql_timestamp);
        }

        if ($changes_request->need_bids) {
            my $changed_bids = $self->_get_changed_bids($accessible_bids, $mysql_timestamp);
            if (@$changed_bids) {
                $data->{cid2changed_bids} = $self->_filter_and_reverse_hash(\%bid2cid, $changed_bids);
            }
        }
    }

    $data->{not_found_ids} = xminus($uniq_bids, $accessible_bids); # недоступные клиенту группы в ответе никак не выделяем, возвращаем как ненайденные

    return $data;
}

=head2 _get_banners_data($bids)

    Получаем из БД идентификаторы группы и кампании для указанных в $bids объявлений.
    Метод возвращает ссылку на массив вида
    [ { 'bid' => bid, 'pid' => adgid, 'cid' => cid },... ]

    $banners_data = $self->_get_banners_data($bids);

=cut

sub _get_banners_data {
    my ($self, $bids) = @_;

    my @banners_data;
    foreach my $bids_chunk (chunks($bids, SQL_WHERE_IDS_CHUNK_SIZE)) {
        push(
            @banners_data,
            @{
                get_all_sql(
                    $self->ppc_shard,
                    [ "SELECT bid, pid, cid FROM banners", WHERE => { bid => $bids_chunk } ]
                )
            }
        );
    }

    return \@banners_data;
}

=head2 _get_changed_bids($bids, $mysql_timestamp)

    Получаем из БД измененные начиная с указанной в $mysql_timestamp даты объявления из списка в $bids
    Метод возвращает ссылку на массив с идентификаторами измененных объявлений

    $changed_bids = $self->_get_changed_bids($bids, $mysql_timestamp);

=cut

sub _get_changed_bids {
    my ($self, $bids, $mysql_timestamp) = @_;

    my @changed_bids;
    foreach my $bids_chunk (chunks($bids, SQL_WHERE_IDS_CHUNK_SIZE)) {
        push(
            @changed_bids,
            @{
                get_one_column_sql(
                    $self->ppc_shard,
                    [
                        'SELECT bid FROM banners',
                        WHERE => {
                            bid__int => $bids_chunk,
                            LastChange__ge => $mysql_timestamp
                        }
                    ]
                )
            }
        );
    }

    return \@changed_bids;
}

=head2 _generate_response_for_check($changes_request, $data)

    Метод из хэша c информацией об изменениях (ссылка на хэш передается в $data) формирует хэш ответа сервиса и возвращает ссылку на него. В $changes_request передается объект запроса (API::Service::Request::Changes)

    $response = $self->_generate_response_for_check($changes_request, $data);

=cut

sub _generate_response_for_check {
    my ($self, $changes_request, $data) = @_;

    my $response;

    if ($data->{accessible_cids}) { # формируем выходные данные с учетом ограничения на количество идентификаторов в ответе
        my (@changed_cids, @changed_stat, @changed_adgids, @changed_bids, @unprocessed_ids);
        for my $cid (nsort @{$data->{accessible_cids}}) {
            if ($data->{min_unprocessed_cid} && $cid >= $data->{min_unprocessed_cid}) {
                push @unprocessed_ids, $changes_request->has_cids ? $cid : ($changes_request->has_adgids ? @{$data->{accessible_cid2adgids}{$cid}} : @{$data->{accessible_cid2bids}{$cid}});
                next;
            }

            if ($data->{changed_cids} && exists $data->{changed_cids}{$cid}){
                push @changed_cids, $cid;
            }

            if ($data->{changed_stat} && exists $data->{changed_stat}{$cid}) {
                push @changed_stat, { CampaignId => $cid, BorderDate => $data->{changed_stat}{$cid} };
            }

            if ($data->{cid2changed_adgids} && exists $data->{cid2changed_adgids}{$cid}) {
                push @changed_adgids, @{$data->{cid2changed_adgids}{$cid}};
            }

            if ($data->{cid2changed_bids} && exists $data->{cid2changed_bids}{$cid}) {
                push @changed_bids, @{$data->{cid2changed_bids}{$cid}};
            }
        }

        $response->{Modified}{CampaignIds} = \@changed_cids if $changes_request->need_cids;
        $response->{Modified}{CampaignsStat} = \@changed_stat if $changes_request->need_stat;
        $response->{Modified}{AdGroupIds} = [ nsort @changed_adgids ] if $changes_request->need_adgids;
        $response->{Modified}{AdIds} = [ nsort @changed_bids ] if $changes_request->need_bids;
        $response->{Unprocessed}{$changes_request->ids_field_name} = [ nsort @unprocessed_ids ] if @unprocessed_ids;
    }

    $response->{NotFound}{$changes_request->ids_field_name} = [ nsort @{$data->{not_found_ids}} ] if @{$data->{not_found_ids}};
    $response->{Timestamp} = $data->{timestamp};

    return $response;
}

=head2 _filter_and_reverse_hash($hash, $filtered_keys)

    Метод из хэша $hash оставляет только ключи, переданные в значении $filtered_keys и формирует обратный хэш, группируя ключи в массив по значению
    my $filtered_reverse_hash = $self->_filter_and_reverse_hash($hash, $filtered_keys);

=cut

sub _filter_and_reverse_hash {
    my ($self, $hash, $filtered_keys) = @_;

    my $result = {};
    for my $key (@$filtered_keys) {
        push @{$result->{$hash->{$key}}}, $key;
    }

    return $result;
}

=head2 _delete_incomplete_block($changed_ids, $limit)

    Eсли количество полученных в $changed_ids данных равно ограничению в SQL-запросе $limit, считаем, что выбраны не все данные в рамках кампании и удаляем их из результата.
    Хэш в $changed_ids может быть двух видов: { bid => cid,... } или { pid => cid,... }
    Возвращаем ссылку на хэш вида { cid => [bid,bid,...],... } или { cid => [pid,pid,...],... } в зависимости от $changed_ids

=cut

sub _delete_incomplete_block {
    my ($self, $changed_ids, $limit) = @_;

    my $changed_ids_by_cid = $self->_reverse_hash($changed_ids);
    my $unprocessed_cid;
    if (keys %{$changed_ids} == $limit) { # если запрос вернул максимальное число значений, удаляем баннеры с максимальным cid, т.к. нет гарантии, что выбрались все баннеры этой кампании
        $unprocessed_cid = max keys %{$changed_ids_by_cid};
        delete $changed_ids_by_cid->{$unprocessed_cid}; # удаляем неполные данные
    }

    return $changed_ids_by_cid, $unprocessed_cid;
}

=head2 _filter_inaccessible_cids($cids)

    Отфильтровываем из списка кампаний недоступные (незавершенные, отсутствующие в БД и недоступные из-за прав)

    $filtered_cids = $self->_filter_inaccessible_cids($cids);

=cut

sub _filter_inaccessible_cids {
    my ($self, $cids) = @_;

    my ($filtered_cids) = $self->_split_inaccessible_cids($cids);

    return $filtered_cids;
}

=head2 _split_inaccessible_cids($cids)

    Разделяем список кампаний на доступные и недоступные (незавершенные, отсутствующие в БД и недоступные из-за прав)
    Метод возвращает два списка идентификаторов кампаний

    ($filtered_cids, $inaccessible_cids) = $self->_split_inaccessible_cids($cids);

=cut

sub _split_inaccessible_cids {
    my ($self, $cids) = @_;
    return [] unless $cids && @$cids;

    my ($filtered_cids, $inaccessible_cids);
    my %restricted_cids = ($self->check_campaigns_read(@$cids), $self->get_unavailable_cids_map($cids, supported_camp_kind => SUPPORTED_CAMP_KIND, pass_archived => 1));  # получаем список недоступных кампаний
    $inaccessible_cids = [ keys %restricted_cids ];
    $filtered_cids = xminus($cids, $inaccessible_cids);

    return ($filtered_cids, $inaccessible_cids);
}

=head2 _get_changed_cids($cids, $mysql_timestamp)

    Метод возвращает отфильтрованный список $cids, измененных не ранее указанной в $mysql_timestamp даты

    $changed_cids = $self->_get_changed_cids($accessible_cids, $mysql_timestamp);

=cut

sub _get_changed_cids {
    my ($self, $cids, $mysql_timestamp) = @_;

    my $changed_cids = get_one_column_sql($self->ppc_shard, [ 'SELECT cid FROM campaigns', WHERE => { cid__int => $cids, type__not_in => ['wallet'], LastChange__ge => $mysql_timestamp } ]);

    return { map { $_ => 1 } @$changed_cids };
}

=head2 _reverse_hash($hash)

    Меняем местами ключи и значения хеша, ключи сохраняются в виде массива, чтобы не потерять их для дублирующихся значений
    для $self->_reverse_hash({ 1 => 10, 2 => 17, 3 => 10, 4 => 15 }) получим { 17 => [ 2 ], 10 => [ 1, 3 ], 15 => [ 4 ] }

=cut

sub _reverse_hash {
    my ($self, $hash) = @_;

    my $reversed_hash = {};
    while (my ($key, $value) = each %$hash) {
        push @{$reversed_hash->{$value}}, $key;
    }

    return $reversed_hash;
}

=head2 _get_campaigns_stat_changes($cids, $mysql_timestamp)

    Получаем сведения о наличии изменений в статистике кампаний, заданных в массиве $cids после указанной в $mysql_timestamp даты. Метод возвращает ссылку на хэш вида { cid => border_date }, где border_date - дата актуальности статистики для указанной кампании: { 10 => '2015-02-28' }

    $changed_stat = $self->_get_campaigns_stat_changes($accessible_cids, $mysql_timestamp);

=cut

sub _get_campaigns_stat_changes {
    my ($self, $cids, $mysql_timestamp) = @_;

    return {} unless $cids && @$cids;

    my $orderid2cid = get_hash_sql($self->ppc_shard, [ "SELECT OrderID, cid FROM campaigns", WHERE => { OrderID__gt => 0, cid => $cids } ]);

    my $stat_updates = get_all_sql(PPCDICT, [ 'SELECT OrderID, MIN(border_date) AS border_date FROM stat_rollbacks', WHERE => { OrderID => [ keys %$orderid2cid ], rollback_time__ge => $mysql_timestamp }, 'GROUP by OrderID' ]);

    return { map { $orderid2cid->{$_->{OrderID}} => $_->{border_date} } @$stat_updates };
}

=head2 checkCampaigns($request)

    Метод позволяет получить информацию о наличии изменений по всем кампаниям (а также в статистике) и дочерним объектам (группам и баннерам, но без уточнения) клиента после указанной даты
    self - объект сервиса
    request - ссылка на хэш, с данными запроса
    {
        Timestamp => '2015-02-02T10:08:22Z'
    }

    Пример использования:
    $changes->checkCampaigns({ Timestamp => '2015-02-10T12:27:35Z' });

    метод возвращает ссылку на хэш с идентификатором кампании и указанием типа изменений (SELF - в кампании, CHILDREN - в дочерних объектах, STAT - в статистике)
    {
        Campaigns => [
            {
                CampaignId => 262,
                ChangesIn  => [ 'CHILDREN', 'STAT' ]
            },
            {
                CampaignId => 264,
                ChangesIn  => [ 'SELF', 'CHILDREN', 'STAT' ]
            }
        ],
        Timestamp        => '2015-02-02T12:25:19Z'
    }

=cut

sub checkCampaigns {
    my ($self, $request) = @_;

    if (my $error = $self->_check_application_access()) {
        return $error;
    }

    my $response = {};
    try {
        my $changes_request = API::Service::Request::Changes->new($request);
        my $mysql_timestamp = $changes_request->mysql_timestamp(); # валидируем и получаем значение обязательного поля Timestamp

        if ($self->_need_use_camp_aggregated_lastchange) {
            $response->{Timestamp} = $self->_get_camp_aggregated_last_processed_timestamp_iso8601();
        }
        $response->{Timestamp} //= $self->_get_current_timestamp_iso8601(); # получаем текущее время по UTC в формате YYYY-MM-DDThh:mm:ssZ
    
        # в общем методе показываем изменения в том числе и по удаленным кампаниям
        my $cids_states = $self->_is_timestamp_in_future($mysql_timestamp) ? [] : $self->_get_client_cids_and_states($mysql_timestamp);
        my $client_cids = [ map { $_->{cid} } @$cids_states ];
        my %inaccessible_cids;
        if (@$client_cids) {
        %inaccessible_cids = (
            $self->check_campaigns_read(@$client_cids),
            $self->get_unavailable_cids_map($client_cids, supported_camp_kind => SUPPORTED_CAMP_KIND,  pass_archived => 1)
        ); # получаем идентификаторы кампаний клиента недоступные оператору
        }
    
        my (%campaign_changes, %orderid2cid, @cids);
        for my $cid_state (@$cids_states) {
            next if $inaccessible_cids{$cid_state->{cid}}; # не обрабатываем кампании, недоступные оператору
            push @cids, $cid_state->{cid};
            $orderid2cid{$cid_state->{OrderID}} = $cid_state->{cid} if $cid_state->{OrderID};
            if ($cid_state->{updated}) {
                ($campaign_changes{$cid_state->{cid}} = API::Service::Changes::CampaingChangeTypes->new())->enable_self(); # для хранения типов изменений используем вспомогательный объект
            }
        }
    
        if (@cids) {
            my $cids_with_changed_children;
            if ( $self->_need_use_camp_aggregated_lastchange() ) {
                $cids_with_changed_children = $self->_get_cids_with_changed_children(\@cids, $mysql_timestamp);
            } else {
                $cids_with_changed_children = $self->_get_cids_with_changed_adgroups(\@cids, $mysql_timestamp);
            my $cids_with_unchanged_children = xminus(\@cids, $cids_with_changed_children);
            if (@$cids_with_unchanged_children) {
                push @$cids_with_changed_children, @{$self->_get_cids_with_changed_banners($cids_with_unchanged_children, $mysql_timestamp)};
            }
            }
    
            for my $cid (@$cids_with_changed_children) {
                ($campaign_changes{$cid} //= API::Service::Changes::CampaingChangeTypes->new())->enable_children();
            }

            if (%orderid2cid) {
                my $cids_with_stat_changes = $self->_get_cids_with_stat_changes(\%orderid2cid, $mysql_timestamp);
                for my $cid (@$cids_with_stat_changes) {
                    ($campaign_changes{$cid} //= API::Service::Changes::CampaingChangeTypes->new())->enable_stat();
                }
            }
        }
    
        $response->{Campaigns} = [ map { { CampaignId => $_, ChangesIn => $campaign_changes{$_}->list() }; } nsort keys %campaign_changes ];
    } catch {
        if ($self->_is_direct_defect_error($_)) { # обрабатываем исключения с объектом Direct::Defect
            $response = $_;
        } else {
            die $_; # остальные исключения пропагируем выше
        }
    };

    return $response;
}

=head2 _need_use_camp_aggregated_lastchange()

    Метод возвращает признак того, что необходимо использовать данные из таблицы ppc.camp_aggregated_lastchange.
    В проде использование данных из таблицы ppc.camp_aggregated_lastchange регулируется опцией use_camp_aggregated_lastchange
    На разработческих средах дополнительно поведением можно управлять через HTTP-заголовок useCampAggregatedLastChange.

=cut

sub _need_use_camp_aggregated_lastchange {
    my ($self) = @_;

    my $use_camp_aggregated_last_change = !$API::Version::is_production ? $self->get_http_request_header('useCampAggregatedLastChange') : undef;

    if (defined $use_camp_aggregated_last_change) {
        if ($use_camp_aggregated_last_change eq 'true') {
            return 1;
        } elsif ($use_camp_aggregated_last_change eq 'false') {
            return 0;
        }
        warn "unknown value '$use_camp_aggregated_last_change' is specified in header useCampAggregatedLastChange";
    }

    return $property->get($API::Settings::USE_CAMP_AGGREGATED_LASTCHANGE_PROPERTY_TTL);
}

=head2 _get_client_cids_and_states($mysql_timestamp)

    Получаем идентификаторы кампаний клиента, OrderID и флаг наличия изменений начиная с указанного в $mysql_timestamp времени.
    Метод возвращает ссылку на массив вида
    [ { 'cid' => cid, 'OrderID' => OrderID, 'updated' => 1 или 0 },... ]

    $cids_states = $self->_get_client_cids_and_states($mysql_timestamp);

=cut

sub _get_client_cids_and_states {
    my ($self, $mysql_timestamp) = @_;

    return get_all_sql($self->ppc_shard, [ 'SELECT cid, OrderID, IF(LastChange >= ?,1,0) AS updated FROM campaigns', WHERE => { uid => $self->subclient_uid, type__not_in => ['wallet'] } ], $mysql_timestamp);
}

=head2 _get_cids_with_changed_adgroups($cids, $mysql_timestamp)

    Получаем идентификаторы кампаний из списка в $cids, в которых произошли изменения в группах, начиная с указанного в $mysql_timestamp времени.
    Метод возвращает ссылку на массив идентификаторов кампаний

    $cids_with_changed_adgroups = $self->_get_cids_with_changed_adgroups($cids, $mysql_timestamp);

=cut

sub _get_cids_with_changed_adgroups {
    my ($self, $cids, $mysql_timestamp) = @_;

    return get_one_column_sql($self->ppc_shard, [ 'SELECT DISTINCT cid FROM phrases', WHERE => { cid => $cids, LastChange__ge => $mysql_timestamp } ]);
}

=head2 _get_cids_with_changed_banners($cids, $mysql_timestamp)

    Получаем идентификаторы кампаний из списка в $cids, в которых произошли изменения в баннерах, начиная с указанного в $mysql_timestamp времени.
    Метод возвращает ссылку на массив идентификаторов кампаний

    $cids_with_changed_banners = $self->_get_cids_with_changed_banners($cids, $mysql_timestamp);

=cut

sub _get_cids_with_changed_banners {
    my ($self, $cids, $mysql_timestamp) = @_;

    return get_one_column_sql($self->ppc_shard, [ 'SELECT DISTINCT cid FROM banners LEFT JOIN banner_images USING (bid)', WHERE => { cid => $cids, _OR => { LastChange__ge => $mysql_timestamp, date_added__ge => $mysql_timestamp } } ]);
}

=head2 _get_cids_with_stat_changes($orderid2cid, $mysql_timestamp)

    Получаем идентификаторы кампаний из списка в $orderid2cid, в которых произошли изменения в статистике, начиная с указанного в $mysql_timestamp времени.
    Метод возвращает ссылку на массив идентификаторов кампаний

    $cids_with_changed_banners = $self->_get_cids_with_changed_banners($cids, $mysql_timestamp);

=cut

sub _get_cids_with_stat_changes {
    my ($self, $orderid2cid, $mysql_timestamp) = @_;

    my $order_ids = get_one_column_sql(PPCDICT, [ 'SELECT DISTINCT OrderID FROM stat_rollbacks', WHERE => { OrderID => [ keys %$orderid2cid ], rollback_time__ge => $mysql_timestamp } ]);
    return [ map {$orderid2cid->{$_}} @$order_ids ];
}

=head2 _get_cids_with_changed_children($cids, $mysql_timestamp)

    Метод возвращает ссылку на массив идентификаторов кампаний, в которых были изменения подобъектов начиная с указанной в $mysql_timestamp даты

=cut

sub _get_cids_with_changed_children {
    my ($self, $cids, $mysql_timestamp) = @_;

    return get_one_column_sql(
        $self->ppc_shard,
        [
            'SELECT cid FROM camp_aggregated_lastchange',
            WHERE => {
                cid => $cids,
                last_change__ge => $mysql_timestamp
            }
        ]
    );
}

=head2 _is_timestamp_in_future($mysql_timestamp)

    Возвращает признак того, что дата и время, переданные в $mysql_timestamp еще не наступили

=cut

sub _is_timestamp_in_future {
    my ($self, $mysql_timestamp) = @_;

    return mysql2unix($mysql_timestamp) > time() ? 1 : 0;
}

1;
