package API::Service::Reports;

use Direct::Modern;

use JSON;
use List::MoreUtils qw/uniq/;
use List::Util qw/any all/;
use Scalar::Util qw/blessed/;
use Try::Tiny;

use HashingTools qw/md5_hex_utf8/;
use Yandex::Clone qw/yclone/;
use Yandex::DateTime qw/date/;
use Yandex::DBTools;
use Yandex::DBShards qw/SHARD_IDS/;
use Yandex::HashUtils qw/hash_copy hash_cut/;
use Yandex::ListUtils qw/xisect/;
use Yandex::I18n qw/iget/;
use Yandex::Memcached::Lock;
use Yandex::Redis::RateLimit;
use Yandex::Trace;

use API::Reports::Builder;

use API::Reports::DataRules qw/
    AD_NETWORK_TYPE_INTERNAL_FIELD
    AD_NETWORK_TYPE_SEARCH
    CUSTOM_DATE_RANGE_TYPE
    QUERY_INTERNAL_FIELD
    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE
    ADGROUP_PERFORMANCE_REPORT_TYPE
    ACCOUNT_PERFORMANCE_REPORT_TYPE
    REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE
    %OPERATORS_MAP_TO_INTERNAL
    %FIELDS_DEPENDENT_ON_GOAL_MAP
    %ATTRIBUTION_MODEL_TYPE_MAP
/;
use API::Service::Reports::ConvertSubs qw/
    convert_field_names_and_report_type_to_group_by
    convert_field_names_to_date_aggregation
    convert_field_names_to_countable_fields
    convert_field_names_to_report_fields
    get_dates_by_range
    convert_page_filter
    convert_page_name_filters
    convert_filter_items
    convert_order_by_items
/;

use API::Reports::FormatTsv 'tsv_line';
use API::Reports::InternalRequestRepresentation;
use API::Reports::OfflineReportTask;
use API::Reports::OfflineReportTaskParameters;

use API::Service::Reports::ProcessingModeChooser;
use API::Service::Reports::Validation;
use API::Service::Campaigns::Types qw/convert_types/;

use API::Settings;
use Campaign::Types qw/get_camp_kind_types/;
use CampaignTools qw/mass_get_is_uac_campaign_by_order_ids/;
use Client qw/get_client_currencies/;
use Client::ClientFeatures qw//;
use Direct::Errors::Messages;
use DirectRedis;
use EnvTools qw/is_beta/;
use PrimitivesIds qw/get_cid2orderid/;
use RBACDirect qw/rbac_check_allow_show_camps/;
use Settings;
use PrimitivesIds qw/get_clientids/;
use Rbac qw/:const get_perminfo/;

use base qw/API::Service::Base/;

use constant API_CONSUMER_NAME => 'api';
use constant TRUE => 1;
use constant SUPPORTED_CAMP_KIND => 'stat_stream_ext';

our $ENABLE_OCCUPIED_WORKERS_PER_SERVER_LIMIT //= is_beta() ? 0 : 1;

my $OCCUPIED_WORKERS_PER_SERVER_LOCK_NAME = "locks/api5/reports/$EnvTools::hostname";

# сколько процессов на одном сервере могут обрабатывать запросы к reports
# рекомендуется 1/3 от MaxClients апача
# какой MaxClients апача, см. etc/soap/apache2/soap.direct.yandex.ru.conf
our $OCCUPIED_WORKERS_PER_SERVER_LIMIT //= 50;

our $DEFAULT_REPORT_ROWS_COUNT_LIMIT //= 1000000;
our $DEFAULT_MULTICLIENT_REPORT_ROWS_COUNT_LIMIT //= 500000;

sub include_request_headers_in_logs {
    return 1;
}

=head2 add_request_headers_to_log()

    Добавляем в логи необходимые заголовки запроса

=cut

sub add_request_headers_to_log {
    my ($self) = @_;

    $self->http_request_headers_for_log->processingMode($self->get_http_request_header('processingMode'));
    $self->http_request_headers_for_log->returnMoneyInMicros($self->get_http_request_header('returnMoneyInMicros'));
    $self->http_request_headers_for_log->skipReportHeader($self->get_http_request_header('skipReportHeader'));
    $self->http_request_headers_for_log->skipColumnHeader($self->get_http_request_header('skipColumnHeader'));
    $self->http_request_headers_for_log->skipReportSummary($self->get_http_request_header('skipReportSummary'));

    return;
} 

=head2 create($request)

    Метод для создания отчета

=cut

sub create {
    my ($self, $request_data) = @_;

    $self->add_request_headers_to_log();    

    state $rl = Yandex::Redis::RateLimit->new(
        time_interval => $API::Settings::API_REPORTS_RATE_LIMIT_INTERVAL_SEC,
        max_requests => $API::Settings::API_REPORTS_RATE_LIMIT_MAX_REQUESTS,
        name => 'api5-reports-requests-number',
        redis => DirectRedis::get_redis() 
    );
    my $is_limit_error;
    try {
        if (!$self->is_operator_super && !$rl->request_ok( $self->subclient_client_id )) {
            $is_limit_error = 1;
        }
    } catch {
        warn "Failed to check API report rate limit: ", $_;
    };
    if ($is_limit_error) {
        return error_RateLimitExceeded(
            iget("Данный метод можно запрашивать не чаще чем %d раз в %d секунд", $rl->max_requests, $rl->time_interval)
        );
    }

    my $mc_lock;
    if ($ENABLE_OCCUPIED_WORKERS_PER_SERVER_LIMIT) {
        $mc_lock = Yandex::Memcached::Lock->new(
            servers => $Settings::MEMCACHED_SERVERS,
            max_locks_num => $OCCUPIED_WORKERS_PER_SERVER_LIMIT,
            entry => $OCCUPIED_WORKERS_PER_SERVER_LOCK_NAME,
        );

        unless ( $mc_lock->get_lock() ) {
            die "Could not get a memcached lock, ran out of workers on this server?";
        }
    }

    my $original_request_data = yclone($request_data);
    $self->_populate_request_data($request_data);

    my $error = $self->_validate_create($request_data);
    return $error if $error;

    my $micros_header = $self->get_http_request_header('returnMoneyInMicros');

    my $request = $self->convert_client_request_to_internal_representation($request_data,
        return_money_in_micros => (!$micros_header || $micros_header ne 'false'),
        lang => Yandex::I18n::current_lang(),
        skip_report_header => ($self->get_http_request_header('skipReportHeader') // '') eq 'true',
        skip_column_header => ($self->get_http_request_header('skipColumnHeader') // '') eq 'true',
        skip_report_summary => ($self->get_http_request_header('skipReportSummary') // '') eq 'true',
    );

    return $request if $request->isa('Direct::Defect');

    _replace_audience_target_id($request);

    my $chosen_processing_mode = API::Service::Reports::ProcessingModeChooser->choose_processing_mode(
        $self->application_id,
        scalar $self->get_http_request_header('processingMode'),
        $request_data, $request );

    return $chosen_processing_mode if blessed($chosen_processing_mode) && $chosen_processing_mode->isa('Direct::Defect');

    my $queue = Yandex::DBQueue->new( PPC( ClientID => $self->subclient_client_id ), API::Reports::OfflineReportTask::DBQUEUE_JOB_TYPE );
    my $reports_count = $queue->count_jobs( ClientID => $self->subclient_client_id );

    ## офлайн-отчёты
    if ( $chosen_processing_mode == API::Service::Reports::ProcessingModeChooser::PROCESSING_MODE_OFFLINE ) {

        my ($enqueue_error, $task);
        my $request_data_md5 = md5_hex_utf8(
            to_json(
                {
                    rdef => $original_request_data,
                    return_money_in_micros => $request->return_money_in_micros,
                    $request->skip_report_header ? (skip_report_header => 1) : (),
                    $request->skip_column_header ? (skip_column_header => 1) : (),
                    $request->skip_report_summary ? (skip_report_summary => 1) : (),
                    lang => $request->lang
                },
                { canonical => 1 }
            )
        );

        try {
            $task = API::Reports::OfflineReportTask->get_or_create(
                API::Reports::OfflineReportTaskParameters->new(
                    client_id => $self->subclient_client_id,
                    report_name => $request->report_name,
                    job_args => $request->to_plain_struct,
                    request_data_md5 => $request_data_md5,
                    dont_check_limit => $self->is_operator_super
                )
            );
            $reports_count ++ if $task->just_created;

        } catch {
            my $error = $_;
            if ( blessed $error && $error->isa('API::Reports::OfflineReportException') ) {
                $enqueue_error = $error->defect;
                if ( $error->isa('API::Reports::QueueLimitExceededException') ) {
                    $reports_count = $error->reports_count;
                } 
            } else {
                die $error;
            }
        };
        $self->http_response_headers->reportsInQueue($reports_count);

        return $enqueue_error if $enqueue_error;

        my $job = $task->dbqueue_job;
        if ( $job->is_finished ) {
            my $result = $task->dbqueue_job->result;
            if ($result->{is_empty_report}) {
                return {
                    type => 'tsv',
                    callback => sub {},
                    meta => { stat_source => $result->{stat_source} },
                };

            } else {
                my $file = $task->get_resulting_file;

                my $url = $file->internal_redirect_url(
                    'Content-Type' => 'text/tab-separated-values; charset=utf-8',
                    'RequestId' => Yandex::Trace::current_span_id(),
                    'reportsInQueue' => $reports_count,
                );

                return {
                    type => 'mds',
                    url => $url,
                    job_id => $task->dbqueue_job_id,
                    meta => { stat_source => $result->{stat_source} },
                };
            }

        } elsif ( $job->is_failed || $job->is_revoked ) {
            ## revoked быть не должно, но если есть, что-то пошло не так, отдадим клиенту ошибку
            my $result = '[none]';
            if ( $job->has_result && defined $job->result ) {
                $result = to_json( $job->result );
            }
            warn "failed report: job status = " . $job->status . ", job_id = " . $job->job_id, ", result = $result";
            $self->_delete_report_from_db($task);
            return error_OperationFailed(iget("Внутренняя ошибка при построении отчёта"));
        }

        my $interval = $task->just_created ? 1 : 10;
        $self->http_response_headers->retryIn($interval);

        return {
            type => $task->just_created ? 'queued' : 'not_ready',
            body => '',
            job_id => $task->dbqueue_job_id,
        };
    }
    
    my $result = API::Reports::Builder->build_report($request);

    $self->http_response_headers->reportsInQueue($reports_count);

    if ( $result->format eq 'tsv' ) {
        return {
            type => 'tsv',
            callback => sub {
                my ($writer) = @_;

                my $iterator = $result->data_iterator;

                $writer->write( $self->_tsv_line( [ $result->title ] ) ) if !$request->skip_report_header;
                $writer->write( $self->_tsv_line( $result->header ) ) if !$request->skip_column_header;

                my $row_count = 0;
                while ( $iterator->has_next_row ) {
                    $writer->write( $self->_tsv_line( $iterator->next_row ) );
                    $row_count++;
                }

                # добавляем последнюю стоку с количеством строк отчета
                $writer->write( $self->_tsv_line( ["Total rows: $row_count"] ) ) if !$request->skip_report_summary;
            },
            meta => { stat_source => $result->stat_source },
        }
    }
}

=head2 _replace_audience_target_id($request)

    DIRECT-94500
    Добавляем в фильтрацию по AudienceTargetId поле criterion_type чтоб не показывать другие типы объектов(например ключевые слова)

=cut

sub _replace_audience_target_id {
    my ($request) = @_;
    if ($request->{provider_request_params}{filter} && $request->{provider_request_params}{filter}{bs_criterion_id}) {
        $request->{provider_request_params}{filter}{criterion_type} = { eq => ["RETARGETING", "MOBILE_APP_CATEGORY", "INTERESTS_AND_DEMOGRAPHICS"] };
    }
}

=head2 _delete_report_from_db($task)

    Удалить данные об отчёте из ppc.api_reports_offline

=cut

sub _delete_report_from_db {
    my ($self, $task) = @_;

    do_sql( PPC(ClientID => $task->ClientID), "DELETE FROM api_reports_offline WHERE ClientID = ? and ReportName = ?",
        $task->ClientID, $task->ReportName );
}

=head2 _convert_request_data_to_report_params($request_data)

    Из данных запроса формируем структуру с параметрами для ядра

=cut

sub _convert_request_data_to_report_params {
    my ($self, $request_data, $lang) = @_;

    my %report_params;

    my $cids = $request_data->{filter}{cid}{eq};
    my $cid2order_ids = get_cid2orderid(cid => $cids);
    my $OrderIds = [ map { exists $cid2order_ids->{$_} ? $cid2order_ids->{$_} : () } @$cids ];
    $report_params{oid} = $OrderIds;
    my $uac_oids = mass_get_is_uac_campaign_by_order_ids($OrderIds);
    $report_params{uac_oid} = [keys %$uac_oids];

    hash_copy(\%report_params, $request_data, qw/filter/);
    delete $report_params{filter}{cid};
    if (exists $report_params{filter}{page}) {
        if (Client::ClientFeatures::has_mol_page_name_support($self->subclient_client_id)
            || Client::ClientFeatures::has_mol_page_name_support($self->operator_client_id)
        ) {
            $report_params{filter}{page_name} = delete $report_params{filter}{page};
        } else {
            $report_params{filter}{page} = convert_page_filter($report_params{filter}{page}, $lang);
        }
    }
    if (exists $report_params{filter}{page_name}) {
        # фильтры по названию площадки (Placement) с операторами eq/ne всегда работали как contains/not_contains
        $report_params{filter}{page_name} =
            convert_page_name_filters($report_params{filter}{page_name});
    }

    $report_params{translocal_params} = { tree => 'api' }; # указываем API-шное транслокальное дерево
    $report_params{ClientID_for_stat_experiments} = $self->subclient_client_id;

    # дополняем список регионов подрегионами
    foreach my $region_key (qw/region_id physical_region/) {
        next if !exists $report_params{filter}{$region_key};
        my $region_filter = $report_params{filter}{$region_key};
        foreach my $operator (keys %$region_filter) {
            $region_filter->{$operator} = Stat::Tools::get_plus_regions_by_geo($region_filter->{$operator}, $report_params{translocal_params});
        }
    }

    if ($request_data->{date_range_type} eq CUSTOM_DATE_RANGE_TYPE) {
        hash_copy(\%report_params, $request_data, qw/start_date end_date/);
    } else {
        my @dates = get_dates_by_range($request_data->{date_range_type}, undef, $OrderIds);
        @report_params{qw/start_date end_date/} = @dates if @dates;
    }

    my $with_winrate = ( any { $_ eq 'ImpressionShare' } @{ $request_data->{field_names} } ) ||
        exists $report_params{filter}->{winrate};

    $report_params{options} = {
        without_totals => 1, # сообщаем БК, что totals присылать не нужно
        consumer => API_CONSUMER_NAME, # задаем использование в словарных полях, значений для API
        filter_by_consumer_values => TRUE, # задаем поддержку в словарных полях в фильтрации значений для API
        four_digits_precision => Client::ClientFeatures::has_stat_4_digits_precision_feature($self->subclient_client_id),
        with_winrate => $with_winrate,
        with_discount => 0,
        operator_ClientID => $self->operator_client_id,
    };
    hash_copy($report_params{options}, $request_data, qw/with_nds currency/);

    $report_params{limits} = hash_copy({}, $request_data, qw/order_by/);
    hash_copy($report_params{limits}, $request_data->{page}, qw/limit offset/) if exists $request_data->{page};
    $report_params{limits}{limit} //= (exists $report_params{filter}{client_id}) ? $DEFAULT_MULTICLIENT_REPORT_ROWS_COUNT_LIMIT : $DEFAULT_REPORT_ROWS_COUNT_LIMIT;

    my $field_names = $request_data->{field_names};
    $report_params{group_by} = convert_field_names_and_report_type_to_group_by($field_names, $request_data->{report_type});
    $report_params{date_aggregation_by} = convert_field_names_to_date_aggregation($field_names);

    $report_params{options}{extra_countable_fields} = []; 
    my $exists_fields_dependent_on_goal = xisect($field_names, [ keys %FIELDS_DEPENDENT_ON_GOAL_MAP ]);
    if (scalar @$exists_fields_dependent_on_goal && exists $request_data->{filter}{goal_ids}) {
        push @{ $report_params{options}{extra_countable_fields} }, map { $FIELDS_DEPENDENT_ON_GOAL_MAP{$_} } @$exists_fields_dependent_on_goal;
    }
    delete $report_params{options}{extra_countable_fields} unless @{ $report_params{options}{extra_countable_fields} };

    my $countable_fields = convert_field_names_to_countable_fields($field_names, $request_data->{report_type});
    if (@$countable_fields) {
        $report_params{options}{extra_countable_fields} //= [];
        push @{$report_params{options}{extra_countable_fields}}, @$countable_fields;
    }

    return \%report_params;
}

=head2 convert_client_request_to_internal_representation
    opts:
        return_money_in_micros - возвращать денежные значения в микроединицах,
        lang - язык отчета (для названий регионов и др. полей, значения в которых зависят от языка)
        skip_report_header - не возвращать в отчете строку с заголовком
        skip_column_header - не возвращать в отчете строку с названиями столбцов
        skip_report_summary - не возвращать в конце отчета строку с количеством строк отчета

    NB: все опции, которые влияют на содержание отчета, должны быть использованы при расчете контрольной суммы
    отчета в коде формирования оффлайн-отчета
=cut

sub convert_client_request_to_internal_representation {
    my ( $self, $client_request, %opts ) = @_;

    my $converter = $self->converter('internal');
    my $converted_request_data = $converter->convert($client_request);
    my $report_type = $converted_request_data->{report_type};

    if ($converted_request_data->{goal_ids}) {
        if ( _has_fields_dependent_on_goal($converted_request_data->{field_names}) ){
            $converted_request_data->{filter} //= [];
            push $converted_request_data->{filter}, delete $converted_request_data->{goal_ids};
            if ($converted_request_data->{attribution_models}) {
                push $converted_request_data->{filter}, delete $converted_request_data->{attribution_models};
            } else {
                push $converted_request_data->{filter}, {field => "AttributionModels", operator => "IN", values =>["LSC"]};
            }
        }
    }
    $converted_request_data->{filter} = convert_filter_items($converted_request_data->{filter}, $report_type)
        if exists $converted_request_data->{filter}; 

    if (exists $converted_request_data->{order_by}) {
        $converted_request_data->{order_by} = convert_order_by_items($converted_request_data->{order_by}, $report_type)
    }

    if ($report_type eq SEARCH_QUERY_PERFORMANCE_REPORT_TYPE) {
        # добавляем фильтрацию записей с пустым поисковым запросом
        push @{$converted_request_data->{filter}{+QUERY_INTERNAL_FIELD}{ne}}, $Stat::Const::SIGNIFICANT_EMPTY_STRING;
        # в отчёте SEARCH_QUERY_PERFORMANCE_REPORT учитываются только показы рекламы на поисках, поэтому добавляем фильтрацию по РСЯ
        $converted_request_data->{filter}{+AD_NETWORK_TYPE_INTERNAL_FIELD}{eq} = AD_NETWORK_TYPE_SEARCH;
    }

    my $cids = $converted_request_data->{filter}{cid}{eq};
    my $client_ids;
    if ($converted_request_data->{filter}{client_login}) {
        $client_ids = $self->_get_accessible_client_ids($converted_request_data->{filter}{client_login}{eq});
        delete $converted_request_data->{filter}{client_login};
        $converted_request_data->{filter}{client_id} = { eq => $client_ids };
    } else {
        $client_ids = [ $self->subclient_client_id ];
        if (any { $_ eq 'ClientLogin' } @{$converted_request_data->{field_names}}) {
            $converted_request_data->{filter}{client_id} = { eq => $client_ids }
        }
    }
    my $campaign_types;
    if ($converted_request_data->{filter}{campaign_type}) {
        $campaign_types = convert_types($converted_request_data->{filter}{campaign_type}{eq});
        delete $converted_request_data->{filter}{campaign_type};
    }

    if ($report_type eq REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE) {
        my $cpm_types = get_camp_kind_types('cpm');
        if ($campaign_types) {
            $campaign_types = xisect($campaign_types, $cpm_types);
        } else {
            $campaign_types = $cpm_types;
        }
    }

    if (exists $converted_request_data->{filter}{banner} && exists $converted_request_data->{filter}{banner}{eq}) {
        $cids = $self->_restrict_campaign_ids_by_ad_ids($cids, $converted_request_data->{filter}{banner}{eq});
    }

    if (exists $converted_request_data->{filter}{adgroup} && exists $converted_request_data->{filter}{adgroup}{eq}) {
        $cids = $self->_restrict_campaign_ids_by_adgroup_ids($cids, $converted_request_data->{filter}{adgroup}{eq});
    }

    # если не заданы списки кампаний или типов, получаем все кампании клиента,
    # если заданы конкретные значения - оставляем только доступные клиенту
    if ((!$cids || @$cids) && (!$campaign_types || @$campaign_types)) {
        $cids = $self->_get_accessible_client_cids_with_stats($cids, $campaign_types, $client_ids);
    } else {
        $cids = [];
    }

    push @$cids, @{ $self->_get_subcampaigns($cids) };

    $converted_request_data->{filter}{cid}{eq} = $cids;

    my $goal_ids = exists $converted_request_data->{filter}{goal_ids} ? $converted_request_data->{filter}{goal_ids}{eq} : [];
    my $attribution_models = exists $converted_request_data->{filter}{attribution_models} ? $converted_request_data->{filter}{attribution_models}{eq} : [];
    my $field_names_in_provider_response = convert_field_names_to_report_fields($converted_request_data->{field_names}, $report_type);
    my $field_names_in_provider_response_with_multi_goals = 
        _get_names_in_provider_response($field_names_in_provider_response, $goal_ids, $attribution_models);

    my $displayed_field_names = _get_displayed_field_names($converted_request_data->{field_names}, $goal_ids, $attribution_models);

    my $provider_request_params = $self->_convert_request_data_to_report_params($converted_request_data, $opts{lang});

    my $need_to_request_report_from_provider = @$cids ? 1 : 0;

    return API::Reports::InternalRequestRepresentation->new(
        need_to_request_report_from_provider => $need_to_request_report_from_provider,
        lang                                 => $opts{lang},
        provider_request_params              => $provider_request_params,
        displayed_field_names                => $displayed_field_names,
        field_names_in_provider_response     => $field_names_in_provider_response_with_multi_goals,
        return_money_in_micros               => $opts{return_money_in_micros},
        skip_report_header                   => $opts{skip_report_header},
        skip_column_header                   => $opts{skip_column_header},
        skip_report_summary                  => $opts{skip_report_summary},
        response_format                      => $converted_request_data->{format},
        report_name                          => $converted_request_data->{report_name},
        report_type                          => $report_type,
    );
}

=head2 _get_accessible_client_ids

    По заданным логинам получаем идентификаторы клиентов и возвращаем список доступных субклиенту

=cut

sub _get_accessible_client_ids {
    my ($self, $client_logins) = @_;

    return unless $client_logins;

    my $complex_operator_uid = $self->operator_role eq $ROLE_SUPER ? $self->subclient_uid : $self->operator_uid;
    my $complex_operator_perminfo = get_perminfo(uid => $complex_operator_uid);
    return unless $complex_operator_perminfo->{mcc_client_ids};
    push @{$complex_operator_perminfo->{mcc_client_ids}}, $complex_operator_perminfo->{ClientID};

    my $request_client_ids = get_clientids(login => [uniq @$client_logins]);
    my $accessible_client_ids = xisect($complex_operator_perminfo->{mcc_client_ids}, $request_client_ids);

    return $accessible_client_ids;
}

=head2 _restrict_campaign_ids_by_ad_ids

    Ограничиваем идентификаторы кампаний, кампаниями баннеров, заданных в фильтре

=cut

sub _restrict_campaign_ids_by_ad_ids {
        my ($self, $cids, $bids) = @_;
        my @cids_for_bids;
        if (@$bids) {
            @cids_for_bids = uniq @{get_one_column_sql($self->ppc_shard, [ 'SELECT cid FROM banners', WHERE => { bid => $bids } ])};
        }

        return defined $cids ? xisect($cids, \@cids_for_bids) : \@cids_for_bids;
}

=head2 _restrict_campaign_ids_by_adgroup_ids

    Ограничиваем идентификаторы кампаний, кампаниями групп, заданных в фильтре

=cut

sub _restrict_campaign_ids_by_adgroup_ids {
        my ($self, $cids, $adgroup_ids) = @_;
        my @cids_for_adgroup_ids;
        if (@$adgroup_ids) {
            @cids_for_adgroup_ids = uniq @{get_one_column_sql($self->ppc_shard, [ 'SELECT cid FROM phrases', WHERE => { pid => $adgroup_ids } ])};
        }

        return defined $cids ? xisect($cids, \@cids_for_adgroup_ids) : \@cids_for_adgroup_ids;
}

=head2 _has_fields_dependent_on_goal

    Проеверяем есть ли в FieldNames поля зависящие от цели

=cut
 
sub _has_fields_dependent_on_goal {
    my ($field_names) = @_;
    for my $field_name (@$field_names) {
        if ( $FIELDS_DEPENDENT_ON_GOAL_MAP{$field_name} ) {
            return 1;
        }
    }
    return 0;
}

=head2 _get_names_in_provider_response

    Получаем внутреннее представление полей, в том числе зависимых от цели

=cut
 
sub _get_names_in_provider_response {
    my ($field_names, $goal_ids, $attribution_models) = @_;
    my @result;
    my %internal_fields_dependent_on_goal = reverse %FIELDS_DEPENDENT_ON_GOAL_MAP;
    for my $field_name (@$field_names) {
        if ( @$goal_ids && $internal_fields_dependent_on_goal{$field_name} ) {
            for my $goal_id (@$goal_ids) {
                for my $attribution_model (@$attribution_models) {
                    push @result, $field_name . '_' . $goal_id . '_' . $ATTRIBUTION_MODEL_TYPE_MAP{$attribution_model};
                }
            }
        } else {
            push @result, $field_name;
        }
    }
    return \@result;
}

=head2 _get_displayed_field_names

    Получаем внешнее представление полей, в том числе зависимых от цели

=cut

sub _get_displayed_field_names {
    my ($field_names, $goal_ids, $attribution_models) = @_;
    my @result;
    for my $field_name (@$field_names) {
        if ( @$goal_ids && $FIELDS_DEPENDENT_ON_GOAL_MAP{$field_name} ) {
            for my $goal_id (@$goal_ids) {
                for my $attribution_model (@$attribution_models) {
                    push @result, $field_name . '_' . $goal_id . '_' . $attribution_model;
                }
            }
        } else {
            push @result, $field_name;
        }
    }
    return \@result;
}

=head2 _tsv_line

Прокси к API::Reports::FormatTsv::tsv_line: нужен, чтобы переопределить в тестах
и в этих тестах проверять данные, а не TSV-строки.

=cut

sub _tsv_line {
    my ($self, $values) = @_;
    return tsv_line($values);
}


sub _populate_request_data {
    my ($self, $request_data) = @_;

    # указываем валюту клиента в качестве валюты запроса
    $request_data->{Currency} = $self->_get_subclient_currency();
    delete $request_data->{IncludeDiscount};
}

sub _validate_create {
    my ($self, $request_data) = @_;

    my $vr = API::Service::Reports::Validation->validate_create_report_request($request_data);
    return if $vr->is_valid;

    return $vr->get_errors->[0];
}

=head2 _get_accessible_client_cids_with_stats(cids, campaign_types, client_ids)

    Получаем идентификаторы кампаний клиента со статистикой
    cids - ограничиться перечисленными кампаниями
    campaign_types - ограничиться перечисленными типами кампаний
    client_ids - получить кампании для заданного списка идентификаторов клиентов
=cut

sub _get_accessible_client_cids_with_stats {
    my ($self, $cids, $campaign_types, $client_ids) = @_;
    $cids //= [];
    $campaign_types //= [];

    $campaign_types = get_camp_kind_types(SUPPORTED_CAMP_KIND)
        unless @$campaign_types;

    my $result_cids = get_one_column_sql(PPC(ClientID => $client_ids),
        [
            'SELECT c.cid FROM campaigns c LEFT JOIN subcampaigns sc ON c.cid = sc.cid',
            WHERE => {
                'c.statusEmpty' => 'No',
                'c.type' => $campaign_types,
                'c.metatype__ne' => 'ecom',
                'c.OrderID__gt' => 0,
                'c.ClientID' => SHARD_IDS,
                (@$cids ? ('c.cid' => $cids) : ()),
                'sc.master_cid__is_null' => 1
            }
        ]
    );

    if (@$result_cids) {
        my $accessible_cids = $self->_get_accessible_cids($result_cids);
        @$result_cids = grep { $accessible_cids->{$_} } @$result_cids;
    }

    return $result_cids;
}

=head2 _get_accessible_cids($cids)

    Возвращает хэш с cid'ами кампаний, для которых оператор имеет права на получение статистики из заданных в $cids

=cut

sub _get_accessible_cids {
    my ($self, $cids) = @_;

    return rbac_check_allow_show_camps(undef, $self->operator_uid, $cids);
}

sub _get_subcampaigns {
    my ($self, $cids) = @_;

    return get_one_column_sql(PPC(cid => $cids), [ 'SELECT cid FROM subcampaigns', WHERE => { master_cid => SHARD_IDS } ]);
}

sub _get_subclient_currency {
    my ($self) = @_;

    return get_client_currencies($self->subclient_client_id)->{work_currency};
}

sub response_ids {
    my ( $self, $response ) = @_;

    if ( $response && ref $response eq 'HASH' && $response->{job_id} ) {
        return [ $response->{job_id} ];
    }

    return undef;
}

sub count_error_warning_objects_in_reponse {
    my ($self, $response) = @_;
    return (0,0);
}

1;
