package API::Service::Reports::ConvertSubs;

use Direct::Modern;

use Exporter qw/import/;

use List::MoreUtils qw/uniq/;

use Yandex::DateTime qw/date now/;
use Yandex::ListUtils qw/as_array/;

use API::Reports::DataRules qw/
    %OPERATORS_MAP_TO_INTERNAL
    %ORDER_BY_FIELD_MAP
    %ORDER_DIRECTION_MAP
    %DATE_RANGE_TYPE_MAP
    %FILTER_PARAMS_MAP
    %DATE_AGGREGATION_MAP
    %FORMAT_MAP
    %RESULT_FIELD_MAP
    %GROUP_BY_FIELD_MAP
    %MANDATORY_GROUP_BY_FIELDS
    $MONEY_MULT
    %SINGLE_VALUE_OPERATOR
    %FIELD_TYPE_MAP
    %COUNTABLE_FIELDS

    DEFAULT_SORT_ORDER

    :types
/;
use Stat::Tools qw//;

our @EXPORT_OK = qw/
    convert_format
    convert_filter_items
    convert_field_names_and_report_type_to_group_by
    convert_field_names_to_date_aggregation
    convert_field_names_to_countable_fields
    convert_field_names_to_report_fields
    convert_page_name_filters
    convert_yes_no
    convert_none
    convert_order_by_items
    get_dates_by_range
    process_report_row
    convert_page_filter
/;

=head3 convert_format($format)

    Конвертирует формат отчёта во внутреннее представление

=cut

sub convert_format {
    my ($format) = @_;

    return exists $FORMAT_MAP{$format} ? $FORMAT_MAP{$format} : 'unknown';
}


=head3 convert_filter_items($filter_items)

    Конвертирует описание фильтра во внутреннее представление.

    $filter_items это ссылка на массив хэшей с полями operator, field, values.
    operator и field преобразуется в форму, используемую в
    Stat::CustomizedArray::StreamedExt.

    Значения values преобразуются в соответствие с типом поля Field.

    Значения денежных полей преобразуются из микроединиц во float.

    Функция возвращает хэш в формате Stat::CustomizedArray::StreamedExt, где ключами являются внутренние названия полей фильтрации.
=cut

sub convert_filter_items {
    my ($filter_items, $report_type) = @_;

    my %result_filter;
    foreach my $filter (@$filter_items) {
        my $operator = $OPERATORS_MAP_TO_INTERNAL{$filter->{operator}};
        my $filter_params = $FILTER_PARAMS_MAP{$report_type}{$filter->{field}};
        my $type = $FIELD_TYPE_MAP{$report_type}{$filter->{field}};
        my $values = $filter->{values};
        if ($type eq TYPE_MONEY || $type eq TYPE_SIGNED_MONEY) {
            # получаем деньги в микроединицах, а в ядро посылаем во Float
            $values = [ map {$_ / $MONEY_MULT} @$values ];
        } elsif ($type eq TYPE_BOOLEAN) {
            $values = [ map { convert_yes_no($_) } @$values ];
        } elsif ($type eq TYPE_ID) {
            $values = [ map { convert_none($_) } @$values ];
        } elsif (my $value_map = exists $filter_params->{map} && $filter_params->{map}) {
            $values = [ map { $value_map->{$_} } @$values ];
        }

        $result_filter{$filter_params->{name}}{$operator} =
            exists $SINGLE_VALUE_OPERATOR{$operator} || exists $filter_params->{single_value} && $filter_params->{single_value}
            ? $values->[0]
            : $values;
    }

    return \%result_filter;
}


=head3 convert_field_names_and_report_type_to_group_by($field_names, $report_type)

    Определяет необходимые группировки, соответствующие полям и типу отчёта

=cut

sub convert_field_names_and_report_type_to_group_by {
    my ($field_names, $report_type) = @_;

    my @fields_group_by = map { exists $GROUP_BY_FIELD_MAP{$report_type}{$_} ? @{as_array($GROUP_BY_FIELD_MAP{$report_type}{$_})} : () } @$field_names;

    my $mandatory_group_by_fields = $MANDATORY_GROUP_BY_FIELDS{$report_type};
    if (ref($mandatory_group_by_fields) eq 'CODE') {
        $mandatory_group_by_fields = $mandatory_group_by_fields->($field_names);
    }

    return [ uniq(@$mandatory_group_by_fields, @fields_group_by) ];
}


=head3 convert_field_names_to_date_aggregation($field_names)

    Определяет необходимую группировку по дате (date_aggregation_by),
    по набору полей.

    Группировка по дате возможна только одна. Если в аргументах будут
    переданы несколько полей, для которых возможна группировка по дате,
    вернётся только одна -- для первого встетившегося поля.

    Если ни одно из полей не вызывает группировку по дате, вернётся
    строка "none".

=cut

sub convert_field_names_to_date_aggregation {
    my ($field_names) = @_;

    foreach my $field_name (@{$field_names}) {
        return $DATE_AGGREGATION_MAP{$field_name} if exists $DATE_AGGREGATION_MAP{$field_name};
    }

    return 'none';
}

=head3 convert_field_names_to_countable_fields($field_names, $report_type)

    Возвращает поля запроса, которые являются метриками и должны быть переданы в ядро в countable_fields

=cut

sub convert_field_names_to_countable_fields {
    my ($field_names, $report_type) = @_;

    my @countable_fields;

    foreach my $field_name (@{$field_names}) {
        next if !exists $COUNTABLE_FIELDS{$field_name} || !exists $RESULT_FIELD_MAP{$report_type}{$field_name};
        push @countable_fields, ref $RESULT_FIELD_MAP{$report_type}{$field_name} ? $RESULT_FIELD_MAP{$report_type}{$field_name}{name} : $RESULT_FIELD_MAP{$report_type}{$field_name};
    }

    return \@countable_fields;
}

=head3 convert_field_names_to_report_fields($field_names, $report_type)

    Преобразует список полей из внешних названий во внутренние для заданного в $report_type типа отчета.

=cut

sub convert_field_names_to_report_fields {
    my ($field_names, $report_type) = @_;

    return [ map { ref $RESULT_FIELD_MAP{$report_type}{$_} ? $RESULT_FIELD_MAP{$report_type}{$_}{name} : $RESULT_FIELD_MAP{$report_type}{$_} } @$field_names ];
}


=head3 convert_yes_no($value)

    Преобразует значения YES/NO во внутреннее представление 1/0

=cut

sub convert_yes_no {
    my ($value) = @_;

    return $value eq 'YES' ? 1 : 0;
}

=head3 convert_order_by_items($order_by_items)

    Конвертирует описание условий сортировки во внутреннее представление.

    $order_by_items это ссылка на массив хэшей с полями field и sort_order.
    field и sort_order преобразуется в форму, используемую в Stat::CustomizedArray::StreamedExt.

=cut

sub convert_order_by_items {
    my ($order_by_items, $report_type) = @_;

    foreach my $order_by (@$order_by_items) {
        $order_by->{field} = $ORDER_BY_FIELD_MAP{$report_type}{$order_by->{field}};
        $order_by->{dir} = $ORDER_DIRECTION_MAP{$order_by->{dir} // DEFAULT_SORT_ORDER};
    }

    return $order_by_items;
}

=head3 get_dates_by_range($type, $date, $OrderIds)

    Преобразует указанный тип диапазона дат в конкретный диапазон,
    заданный датой начала и датой конца.

    Если не передана дата $date, диапазон дат основывается на
    сегодняшней дате

=cut

sub get_dates_by_range {
    my ($type, $date, $OrderIds) = @_;

    return $DATE_RANGE_TYPE_MAP{ $type // 'TODAY' }->( defined $date ? date($date) : now(), $OrderIds );
}


=head2 convert_page_filter($filter)

    Конвертируем список названий мест размещения в идентификаторы

=cut

sub convert_page_filter {
    my ($filter, $lang) = @_;

    my ($op) = keys %$filter;

    return { $op => Stat::Tools::convert_page_name_filter_to_ids($filter->{$op}, $op, $lang) };
}

=head3 convert_page_name_filters($filters)

    Конвертирует операторы eq/ne в contains/not_contains у фильтра по названию площадки.
    Раньше фильтры с оператором eq превращались в список PageID по подстроке и передавались в МОЛ c оператором eq.
    Теперь передаём оператор contains напрямую в МОЛ.

    Внешние операторы IN и NOT_IN переводятся в eq и ne в convert_filter_items.

=cut

sub convert_page_name_filters {
    my ($filters) = @_;

    my $result = {%$filters};
    if (exists $result->{eq}) {
        $result->{contains} = delete $result->{eq};
    }
    if (exists $result->{ne}) {
        $result->{not_contains} = delete $result->{ne};
    }
    return $result;
}

=head3 convert_none($value)

    Преобразует значения NONE для id'шников во внутреннее значение 0

=cut

sub convert_none {
    my ($value) = @_;

    return $value eq 'NONE' ? 0 : $value;
}


1;
