package API::Service::Reports::ProcessingModeChooser;
use Direct::Modern;

use Yandex::DateTime;
use Yandex::I18n;

use API::Reports::DataRules qw/
    ACCOUNT_PERFORMANCE_REPORT_TYPE
    CAMPAIGN_PERFORMANCE_REPORT_TYPE
    STRATEGY_PERFORMANCE_REPORT_TYPE
    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE
/;
use API::Settings;
use Direct::Errors::Messages;
use Property;

=head2 PROCESSING_MODE_ONLINE

=cut

use constant PROCESSING_MODE_ONLINE => 1;

=head2 PROCESSING_MODE_OFFLINE

=cut

use constant PROCESSING_MODE_OFFLINE => 2;

=head2 BY_DEFAULT_REQUIRE_OFFLINE_MODE_FOR_DAYS_RANGE_TYPE_GREATER_THAN

    Константа, в днях указывающая за сколько, по умолчанию дней, мы позволяем
    строить отчет в онлайн режиме, также можно задать через свойство в БД,
    см. $API::Settings::REQUIRE_OFFLINE_MODE_FOR_DATE_RANGE_TYPE_GREATER_THAN_PROPERTY

=cut

use constant BY_DEFAULT_REQUIRE_OFFLINE_MODE_FOR_DAYS_RANGE_TYPE_GREATER_THAN => 7;

=head2 BY_DEFAULT_REQUIRE_OFFLINE_MODE_FOR_LIST_OF_CAMPAIGNS_LONGER_THAN

    Константа, указывающая сколько, по умолчанию, кампаний можно задать в
    отчете чтобы его можно строить отчет в онлайн режиме, также можно задать
    через свойство в БД,
    см. $API:Settings::REQUIRE_OFFLINE_MODE_FOR_LIST_OF_CAMPAIGNS_LONGER_THAN_PROPERTY

=cut

use constant BY_DEFAULT_REQUIRE_OFFLINE_MODE_FOR_LIST_OF_CAMPAIGNS_LONGER_THAN => 10;


=head2 OFFLINE_MODE_ADJUSTING_PROPERTIES_TTL_SECS

    TTL для кэша настроект выбора между онлайн и оффлайн режимом в построения отчета в секундах

=cut

use constant OFFLINE_MODE_ADJUSTING_PROPERTIES_TTL_SECS => 30;

my $online_mode_days_range_property = Property->new($API::Settings::REQUIRE_OFFLINE_MODE_FOR_DATE_RANGE_TYPE_GREATER_THAN_PROPERTY);
my $online_mode_campaigns_limit_property = Property->new($API::Settings::REQUIRE_OFFLINE_MODE_FOR_LIST_OF_CAMPAIGNS_LONGER_THAN_PROPERTY);

sub _require_offline_mode {
    my ( $class, $application_id, $request_data, $request_internal_rep ) = @_;

    # используем онлайн-режим для некоторых приложений
    if ($API::Settings::NEED_ONLINE_REPORT_APP_ID->{$application_id}) {
        return 0;
    }

    # отчёт по поисковым фразам всегда большой и тяжёлый
    my $report_type = $request_data->{report_type};
    if ( $report_type eq SEARCH_QUERY_PERFORMANCE_REPORT_TYPE ) {
        return 1;
    }

    # пустой отчёт всегда можно отдать онлайн
    if ( !$request_internal_rep->need_to_request_report_from_provider ) {
        return 0;
    }

    # отчёт за всё время всегда большой и тяжёлый
    if ( $request_data->{date_range_type} eq 'ALL_TIME' ) {
        return 1;
    }

    my $provider_request = $request_internal_rep->provider_request_params;

    return $class->require_offline_mode_adjusted($provider_request, $report_type);
}

=head2 require_offline_mode_adjusted($provider_request, $report_type)

    По данным запроса в формате МОЛ и текущей настройке, которую можно задать в интерфейсе
    определяем, как строить отчет, онлайн или оффлайн

    0 - отчет строится сразу
    1 - откладываем построение отчета в очередь

=cut

sub require_offline_mode_adjusted {
    my ($class, $provider_request, $report_type) = @_;

    my $day_secs = 86400;
    my $ttl = OFFLINE_MODE_ADJUSTING_PROPERTIES_TTL_SECS;
    my ( $start_date, $end_date ) = map { date( $provider_request->{$_} ) } qw( start_date end_date );
    my $report_range_seconds = $end_date->epoch - $start_date->epoch + $day_secs;
    my $report_campaigns_count = @{ $provider_request->{oid} };

    my ($seconds_ok_for_online_processing, $campaigns_ok_for_online_processing);
    if ($report_type eq ACCOUNT_PERFORMANCE_REPORT_TYPE || $report_type eq CAMPAIGN_PERFORMANCE_REPORT_TYPE 
        || $report_type eq STRATEGY_PERFORMANCE_REPORT_TYPE) {
        $seconds_ok_for_online_processing = $online_mode_days_range_property->get($ttl);
        $seconds_ok_for_online_processing *= $day_secs if $seconds_ok_for_online_processing;
        $campaigns_ok_for_online_processing = $online_mode_campaigns_limit_property->get($ttl);
    }

    $seconds_ok_for_online_processing ||= BY_DEFAULT_REQUIRE_OFFLINE_MODE_FOR_DAYS_RANGE_TYPE_GREATER_THAN * $day_secs;
    $campaigns_ok_for_online_processing ||= BY_DEFAULT_REQUIRE_OFFLINE_MODE_FOR_LIST_OF_CAMPAIGNS_LONGER_THAN;

    # отчёты за небольшой период и по не большому количеству кампаний должны строиться быстро
    if ($report_range_seconds <= $seconds_ok_for_online_processing
        && $report_campaigns_count <= $campaigns_ok_for_online_processing)
    {
        return 0;
    }

    # всякая мумбо-жубмо-machine-learning-магия для определения способа
    # построения отчета могла бы быть здесь

    return 1;
}

=head2 choose_processing_mode

    Возвращает числовую константу PROCESSING_MODE_* или Direct::Defect, если клиент хочет что-то, чего делать нельзя.

=cut

sub choose_processing_mode {
    my ( $class, $application_id, $mode_requested_by_client, $request_data, $request_internal_rep ) = @_;

    $mode_requested_by_client //= 'auto';

    if ( $mode_requested_by_client eq 'offline' ) {
        return PROCESSING_MODE_OFFLINE;
    }

    if ( $mode_requested_by_client ne 'auto' && $mode_requested_by_client ne 'online' ) {
        return error_BadRequest( iget( 'Неправильный заголовок processingMode: %s', $mode_requested_by_client ) );
    }

    my $require_offline_mode = $class->_require_offline_mode( $application_id, $request_data, $request_internal_rep );

    if ( $mode_requested_by_client eq 'auto' ) {
        return $require_offline_mode ? PROCESSING_MODE_OFFLINE : PROCESSING_MODE_ONLINE;
    }

    if ( $mode_requested_by_client eq 'online' ) {
        if ($require_offline_mode) {
            return error_ReportCannotBeBuiltOnline( iget('Отчет невозможно построить в режиме online') );
        }
        return PROCESSING_MODE_ONLINE;
    }
}

1;
