package API::Service::Reports::Validation;
use Direct::Modern;

use List::UtilsBy qw(count_by);
use List::MoreUtils qw(all any none);
use Yandex::ListUtils qw(xisect);

use Settings;

use Yandex::I18n;
use Yandex::TimeCommon 'mysql2unix';
use Yandex::Validate qw( is_valid_date is_valid_float is_valid_id is_valid_int );

use Direct::Errors::Messages;
use Direct::ValidationResult;
use GeoTools 'validate_geo';

use API::Reports::DataRules qw/
    %FIELD_TYPE_MAP
    %DATE_AGGREGATION_MAP
    %FILTER_PARAMS_MAP
    %OPERATOR_DESC_MAP
    %ORDER_BY_FIELD_MAP
    %INCOMPATIBLE_FIELDS_MAP
    %RESULT_FIELD_MAP
    %REPORT_TYPE_CONF
    %MANDATORY_FIELDS
    @CRITERIA_RELATED_FILTER_FIELDS
    CUSTOM_DATE_RANGE_TYPE
    %FIELDS_DEPENDENT_ON_GOAL_MAP
    :types
/;

use API::Service::Reports::ConvertSubs qw/get_dates_by_range/;

use Stat::Const qw/$BS_AGE_45_54_AND_AGE_55_DATE/;

our $MAX_STRING_LENGTH = 4096;

our $MAX_XSD_INT = 2147483647;

our $MAX_MULTICLIENT_LIMIT = 500000;

our $DATE_RELATED_FIELDS = [ sort keys %DATE_AGGREGATION_MAP ];

our $MAX_REPORT_NAME_LENGTH = 255;

our $FILTER_VALUES_LIMIT = 10000;

our $GOAL_IDS_LIMIT = 10;

use constant YND_FIXED => 'YND_FIXED';
use constant YES => 'YES';
use constant NO => 'NO';
use constant TRUE => 1;
use constant FIELD_NAMES_FIELD => 'FieldNames';
use constant REPORT_NAME_FIELD => 'ReportName';
use constant REPORT_TYPE_FIELD => 'ReportType';
use constant FILTER_FIELD => 'Filter';
use constant FIELD_INSIDE_FILTER => FILTER_FIELD.'.Field';
use constant OPERATOR_FIELD => 'Operator';
use constant OPERATOR_INSIDE_FILTER => FILTER_FIELD.'.'.OPERATOR_FIELD;
use constant VALUES_INSIDE_FILTER => FILTER_FIELD.'.Values';
use constant ORDER_BY_FIELD => 'OrderBy';
use constant FIELD_INSIDE_ORDER_BY => ORDER_BY_FIELD.'.Field';
use constant ERROR_MESSAGE_FIELD_DELIMETER => ', ';
use constant DATE_FROM_FIELD => 'DateFrom';
use constant DATE_TO_FIELD => 'DateTo';
use constant CUSTOM_DATE_RANGE_FIELDS => [DATE_FROM_FIELD, DATE_TO_FIELD];
use constant DATE_RANGE_TYPE_FIELD => 'DateRangeType';
use constant MIN_GOAL_ID => 1;
use constant MAX_GOAL_ID => 4_000_000_000;


=head2 validate_create_report_request

Логическая валидация запроса на создание отчёта. Запускать надо после того, как валидация по xsd уже прошла успешно.
Возвращает ValidationResult, в котором ошибки (если есть) привязаны к полям, как они называются в структуре
данных запроса от клиента.

=cut

sub validate_create_report_request {
    my ( $class, $request_data ) = @_;

    my $vr = Direct::ValidationResult->new;
    my $field_names = $request_data->{+FIELD_NAMES_FIELD};
    my $report_type = $request_data->{+REPORT_TYPE_FIELD};
    my $has_goals = exists $request_data->{Goals} ? 1 : 0;
    $vr->add( FIELD_NAMES_FIELD() => $class->_validate_create_field_names($field_names, $report_type) );
    $vr->add( DateRange => $class->_validate_create_date_range($request_data->{DateRangeType}, $request_data->{SelectionCriteria}, $report_type ) );

    my $selection_criteria_vr = Direct::ValidationResult->new;

    $selection_criteria_vr->add( SelectionCriteria => $class->_validate_create_filter_items( $request_data->{SelectionCriteria}->{Filter}, $field_names, $report_type, $has_goals ) );
    $vr->add( SelectionCriteria => $selection_criteria_vr );

    my $has_age_error = !$class->_is_valid_filter_age($request_data->{DateRangeType}, $request_data->{SelectionCriteria});
    if ($has_age_error) {
        my $date = $BS_AGE_45_54_AND_AGE_55_DATE;
        $date =~ s/(\d{4})(\d{2})(\d{2})/$1-$2-$3/g;
        $vr->add_generic(error_WrongSelectionCriteria( 
            iget('Для поля %1$s, указанного в параметре %2$s, значение AGE_45 в параметре %3$s не поддерживается с %4$s, используйте AGE_45_54 и AGE_55', 
            "Age", FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER, $date)
        ));
    }

    if (exists $request_data->{AttributionModels}) {
         $vr->add(AttributionModels => error_BadParams("Поле AttributionModels может быть указано только в случае, если указано поле Goals"))
            if !exists $request_data->{Goals};

        my $error = $class->_validate_dups_in_array_fields($request_data->{AttributionModels}, 'AttributionModels');
        $vr->add(AttributionModels => $error) if $error;
    }
    if (exists $request_data->{Goals}) {
        my $goals = $request_data->{Goals};
        $vr->add(Goals => error_LimitExceeded_ArraySizeExceeded(undef, param => 'Goals', limit => $GOAL_IDS_LIMIT)) 
            if (@$goals == 0 ||  @$goals > $GOAL_IDS_LIMIT);

        my $error = $class->_validate_dups_in_array_fields($goals, 'Goals');
        $vr->add(Goals => $error) if $error;

        my @invalid_goal_ids = grep { !is_valid_int($_, MIN_GOAL_ID, MAX_GOAL_ID) } @$goals;
        if (@invalid_goal_ids) {
            $vr->add(Goals => error_BadParams(iget('В параметре %1$s указаны недопустимые значения: %2$s', 'Goals', join(ERROR_MESSAGE_FIELD_DELIMETER, @invalid_goal_ids))));
        }
    }

    if (exists $request_data->{Page}) {
        my $is_multiclient_request = ($request_data->{SelectionCriteria}{Filter} && any { $_->{Field} eq 'ClientLogin' } @{$request_data->{SelectionCriteria}{Filter}}) ? 1 : 0;
        $vr->add( Page => $class->_validate_create_page($request_data->{Page}, $is_multiclient_request));
    }
    if (exists $request_data->{OrderBy}) {
        $vr->add(ORDER_BY_FIELD() => $class->_validate_create_order_by($request_data->{OrderBy}, $field_names, $report_type, $has_goals));
    }
    $vr->add(REPORT_NAME_FIELD() => $class->_validate_create_report_name($request_data->{ReportName}));

    return $vr;
}

=head2 _is_valid_filter_age

 Если клиент запрашивает отчет только с фильтром AGE_45 за период после отлючения этой возрастной категории в БК возвращаем false

=cut

sub _is_valid_filter_age {
    my ($class, $date_range_type, $selection_criteria) = @_;

    my $date_from_ymd;
    if ($date_range_type eq CUSTOM_DATE_RANGE_TYPE) {
        $date_from_ymd = $selection_criteria->{+DATE_FROM_FIELD};
    } elsif ($date_range_type ne 'ALL_TIME') {
        my @dates = get_dates_by_range($date_range_type);
        $date_from_ymd = $dates[0];
    }
    $date_from_ymd =~ s/-//g if $date_from_ymd;
    
    if ($selection_criteria->{+FILTER_FIELD}) {
        for my $filter (@{ $selection_criteria->{+FILTER_FIELD} }) {
            if ($filter->{Field} eq "Age") {
                my %age_map = map {$_ => 1} @{$filter->{Values}};
                if (exists $age_map{AGE_45} && (!exists $age_map{AGE_45_54} || !exists $age_map{AGE_55})) {
                    if ($date_from_ymd ge $BS_AGE_45_54_AND_AGE_55_DATE || $date_range_type eq 'ALL_TIME') {
                        return 0;
                    }
                }
            }
        }
    }
    return 1;
}

sub _validate_create_field_names {
    my ( $class, $field_names, $report_type ) = @_;

    my $vr = Direct::ValidationResult->new;

    my %field_count = count_by { $_ } @$field_names;

    my @missed_mandatory_fields = grep { !$field_count{$_} } @{$MANDATORY_FIELDS{$report_type} // []};
    if (@missed_mandatory_fields) {
        $vr->add_generic(
            error_BadParams(
                iget('В параметре %1$s отсутствуют значения %2$s, обязательные для типа отчета %3$s', FIELD_NAMES_FIELD, join(', ', @missed_mandatory_fields), $report_type)
            )
        );
    }

    my $supported_fields = $RESULT_FIELD_MAP{$report_type};
    my @unsupported_field_names = grep { !exists $supported_fields->{$_} } sort keys %field_count;
    if (@unsupported_field_names) {
        $vr->add_generic(
            error_BadParams(
                iget( 'В параметре %1$s указаны значения %2$s, недопустимые для типа отчета %3$s', FIELD_NAMES_FIELD, join(', ', @unsupported_field_names), $report_type )
            )
        );
    }

    foreach my $field_group ($DATE_RELATED_FIELDS) {
        if ((grep { exists $field_count{$_} } @$field_group) > 1) {
            $vr->add_generic(
                error_BadParams(
                    iget( 'В параметре %1$s может быть указано только одно из значений %2$s', FIELD_NAMES_FIELD, join(', ', @$field_group) )
                )
            );
        }
    }

    my @dup_fields = grep {$field_count{$_} > 1} keys %field_count;
    foreach my $field (@dup_fields) {
        $vr->add_generic(
            error_DuplicateParameterValues(iget('В параметре %1$s значение %2$s указано более одного раза', FIELD_NAMES_FIELD, $field))
        );
    }

    my $incompatible_fields_map = exists $INCOMPATIBLE_FIELDS_MAP{$report_type} ? $INCOMPATIBLE_FIELDS_MAP{$report_type} : {};
    foreach my $field (grep { exists $field_count{$_} } sort keys %$incompatible_fields_map) {
        my @incompatible_fields = grep { exists $supported_fields->{$_} } @{$incompatible_fields_map->{$field}};
        if (any { exists $field_count{$_} } @incompatible_fields) {
            $vr->add_generic(
                error_BadParams(
                    iget(
                        'В параметре %1$s указано значение %2$s, которое несовместимо со значениями %3$s для типа отчета %4$s',
                        FIELD_NAMES_FIELD,
                        $field,
                        join(', ', @incompatible_fields),
                        $report_type
                    )
                )
            );
        }
    }

    return $vr;
}

sub _validate_create_date_range {
    my ( $class, $date_range_type, $selection_criteria, $report_type ) = @_;

    my $vr = Direct::ValidationResult->new;

    if ($date_range_type eq CUSTOM_DATE_RANGE_TYPE) {
        for my $fieldname (@{+CUSTOM_DATE_RANGE_FIELDS}) {
            my $value = $selection_criteria->{$fieldname};
            unless ( defined $value && is_valid_date($value) && $value =~ /^\d{4}-\d{2}-\d{2}$/ ) {
                $vr->add( $fieldname => error_WrongSelectionCriteria( iget( 'В параметре %s должна быть указана корректная дата в формате YYYY-MM-DD', $fieldname ) ) );
            } elsif ($fieldname eq DATE_FROM_FIELD && !is_valid_date($value, no_future => TRUE)) {
                $vr->add( $fieldname => error_WrongSelectionCriteria( iget( 'Дата в параметре %s должна быть не позднее текущей даты', DATE_FROM_FIELD ) ) );
            }
        }

        if ( $vr->is_valid ) {
            if ( $selection_criteria->{+DATE_FROM_FIELD} gt $selection_criteria->{+DATE_TO_FIELD} ) {
                $vr->add_generic( error_WrongSelectionCriteria( iget('Дата в параметре %1$s должна быть не ранее даты в параметре %2$s', DATE_TO_FIELD, DATE_FROM_FIELD) ) );
            }
        }

        if ( $vr->is_valid ) {
            my $date_to = $selection_criteria->{+DATE_TO_FIELD};
            my $availability_date = $REPORT_TYPE_CONF{$report_type}->{availability_date_getter}->();

            if ( mysql2unix($date_to) < mysql2unix($availability_date) ) {
                $vr->add( +DATE_TO_FIELD => error_WrongSelectionCriteria( $REPORT_TYPE_CONF{$report_type}->{not_available_error_msg_getter}->( $report_type, DATE_TO_FIELD, $availability_date ) ) );
            }
        }
    } else {
        for my $fieldname (@{+CUSTOM_DATE_RANGE_FIELDS}) {
            if (exists $selection_criteria->{$fieldname}) {
                $vr->add_generic(
                    error_WrongSelectionCriteria(iget('Параметр %1$s можно указывать только при значении "%2$s" параметра %3$s', $fieldname, CUSTOM_DATE_RANGE_TYPE, DATE_RANGE_TYPE_FIELD))
                );
            }
        }
    }

    return $vr;
}

sub _validate_create_filter_items {
    my ( $class, $filter_items, $field_names, $report_type, $has_goals ) = @_;

    my $vr = Direct::ValidationResult->new;

    my %filter_fields;
    for my $item (@$filter_items) {
        $vr->next;
        my $field = $item->{Field};

        if (!exists $FILTER_PARAMS_MAP{$report_type}{$field}) {
            $vr->add(Field => error_WrongSelectionCriteria(iget('В параметре %1$s указано значение %2$s, недопустимое для типа отчета %3$s', FIELD_INSIDE_FILTER, $field, $report_type)));
            next;
        }

        if ($filter_fields{$field}++) {
            $vr->add(Field => error_WrongSelectionCriteria( iget( 'В параметре %1$s значение %2$s указано более одного раза', FIELD_INSIDE_FILTER, $field) ) );
            next;
        }

        my $operator = $item->{+OPERATOR_FIELD};
        my $values = $item->{Values};

        my $filter_params = $FILTER_PARAMS_MAP{$report_type}{$field};
        my $type = $FIELD_TYPE_MAP{$report_type}{$field};

        if (all {$_ ne $operator} @{$filter_params->{ops}}) {
            $vr->add( OPERATOR_FIELD() => error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, допустимы только операторы %3$s в параметре %4$s для типа отчета %5$s', $field, FIELD_INSIDE_FILTER, join(ERROR_MESSAGE_FIELD_DELIMETER, @{$filter_params->{ops}}), OPERATOR_INSIDE_FILTER, $report_type ) ) );
            next;
        }
        if (!$OPERATOR_DESC_MAP{$operator}{takes_array} && @$values > 1) {
            $vr->add( OPERATOR_FIELD() => error_WrongSelectionCriteria( 
                iget( 'Для поля %1$s, указанного в параметре %2$s, и оператора %3$s, указанного в параметре %4$s, допустимо только одно значение в параметре %5$s', $field, FIELD_INSIDE_FILTER, $operator, OPERATOR_INSIDE_FILTER, VALUES_INSIDE_FILTER ) ) );
            next;
        } elsif ($OPERATOR_DESC_MAP{$operator}{takes_array} && @$values > $FILTER_VALUES_LIMIT) {
            $vr->add( OPERATOR_FIELD() => error_WrongSelectionCriteria( 
                iget( 'Для поля %1$s, указанного в параметре %2$s, и оператора %3$s, указанного в параметре %4$s, допустимо не более %5$d значений в параметре %6$s', $field, FIELD_INSIDE_FILTER, $operator, OPERATOR_INSIDE_FILTER, $FILTER_VALUES_LIMIT, VALUES_INSIDE_FILTER ) ) );
            next;
        }

        foreach my $val (@$values) {
            my $error = $class->_validate_filter_item_value($field, $val, $filter_params, $type, $operator, $report_type);
            if ($error) {
                $vr->add( Values => $error);
                last;
            }
        }
    }

    if ((grep { $filter_fields{$_} } @CRITERIA_RELATED_FILTER_FIELDS) > 1) {
        $vr->add_generic(
            error_WrongSelectionCriteria(
                iget(
                    'В параметре %1$s может быть указано только одно из значений %2$s',
                    FIELD_INSIDE_FILTER,
                    join(ERROR_MESSAGE_FIELD_DELIMETER, @CRITERIA_RELATED_FILTER_FIELDS)
                )
            )
        );
    }

    my %field_names = map { $_ => undef } @$field_names;
    my $incompatible_fields_map = exists $INCOMPATIBLE_FIELDS_MAP{$report_type} ? $INCOMPATIBLE_FIELDS_MAP{$report_type} : {};
    foreach my $field (keys %$incompatible_fields_map) {
        my $incompatible_fields = $incompatible_fields_map->{$field};
        if (exists $filter_fields{$field}) {
            if (any { exists $filter_fields{$_} } @$incompatible_fields) {
                $vr->add_generic(
                    error_WrongSelectionCriteria(
                        iget(
                            'В параметре %1$s указано значение %2$s, которое несовместимо со значениями %3$s',
                            FIELD_INSIDE_FILTER,
                            $field,
                            join(ERROR_MESSAGE_FIELD_DELIMETER, grep { exists $FILTER_PARAMS_MAP{$report_type}{$_} } @$incompatible_fields),
                        )
                    )
                );
            }

            if (any { exists $field_names{$_} } @$incompatible_fields) {
                $vr->add_generic(
                    error_WrongSelectionCriteria(
                        iget(
                            'В параметре %1$s указано значение %2$s, которое несовместимо с заданными в параметре %3$s значениями %4$s для типа отчета %5$s',
                            FIELD_INSIDE_FILTER,
                            $field,
                            FIELD_NAMES_FIELD,
                            join(ERROR_MESSAGE_FIELD_DELIMETER, @$incompatible_fields),
                            $report_type
                        )
                    )
                );
            }
        }

        my @incompatible_filter_fields = grep { exists $FILTER_PARAMS_MAP{$report_type}{$_} } @$incompatible_fields;
        if (exists $field_names{$field} && any { exists $filter_fields{$_} } @incompatible_filter_fields) {
            $vr->add_generic(
                error_WrongSelectionCriteria(
                    iget(
                        'В параметре %1$s указано одно из значений %2$s, которые несовместимы с заданным в параметре %3$s значением %4$s для типа отчета %5$s',
                        FIELD_INSIDE_FILTER,
                        join(ERROR_MESSAGE_FIELD_DELIMETER, @incompatible_filter_fields),
                        FIELD_NAMES_FIELD,
                        $field,
                        $report_type
                    )
                )
            );
        }
    }

    my $exists_fields_dependent_on_goal = xisect([keys %FIELDS_DEPENDENT_ON_GOAL_MAP], [keys %filter_fields]);
    if (@$exists_fields_dependent_on_goal && $has_goals) {
        $vr->add_generic(
            error_WrongSelectionCriteria(
                iget( 'В параметре %1$s значения %2$s недопустимы если задано поле Goals', FIELD_INSIDE_FILTER, join(', ', @$exists_fields_dependent_on_goal) )
            )
        );
    }

    return $vr;
}


=head2 _validate_filter_item_value

Возвращает Direct::Defect в случае ошибки; undef, если всё хорошо.

=cut

sub _validate_filter_item_value {
    my ( $class, $name, $value, $filter_params, $type, $operator, $report_type ) = @_;

    die 'need $report_type' unless $report_type;

    if ( $type eq TYPE_ID) {
        unless ( is_valid_id($value) || $value eq 'NONE' ) {
            return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, задан некорректный идентификатор в параметре %3$s', $name, FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER ) );
        }
    } elsif ( $type eq TYPE_BOOLEAN ) {
        if ($value ne 'Yes' && $value ne 'No') {
            return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, допустимы только значения %3$s в параметре %4$s', $name, FIELD_INSIDE_FILTER, 'Yes, No', VALUES_INSIDE_FILTER ) );
        }
    } elsif ( $type eq TYPE_INTEGER) {
        if ( !is_valid_int($value) ) {
            return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, значение в параметре %3$s должно быть целочисленным', $name, FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER ) );
        }
    } elsif ( $type eq TYPE_MONEY) {
        if ( !is_valid_int($value) || $value < 0) {
            return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, значение в параметре %3$s должно быть целочисленным и неотрицательным', $name, FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER ) );
        }
    } elsif ( $type eq TYPE_SIGNED_MONEY) {
        if ( !is_valid_int($value)) {
            return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, значение в параметре %3$s должно быть целочисленным', $name, FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER ) );
        }
    } elsif ( $type eq TYPE_FLOAT) {
        if ( !is_valid_float($value) ) {
            return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, задано некорректное значение в параметре %3$s', $name, FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER ) );
        }
    } elsif ( $type eq TYPE_GEO_REGION ) {
        my $geo_error = validate_geo($value);
        if ( defined $geo_error && $geo_error ne '' ) {
            return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, задан некорректный регион в параметре %3$s', $name, FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER ) );
        }
    } elsif ( $type eq TYPE_STRING ) {
        if ( length $value > $MAX_STRING_LENGTH ) {
            return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, значение в параметре %3$s должно содержать не более %4$d символов', $name, FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER, $MAX_STRING_LENGTH ) );
        }
    } else {
        die "Invalid type=$type for field=$name";
    }

    if (exists $filter_params->{enum} && none {$_ eq $value} @{$filter_params->{enum}}) {
        return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, допустимы только значения %3$s в параметре %4$s для типа отчета %5$s', $name, FIELD_INSIDE_FILTER, join(', ', @{$filter_params->{enum}}), VALUES_INSIDE_FILTER, $report_type ) );
    }

    if (exists $filter_params->{non_negative} && $filter_params->{non_negative} && $value < 0) {
        return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, значение в параметре %3$s должно быть неотрицательным', $name, FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER ) );
    }

    if (exists $filter_params->{positive_lt} && $filter_params->{positive_lt} && $operator eq 'LESS_THAN' && $value <= 0) {
        return error_WrongSelectionCriteria( iget( 'Для поля %1$s, указанного в параметре %2$s, значение в параметре %3$s должно быть положительным', $name, FIELD_INSIDE_FILTER, VALUES_INSIDE_FILTER ) );
    }
    return undef;
}


sub _validate_create_page {
    my ($class, $page, $is_multiclient_request) = @_;

    my $vr = Direct::ValidationResult->new;
    if ($page->{Limit} < 1) {
        $vr->add(Limit => error_IncorrectPage_NonpositiveLimit());
    } elsif ($is_multiclient_request && $page->{Limit} > $MAX_MULTICLIENT_LIMIT) {
        $vr->add(Limit => error_IncorrectPage_LimitExceeded(undef, limit => $MAX_MULTICLIENT_LIMIT));
    } elsif ($page->{Limit} > $MAX_XSD_INT) {
        $vr->add(Limit => error_IncorrectPage_LimitExceeded(undef, limit => $MAX_XSD_INT));
    }

    if (exists $page->{Offset}) {
        if ($page->{Offset} < 0) {
            $vr->add(Offset => error_IncorrectPage_NegativeOffset());
        }
        elsif ($page->{Offset} > $MAX_XSD_INT) {
            $vr->add(Limit => error_IncorrectPage_OffsetExceeded(undef, limit => $MAX_XSD_INT));
        }
    }

    return $vr;
}

=head2 _validate_create_order_by($order_by, $field_names, $report_type)

    Валидируем блок запроса OrderBy с учетом возвращаемых полей для заданного в $report_type типа отчета

=cut

sub _validate_create_order_by {
    my ($class, $order_by, $field_names, $report_type, $has_goals) = @_;

    my $vr = Direct::ValidationResult->new;

    my %order_by_fields = count_by {$_->{Field}} @$order_by;
    my @unsupported_field_names = grep { !exists $ORDER_BY_FIELD_MAP{$report_type}{$_} } sort keys %order_by_fields;
    if (@unsupported_field_names) {
        $vr->add_generic(
            error_BadParams(
                iget( 'В параметре %1$s указаны значения %2$s, недопустимые для типа отчета %3$s', FIELD_INSIDE_ORDER_BY, join(', ', @unsupported_field_names), $report_type )
            )
        );
    }

    my @dup_fields = grep {$order_by_fields{$_} > 1} keys %order_by_fields;
    foreach my $field (@dup_fields) {
        $vr->add_generic(
            error_DuplicateParameterValues(iget('В параметре %1$s значение %2$s указано более одного раза', FIELD_INSIDE_ORDER_BY, $field))
        );
    }

    my %field_names_set = map {$_ => 1} @$field_names;
    foreach my $field (keys %order_by_fields) {
        if ($DATE_AGGREGATION_MAP{$field} && !$field_names_set{$field}) {
            $vr->add_generic(error_BadParams(iget('В параметре %1$s указано значение %2$s, отсутствующее в параметре %3$s', FIELD_INSIDE_ORDER_BY, $field, FIELD_NAMES_FIELD)));
        }
    }

    my $exists_fields_dependent_on_goal = xisect([keys %FIELDS_DEPENDENT_ON_GOAL_MAP], [keys %order_by_fields]);
    if (@$exists_fields_dependent_on_goal && $has_goals) { 
        $vr->add_generic(
            error_BadParams(
                iget( 'В параметре %1$s значения %2$s недопустимы если задано поле Goals', FIELD_INSIDE_ORDER_BY, join(', ', @$exists_fields_dependent_on_goal) )
            )
        );
    }
    return $vr;
}

=head2 _validate_create_report_name($report_name)

    Валидируем заданное пользователем название отчета

=cut

sub _validate_create_report_name {
    my ($class, $report_name) = @_;

    my $vr = Direct::ValidationResult->new;

    if ($report_name =~ $Settings::DISALLOW_BANNER_LETTER_RE) {
        $vr->add(REPORT_NAME_FIELD() => error_BadParams(iget('В параметре %s содержатся запрещенные символы', REPORT_NAME_FIELD)));
    }

    if (length($report_name) > $MAX_REPORT_NAME_LENGTH) {
        $vr->add(REPORT_NAME_FIELD() => error_BadParams(iget('Значение в параметре %1$s должно содержать не более %2$d символов', REPORT_NAME_FIELD, $MAX_REPORT_NAME_LENGTH)));
    }
    return $vr;
}


=head2 _validate_dups_in_array_fields($item)

    Проверяем наличие дублей в массивах значений некоторых входных параметров

=cut

sub _validate_dups_in_array_fields {
    my ($self, $items, $field) = @_;
    my %items_hash;
    $items_hash{$_}++ foreach (@$items);
    my @duplicate = grep {$items_hash{$_} > 1} sort keys %items_hash;
    if (scalar @duplicate) {
        my $duplicate_str = join (',' => @duplicate);
        if (scalar @duplicate == 1) {
            return error_DuplicatedArrayItem(
                iget('Элемент %s присутствует в списке %s более одного раза', $duplicate_str, $field)
            );
        } else {
            return error_DuplicatedArrayItem(
                iget('Элементы %s присутствуют в списке %s более одного раза', $duplicate_str, $field)
            );
        }
    }
    return;
}

1;
