package API::Service::Request::Changes;

=pod

    $Id$

=head1 NAME

    API::Service::Request::Changes

=head1 SYNOPSIS

    use API::Service::Request::Changes;

    my $changes_request = API::Service::Request::Changes->new($request);

=head1 DESCRIPTION

    Методы для валидации и получения значений полей запроса сервиса Changes

=cut

use strict;
use warnings;
use utf8;
use feature 'state';

use List::MoreUtils qw/uniq/;
use Try::Tiny;

use Yandex::DateTime qw/iso8601_2_mysql/;
use Yandex::I18n;
use Yandex::Validate qw/is_valid_int/;

use Direct::Errors::Messages;

use Mouse;

my $CHECK_CAMPAIGN_IDS_LIMIT = 3000; # ограничение на количество переданных в запросе идентификаторов кампаний
my $CHECK_ADGROUP_IDS_LIMIT = 10000; # ограничение на количество переданных в запросе идентификаторов групп
my $CHECK_AD_IDS_LIMIT = 50000; # ограничение на количество переданных в запросе идентификаторов объявлений

has request => (is => 'ro', isa => 'HashRef');
has field_names => (is => 'ro', isa => 'HashRef', lazy => 1, builder => '_field_names_builder');
has need_cids => (is => 'ro', isa => 'Bool', lazy => 1, default => sub { exists shift->field_names->{CampaignIds} });
has need_stat => (is => 'ro', isa => 'Bool', lazy => 1, default => sub { exists shift->field_names->{CampaignsStat} });
has need_adgids => (is => 'ro', isa => 'Bool', lazy => 1, default => sub { exists shift->field_names->{AdGroupIds} });
has need_bids => (is => 'ro', isa => 'Bool', lazy => 1, default => sub { exists shift->field_names->{AdIds} });
has ids_field_name => (is => 'ro', isa => 'Str', lazy => 1, builder => '_ids_field_name_builder');
has has_cids => (is => 'ro', isa => 'Bool', lazy => 1, default => sub { shift->ids_field_name eq 'CampaignIds' });
has has_adgids => (is => 'ro', isa => 'Bool', lazy => 1, default => sub { shift->ids_field_name eq 'AdGroupIds' });
has has_bids => (is => 'ro', isa => 'Bool', lazy => 1, default => sub { shift->ids_field_name eq 'AdIds' });

sub BUILDARGS {
    my ($class, $request) = @_;

    return { request => $request };
}

=head2 has_timestamp

    Возвращает флаг наличия в запросе поля Timestamp

    die error_BadParams() unless $change_request->has_timestamp();

=cut

sub has_timestamp {
    my ($self) = @_;

    return exists $self->request()->{Timestamp} ? 1 : 0;
}

=head2 mysql_timestamp

    Проверяем наличие поля Timestamp в запросе. Конвертируем его в формат mysql: значение должно быть в формате YYYY-MM-DDThh:mm:ssZ и должно корректно преобразовываться в формат mysql.
    Если значение поля не прошло проверку или не сконвертировалось в формат mysql, генерируется исключение с объектом ошибки

    $mysql_timestamp = $change_request->mysql_timestamp();

=cut

sub mysql_timestamp {
    my ($self) = @_;

    die error_BadParams(iget('Параметр %s должен быть указан', 'Timestamp')) unless $self->has_timestamp();

    my $timestamp = $self->request()->{Timestamp};
    if ($timestamp !~ /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}Z$/) { # дата должна строго соответствовать шаблону
        die error_BadParams(iget('В параметре %s должна быть указана дата в формате YYYY-MM-DDThh:mm:ssZ', 'Timestamp'));
    }

    my $mysql_timestamp;
    try {
        $mysql_timestamp = iso8601_2_mysql($timestamp);
    } catch {
        die error_BadParams(iget('В параметре %s должна быть указана корректная дата', 'Timestamp')); # возвращаем объект ошибки
    };

    return $mysql_timestamp;
}

=head2 _field_names_builder

    Проверяем наличие поля FieldNames в запросе. Проверяем допустимость значений по списку.
    Возвращаем хэш с переданными значениями, либо генерируем исключение с объектом ошибки.
    Метод неявно вызывается при вызове field_names().

=cut

sub _field_names_builder {
    my ($self) = @_;

    my $request = $self->request();

    return { map { $_ => 1 } @{$request->{FieldNames}} };
}

=head2 _ids_field_name_builder

    Проверяем наличие только одного из полей CampaignIds, AdGroupIds, AdIds в запросе. Проверяем допустимость значений по списку.
    Возвращаем хэш с переданными значениями, либо генерируем исключение с объектом ошибки.
    Метод неявно вызывается при вызове ids_field_name().

=cut

sub _ids_field_name_builder {
    my ($self) = @_;

    my $request = $self->request();

    state $IDS_FIELD_NAMES = [qw/CampaignIds AdGroupIds AdIds/]; # используем в качестве констант локальные переменные, сохраняющие значение между вызовами и инициализирующиеся один раз
    my $ids_field_name;
    my $fields_count = 0;
    for my $field_name (@$IDS_FIELD_NAMES) {
        if (exists $request->{$field_name}) {
            last if ++$fields_count > 1;
            $ids_field_name = $field_name;
        }
    }

    if ($fields_count != 1) {
        state $IDS_FIELD_NAMES_STR = join(', ', @$IDS_FIELD_NAMES);
        die error_RequiredAtLeastOneOfParameters(iget('Один из параметров %s должен быть указан', $IDS_FIELD_NAMES_STR)) unless ($fields_count);
        die error_PossibleOnlyOneParameter(iget('Только один параметр из %s может быть указан', $IDS_FIELD_NAMES_STR));
    }

    return $ids_field_name;
}

=head2 uniq_cids()

    Проверяем идентификаторы кампаний, переданные в запросе, отфильтровываем дублирующиеся и нечисловые значения.
    Возвращается ссылка на массив с проверенными идентификаторами или генерируются исключения с объектом ошибки

    my $uniq_cids = $change_request->uniq_cids();

=cut

sub uniq_cids {
    return shift->_uniq_ids('CampaignIds', $CHECK_CAMPAIGN_IDS_LIMIT);
}

=head2 uniq_adgids()

    Проверяем идентификаторы групп, переданные в запросе, отфильтровываем дублирующиеся и нечисловые значения.
    Возвращается ссылка на массив с проверенными идентификаторами или генерируются исключения с объектом ошибки

    my $uniq_adgids = $change_request->uniq_adgids();

=cut

sub uniq_adgids {
    return shift->_uniq_ids('AdGroupIds', $CHECK_ADGROUP_IDS_LIMIT);
}

=head2 uniq_bids()

    Проверяем идентификаторы объявлений, переданные в запросе, отфильтровываем дублирующиеся и нечисловые значения.
    Возвращается ссылка на массив с проверенными идентификаторами или генерируются исключения с объектом ошибки

    my $uniq_bids = $change_request->uniq_bids();

=cut

sub uniq_bids {
    return shift->_uniq_ids('AdIds', $CHECK_AD_IDS_LIMIT);
}

=head2 _uniq_ids($field_name, $limit)

    Проверяем идентификаторы, переданные в указанном поле запроса, отфильтровываем дублирующиеся и нечисловые значения.
    Возвращается ссылка на массив с проверенными идентификаторами или генерируются исключения с объектом ошибки

    $uniq_cids = $change_request->_uniq_ids('CampaignIds', $CHECK_CAMPAIGN_IDS_LIMIT);

=cut

sub _uniq_ids {
    my ($self, $field_name, $limit) = @_;

    my $request = $self->request();

    if (defined $limit && @{$request->{$field_name}} > $limit) {
        die error_RequestLimitExceeded(iget('Количество элементов в параметре %s не должно превышать %d', $field_name, $limit));
    }

    my $filtered_ids = [ grep { is_valid_int($_) } uniq @{$request->{$field_name}} ];
    if (!@$filtered_ids) {
        die error_BadParams(iget('Массив %s не должен быть пустым', $field_name));
    }

    return $filtered_ids;
}

__PACKAGE__->meta->make_immutable();

1;

__END__
