package API::Service::Request::Get;

use strict;
use warnings;
use utf8;

use List::MoreUtils qw/any/;
use Carp;

use Yandex::I18n qw/iget/;
use Direct::Errors::Messages;
use Yandex::ListUtils qw/ xminus /;

=pod

    $Id$

=head1 NAME

    API::Service::Request::Get

=head1 SYNOPSIS

    use API::Service::Request::Get;

    my $get_request = API::Service::Request::Get->new($request);
    my $limit = $get_request->page_limit;
    my $offset = $get_request->page_limit;
    my $mapper = Model::Mapper::Keywords->new($shard);
    $mapper->set_limit($limit);
    $mapper->set_offset($offset);

    return $mapper->get_by_group_id($group_id)

=head1 DESCRIPTION

    Обертка для SOAP/WSDL Get

=head1 METHODS

=head2 new($request)

    $request запрос поступающий в операцию сервиса

=head2 page_limit

    Возвращает элемент Limit из Page, undef если не найден элемент Page или
    Limit

=head2 page_offset

    Возвращает элемент offset из Page, undef если не найден элемент Page или
    Offset

=head2 field_names

    Массив полей из FieldNames

=head2 has_selection_criteria

    True если в SelectionCriteria задан в запросе

=head2 has_selection_ids

    True если в SelectionCriteria есть <Что-то>Ids

=head2 selection_attribute($name)

    Список строк значений поля $name

=head2 selection_criteria

    SelectionCriteria из запроса, как есть

=head2 to_array($value)

    Преобразует ссылку на массив в список, скаляр в список из одного значения, нужно в случае если WSDL десериализоватор не преобразует массивы из одного элемента в массив (SOAP::WSDL например)

=head2 to_strings($arrayref_or_value)

    Список строк по ссылке на массив значений или одно значение 

=head2 to_numbers($arrayref_or_value)

    Список чисел по ссылке на массив значений или одно значение 

=head2 selection_ids($ids_name)

    Список числе, Id-шников, из SelectionCriteria по имени $ids_name

=cut

=head2 has_selection_attribute($criteria_name)

    True если в SelectionCriteria задан соответствующий элемент

=cut

=head2 is_in_field_names($field_name)

    True если поле запрошено в FieldNames

=cut

use Mouse;

# сделано переопределяемым
has default_limit => (is => 'rw', isa => 'Int', default => sub { 10_000 });
has request => (is => 'ro', isa => 'HashRef');
has _field_names_hash => (is => 'ro', isa => 'HashRef', lazy => 1, default => sub {
    return { map { $_ => undef } shift->field_names }
});

sub BUILDARGS {
    my $class = shift;
    my $request = shift;
    return { request => $request };
}

sub selection_criteria {
    my $self = shift;
    $self->request->{SelectionCriteria} //= {};
    return $self->request->{SelectionCriteria};
}

sub selection_ids {
    my $self = shift;
    my $criteria_name = shift;
    return $self->to_numbers($self->_selection($criteria_name));
}

sub selection_attribute {
    my $self = shift;
    my $criteria_name = shift;
    return $self->to_strings($self->_selection($criteria_name));
}

sub _selection {
    my $self = shift;
    my $criteria_name = shift or croak 'criteria name not set';
    return $self->selection_criteria->{$criteria_name};
}

sub page_limit {
    my $self = shift;
    return ( exists $self->request->{Page}
            && exists $self->request->{Page}->{Limit}
    ) ? $self->request->{Page}->{Limit} : $self->default_limit;
}

sub page_offset {
    my $self = shift;
    return ($self->request->{Page} ? $self->request->{Page}->{Offset} : 0) || 0;
}

sub field_names {
    my $self = shift;
    return $self->to_strings($self->request->{FieldNames});
}

sub has_selection_criteria {
     my $self = shift;
    return $self->request->{SelectionCriteria} ? 1 : 0;
}

sub has_selection_attribute {
    my $self = shift;
    my $criteria_name = shift;
    return defined $self->_selection($criteria_name) ? 1 : 0;
}

sub has_selection_ids {
    my $self = shift;
    for my $criteria (keys %{$self->selection_criteria}) {
        if ($criteria =~ /Ids$/) {
            return 1;
        }
    }
    return 0;
}

sub is_in_field_names {
    my ($self, $field_name) = @_;
    my @fields = $self->to_strings($self->request->{FieldNames});
    if ( any { $_ eq $field_name } @fields ) {
        return 1;
    }
    return;
}

sub to_array {
    my ($self, $field) = @_;

    return unless defined $field;

    if(ref $field eq 'ARRAY') {
        return @$field
    } else {
        return ( $field );
    }
}

sub to_strings {
    my $self = shift;
    my @fields = $self->to_array(shift) or ();
    return map { "$_" } @fields;
}

sub to_numbers {
    my $self = shift;
    my @fields = $self->to_array(shift) or ();
    return map { $_+0 } @fields;
}

=head2 validate(%rules)

    Валидирует SelectionCriteria и Page по заданным правилам %rules = (
        at_least_one => [список полей одно из которых необходимо], # может быть не задано
        limits => {
            field => $limit_spec # поле и его лимит
        }, enums => {
            field => [qw/возможные значения/]
        }
    )

    $limit_spec может быть целочисленным указанием либо ссылок на массив вида
    [ $limit, $error_description ]

    enums служит для указания возможных значений перечислений, для тех случаев,
    когда они недостаточно строго описаны в WSDL (описаны как строки или набор
    значений шире чем допустимый в данном случае)

    Лимит проверяется на не превышение default_limit, обнуление default_limit отменяет проверку

    В случае успеха валидации возвращает false, иначе ошибку

=cut

sub validate {
    my ($self, %rules) = @_;
    my @at_least_one = @{$rules{at_least_one}||[]};

    if($self->has_selection_criteria) {
        return error_WrongSelectionCriteria_NoNeededParams(undef, params => join(', ', @at_least_one ))
            if scalar @at_least_one && !(any { $self->_selection($_) and scalar @{$self->_selection($_)} } @at_least_one);

        foreach my $field (sort keys %{$rules{limits}||{}}) {
            my $values = $self->_selection($field) or next;
            my $limit_spec = $rules{limits}->{$field};
            my ($limit, $message) = (ref $limit_spec eq 'ARRAY' ? (@$limit_spec) : ($limit_spec, undef));
            if(@$values > $limit) {
                return error_WrongSelectionCriteria_LimitExceeded($message, param => $field, limit => $limit)
            }
        }

        foreach my $field (keys %{$rules{enums}||{}}) {
            my @values = @{ $rules{enums}->{$field} };
            if (my $e = $self->_check_selection_criteria_enum($field, @values)) {
                return $e;
            }
        }
    }

    return error_IncorrectPage_LimitExceeded(undef, limit => 10_000) if $self->default_limit && $self->page_limit > $self->default_limit;

    return error_IncorrectPage_NonpositiveLimit() if $self->page_limit <= 0;
    return error_IncorrectPage_NegativeOffset() if $self->page_offset < 0;

    return;
}

=head2 _check_selection_criteria_enum($attribute_name, @allowed_values)

    Проверяет значение указанного аттрибута в SelectionCriteria
        если значение задано не верно, то возвращает ошибку, undef в ином случае

=cut

sub _check_selection_criteria_enum {
    my ($self, $name, @allowed_values) = @_;
    if ( my @values = $self->selection_attribute( $name ) ) {
        my $diff = xminus( \@values, \@allowed_values);
        if ( @$diff ) {
            return error_WrongSelectionCriteria(
                iget('Поддерживаются только следующие значения для поля %s: %s', $name, join ', ' => @allowed_values),
            );
        }
    }
}

__PACKAGE__->meta->make_immutable();

1;

__END__
