package API::Service::Statuses;

use strict;
use warnings;
use utf8;

=pod

    $Id$

=head1 NAME

    API::Service::Statuses

=head1 SYNOPSIS

=head1 DESCRIPTION

    Модуль содержит методы, преднозначенные для работы со статусами и состояниями,
    необходимые для всех сервисов в которых есть статусы и состояния

=cut

use Exporter qw/import/;
our @EXPORT_OK = qw/get_field_value_depends_on_rules
                    merge_statuses_where
/;
use Carp qw/croak confess/;

use List::MoreUtils qw/any none/;
use Scalar::Util qw/looks_like_number/;
use Yandex::DateTime qw/now datetime/;

=head2 get_field_value_depends_on_rules($model, $rules_collection)

    Принимает модель (хэш с полями) и набор правил вида
    [
        [ RuleName1 => { condition_set11 }, { condition_set1K } ],
        [ RuleNameN => { condition_setN1 }, { condition_setNJ } ]
    ]
    таких как например $API::Service::Campaigns::Statuses::STATE_RULES

    В правила перебираются последовательно, в случае если значения полей модели
    отвечают хотя бы одному из наборов условий правила, то возращается имя этого правила.
    Если модель не подошла ни под одно правило, возвращается undef

    Правила совпадают по формату с sql_condition из Yandex::DBTools

=cut

sub get_field_value_depends_on_rules {
    my ($model, $rules_collection) = @_;
    for my $rule (@$rules_collection) {
        my ($key, @conditions_sets) = @$rule;
        for my $conditions_set (@conditions_sets) {
            return $key if _model_match_conditions($model, $conditions_set)
        }
    }
    return;
}

sub _model_match_conditions {
    my ($model, $conditions_set) = @_; # model_hash, { field__modifier => expected_value, .. }
    for my $expression (keys %$conditions_set) {
        next if $expression eq 'EXTRA_WHERE';
        my $matched;
        my $expected = $conditions_set->{$expression};
        my ($field, $operator, $dont_quote_flag) = _split_expression($expression);

        return unless exists $model->{$field}; # поля нет в модели
        my $got = $model->{$field};

        # одно условие не подошло, значит модель не подходит под набор
        return unless $dont_quote_flag
            ? _matcher_sql_expression($got,$operator, $expected)
            : _matcher($got,$operator, $expected);
    }

    return 1; # все условия подошли
}

sub _matcher_sql_expression {
    my ($got, $operator, $expected) = @_;
    confess "dont_quote can't be anything beside string" if ref $expected;
    return 1 if _mysql_expression_to_sub($operator, $expected)->($got);
}

sub _matcher { # got matches expected via operator
    my ($got, $operator, $expected) = @_;
    if ( $operator eq 'is_null' ) {
        return 1 if !defined $got;
    } elsif ( $operator eq 'is_not_null' ) {
        return 1 if defined $got;
    } elsif ( ref $expected eq 'ARRAY' ) {
        confess "operator $operator is not applicable for array" if $operator =~/lt|gt|le|ge/;
        if ($operator eq 'ne') {
            return 1 if none { equal($_, $got) } @$expected;
        } else {
            return 1 if any { equal($_, $got) } @$expected;
        }
    } else {
        return 1 if compare($got, $expected, $operator);
    }
    return;
}

sub _split_expression {
    my $condition = shift;
    my $operator;
    my $dont_quote_flag = 0;
    $condition =~ s/__sql_ignore$//g; # спец. модификатор нужен для игнорирования условия при формировании SQL WHERE
    my ($field, @modificators) = split('__', $condition);
    foreach my $modificator (@modificators) {
        confess "unknown modificator in condition $condition"
            unless $modificator =~ /^(dont_quote|eq|ne|lt|gt|le|ge|is_null|is_not_null)$/;
        if($modificator eq 'dont_quote') {
            $dont_quote_flag = 1;
        } else {
            $operator = $modificator;
        }
    }
    $operator //= 'eq';
    return ($field, $operator, $dont_quote_flag);
}

sub _mysql_expression_to_sub {
    my ($operator, $mysql_expression) = @_; # (string, string)
    my $sub;
    if( any { $_ eq $mysql_expression } ('NOW()', 'CURDATE()') ) {
        $sub = sub {
            my $value = shift or return;
            if ($value =~ /^\d{1,4}\-\d\d\-\d\d$/) {
                return compare($value, now()->ymd, $operator);
            } else {
                return compare(datetime($value)->epoch, time(), $operator);
            }
        };
    } else {
        confess "Unknown expression $mysql_expression";
    }
    return $sub;
}

=head2 equal

    Сравнивает чмсла как числа, строки как строки

=cut

sub equal {
    my ($x, $y) = @_;
    if (looks_like_number($x) && looks_like_number($y)){
       return $x == $y;
    } elsif (! defined $x || ! defined $y) {
       return ! defined $x && ! defined $y;
    } else {
       return $x eq $y;
    }
}

=head2 compare

    Сравнивает чмсла как числа, строки как строки
    Принимает: 2 числа и оператор (eq|ne|lt|gt)

=cut

sub compare {
    my ($x, $y, $operator) = @_;
    my $result;
    if (looks_like_number($x) && looks_like_number($y)){
       $result = $x <=> $y;
    } else {
       $result = ($x // '') cmp ($y // '');
    }
    if ($operator eq 'eq') {
        return $result == 0 ? 1 : 0;
    } elsif ($operator eq 'ne') {
        return $result == 0 ? 0 : 1;
    } elsif ($operator eq 'lt') {
        return $result == -1 ? 1 : 0;
    } elsif ($operator eq 'gt') {
        return $result == 1 ? 1 : 0;
    } elsif ($operator eq 'le') {
        return ($result == -1) || ($result == 0) ? 1 : 0;
    } elsif ($operator eq 'ge') {
        return ($result == 1) || ($result == 0) ? 1 : 0;
    }
}

1;
