package API::Units::Costs;

use strict;
use warnings;
use utf8;

use Carp qw/ confess /;
use Mouse;
use API::Settings;

=pod

=head1 NAME

    API::Units::Costs - класс, позволяет определить стоимость вызова любой из
    операций для любого из сервисов.

=head1 SYNOPSIS

    use API::Units::Costs qw/ get_costs /;

    my $costs = get_costs(keywords => add);

=head1 DESCRIPTION

    Каждая операция в API Direct-а имеет стоимость ее вызова в баллах, которая
    складывается из постоянной части - стоимости собственно вызова операции, и
    переменной части - суммы стоимостей для объектов - параметров операции.

=head1 VARIABLES

=head2 COSTS

    Таблица стоимостей, хеш вида
    (
        keywords => {
            add => {
                call          => 1,
                keyword       => 5,
                object_error  => 3,
                request_error => 3,
            },
        },
    )

=head2 APPLICATION_MULTIPLIERS

    Таблица бальных множителей для приложений, хэш вида

    (
        application_id => multiplier,
    )
    где множитель это число с плавающей точкой

=cut

our %APPLICATION_MULTIPLIERS = (
    $API::Settings::HMEPAS_APP_ID             => 1.5,  # hmepas
    $API::Settings::AUTO_TESTS_APP_ID         => 0.8,  # автотесты
    $API::Settings::DC2_APP_ID                => 0.3,  # Директ.Коммандер
    $API::Settings::COMMANDER_APP_ID          => 0.3,  # Директ.Коммандер (на node.js)
    $API::Settings::YNDX_SERVICES_APP_ID      => 0.01, # Приложение Услуг
    $API::Settings::YNDX_SERVICES_2_APP_ID    => 0.01, # Приложение Услуг (Заполнение кампаний)
    $API::Settings::GEOPRODUCT_APP_ID         => 0.01, # Приложение Геопродукта
    $API::Settings::K50_1_APP_ID              => 0.1, # Приложение К50 №1
    $API::Settings::K50_2_APP_ID              => 0.1, # Приложение К50 №2
    $API::Settings::ZEN_PROMO_ID              => 0.01, # Дзен Промо
    $API::Settings::ZEN_APP_ID                => 0.01, # Дзен Поддержка авторов
);

my $DEFAULT_ERROR_COST = 20;

=head2 $COMMON_REQUEST_ERROR

    Стоимость ошибки на случай если до инициализации сервиса дело не дошло
    (ошибка парсинга запроса), но пользователя удалось авторизовать

=cut

our $COMMON_REQUEST_ERROR = 50;

our %COSTS = (
    'Campaigns' => {
        get => {
            call          => 10,
            campaign      => 1,
            request_error => $DEFAULT_ERROR_COST,
        },
        add => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        update => {
            call          => 10,
            campaign      => 3,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        delete => {
            call          => 10,
            campaign      => 2,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        suspend => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        resume => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        archive => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        unarchive => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
    },
    'CampaignsExt' => {
        get => {
            call          => 10,
            campaign      => 1,
            request_error => $DEFAULT_ERROR_COST,
        },
        add => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        update => {
            call          => 10,
            campaign      => 3,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        delete => {
            call          => 10,
            campaign      => 2,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        suspend => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        resume => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        archive => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        unarchive => {
            call          => 10,
            campaign      => 5,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
    },
    'VCards' => {
        get => {
            call          => 15,
            vcard         => 1,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        add => {
            call          => 20,
            vcard         => 20,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        delete => {
            call          => 10,
            vcard         => 0,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
    },
    'Sitelinks' => {
        get => {
            call          => 15,
            set           => 1,
            request_error => $DEFAULT_ERROR_COST,
        },
        add => {
            call          => 20,
            set           => 20,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        delete => {
            call          => 10,
            set           => 0,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
    },
    'Changes' => {
        checkDictionaries => {
            call => 10,
            request_error => $DEFAULT_ERROR_COST,
        },
        checkCampaigns => {
            call => 10,
            request_error => $DEFAULT_ERROR_COST,
        },
        check => {
            call => 10,
            request_error => $DEFAULT_ERROR_COST,
        }
    },
    Reports => {
        create => {
            call => 0,
            object_error => 0,
            request_error => 0,
        },
    },
    'AdExtensions' => {
        add => {
            call          => 5,
            callout       => 1,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        get => {
            call          => 5,
            callout       => 1,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        delete => {
            call          => 10,
            callout       => 1,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
    },
    'AdImages' => {
        get => {
            call          => 15,
            adimage       => 1,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        add => {
            call          => 20,
            adimage       => 20,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
        delete => {
            call          => 10,
            adimage       => 0,
            object_error  => $DEFAULT_ERROR_COST,
            request_error => $DEFAULT_ERROR_COST,
        },
    },
);

=head2 service

    Название сервиса, для операций которого рассчитываются стоимости

=cut

has service => ( is => 'ro', isa => 'Str', required => 1 );

=head2 operation

    Название операции, для которой рассчитывается стоимость

=cut

has operation => ( is => 'ro', isa => 'Str', required => 1 );

=head1 METHODS

=head2 new

    Конструктор

    Параметры:
        service   - название сервиса, строка, обязательный
        operation - название операции, строка, обязательный

    Результат:
        Объект класса API::Units::Costs

=cut

sub BUILDARGS {
    my ( $class, $service, $operation ) = @_;

    confess 'Bad service given' unless $service and exists $COSTS{ $service };

    my $service_costs = $COSTS{ $service };

    confess 'Bad operation given' unless $operation and exists $service_costs->{ $operation };

    return {
        service   => $service,
        operation => $operation,
    };
}

sub _get_cost {
    my ( $self, $type ) = @_;

    return $COSTS{ $self->service }->{ $self->operation }->{ $type } || 0;
}

=head2 get_cost_for_operation

    Возвращает стоимость вызова операции

    Параметры:
        нет

    Результат:
        Число, стоимость вызова операции в баллах

=cut

sub get_cost_for_operation {
    my $self = shift;

    return $self->_get_cost('call');
}

=head2 get_cost_for_objects

    Возвращает стоимость вызова операции для указанного количества объектов
    указанного типа

    Параметры:
        type   - тип объектов - аргументов операции, строка
        number - количество объектов - аргументов операции, число

    Результат:
        Число, стоимость в баллах

=cut

sub get_cost_for_objects {
    my ( $self, $type, $number ) = @_;

    return $self->_get_cost( $type ) * ( $number || 0 );
}

=head2 get_cost_for_objects_with_errors

    Возвращает стоимость в баллах для объектов с ошибками

    Параметры:
        number - количество объектов - аргументов операции с ошибками, число

    Результат:
        Число, стоимость в баллах

=cut

sub get_cost_for_objects_with_errors {
    my ( $self, $number ) = @_;

    return $self->_get_cost('object_error') * ( $number || 0 );
}

=head2 get_cost_for_request_error

    Возвращает стоимость ошибки запроса

    Параметры:
        нет

    Результат:
        Число, стоимость ошибки запроса в баллах

=cut

sub get_cost_for_request_error {
    my $self = shift;

    return $self->_get_cost('request_error');
}

=head2 get_app_coef($application_id)

    По строке с applicaiton_id возвращает множитель по списываемым баллам в виде числа с плавающей точкой

=cut

sub get_app_coef {
    my ($application_id) = @_;
    return exists $APPLICATION_MULTIPLIERS{$application_id} && defined $APPLICATION_MULTIPLIERS{$application_id}
        ? $APPLICATION_MULTIPLIERS{$application_id}
        : 1;
}

__PACKAGE__->meta->make_immutable();

1;
