#!/usr/bin/perl
use Direct::Modern;

use Test::More;
use Test::Exception;

=pod

    $Id$

=head1 NAME

    schema_validation.t

=head1 DESCRIPTION

    Тест проверяющий что WSDL::JSON::Schema создает схему по валидной с нашей
    точки зрения WSDL-ки и не создает по не валидным. Под валидностью
    понимается соответствие нашиму соглашению о том какие фичи WSDL-я мы
    используем при описании своих сервисов, подробнее см WSDL::JSON::Schema;

    Также чтобы не создавать схему два раза в этом же тесте мы проверяем
    WSDL::JSON::Validate

=cut

use lib::abs;

use Try::Tiny;
use Storable qw/dclone/;

use my_inc '../../../..';

use API::Services;
use WSDL::JSON::Schema;
use WSDL::JSON::Schema::Sequence;
use WSDL::JSON::Validate;

use XML::LibXML;

binmode(STDOUT, ":utf8");

my $parser = XML::LibXML->new;
my $path_wsdls = lib::abs::path('./wsdls');

my $services = API::Services->new([
    "$path_wsdls/ok.wsdl"
]);

my $nss = $services->namespaces;

subtest ValidWSDLOperations => sub {
    $services->map_thru_service_operation_ns_parts(sub {
        my ($service, $operation, $ns, $request, $response) = @_;
        my $schema = WSDL::JSON::Schema->new($nss, $ns);
        ok(my $req_schema = $schema->build($request), "$service $operation request json schema builded");
        ok(my $response_schema = $schema->build($response), "$service $operation response json schema builded");
        my %req = (ReshardRequest => $req_schema);

        my $params = { Reshard => {
            JobId => 123,
            DestinationShard => 6,
            HowSoon => 'TODAY',
            IgnoreCampaignIds => [qw/1 2/],
            SomeExtraData => [],
            SomeDecimal => 10.0,
            SomeString => 'asfd',
        }};

        ok(exists $params->{Reshard}{SomeExtraData}, "empty array present");
        ok(v(%req, $params), "proper request validated");
        ok(!exists $params->{Reshard}{SomeExtraData}, "empty array removed after validation");

        validation_fails(%req, {
            Reshard => {
                JobId => undef,
            }
        }, "JobId не может иметь значение null");

        ok(v(%req, {
            Reshard => {
                JobId => 123,
                DestinationShard => undef,
                HowSoon => 'TODAY'
            }
        }), "undef is ok for nillable");

        validation_fails(%req, [], "ReshardRequest не может содержать массив");

        validation_fails(%req, {
            Reshard => {
                JobId => 123,
                SomeExtraData => 123,
                HowSoon => 'TODAY',
                DestinationShard => 7,
            }
        }, "SomeExtraData должен содержать массив");

        validation_fails(%req, {
            Reshard => {
                JobId => 123,
                SomeExtraData => [undef],
                HowSoon => 'TODAY',
                DestinationShard => 7,
            }
        }, "Элемент массива SomeExtraData не может иметь значение null");

        validation_fails(%req, {
            Reshard => {
                JobId => "asdf",
                HowSoon => 'TODAY',
                DestinationShard => 7,
            }
        }, "JobId должен содержать целочисленное значение");

        validation_fails(%req, {
            Reshard => {
                JobId => 123,
                HowSoon => 'TODAY',
                DestinationShard => 7,
                Flags => [ "BABA" ]
            }
        }, "Элемент массива Flags содержит неверное значение перечисления");

        validation_fails(%req, {
            Reshard => {
                JobId => 123,
                HowSoon => 'TODAY',
                DestinationShard => 7,
                Users => [
                    { Id => 123 }
                ]
            }
        }, "В элементе массива Users отсутствует обязательное поле Name");

        validation_fails(%req, {
            Reshard => {
                JobId => 123,
                HowSoon => 'WRONG_ENUM'
            }
        }, "HowSoon содержит неверное значение перечисления");

        validation_fails(%req, {
            Reshard => {
                JobId => 123,
                UnknownField => 'asdf',
            }
        }, "Reshard содержит неизвестное поле UnknownField");

        validation_fails(%req, {
            Reshard => 123
        }, "Reshard должен содержать объект");

        validation_fails(%req, {
            Reshard => {
                JobId => 123
            }
        }, "В Reshard отсутствует обязательное поле HowSoon");

        validation_fails(%req, {
            Reshard => {
                JobId => 123,
                HowSoon => 'TODAY',
                DestinationShard => 7,
                SomeDecimal => 'asfd'
            }
        }, "SomeDecimal должен содержать дробное значение");

        validation_fails(%req, {
            Reshard => {
                JobId => 123,
                SomeString => {},
                HowSoon => 'TODAY',
                DestinationShard => 7,
            }
        }, "SomeString должен содержать строку");

        my $wrong_types = dclone($req_schema);
        $wrong_types->{type}->element('Reshard')->{type}->element('DestinationShard')->{type} = 'unknown';
        validation_fails(ReshardRequest => $wrong_types, {
            Reshard => {
                JobId => 123,
                HowSoon => 'TODAY',
                DestinationShard => 7,
                DestinationShard => 123
            }
        }, "unknown type");
        $wrong_types->{type}->element('Reshard')->{type} = bless {}, 'Unknown';
        validation_fails(ReshardRequest => $wrong_types, {
            Reshard => (bless {}, 'Smth')
        }, "strange type");

        $wrong_types->{type}->element('Reshard')->{type} = {};
        validation_fails(ReshardRequest => $wrong_types, {
            Reshard => (bless {}, 'Smth')
        }, "strange type");

        $wrong_types->{type}->element('Reshard')->{type} = WSDL::JSON::Schema::Sequence->new(name => 'strange');
        validation_fails(ReshardRequest => $wrong_types, {
            Reshard => (bless {}, 'Smth')
        }, "strange type");

        # Легально, через WSDL::Schema::JSON такую схему получить нельзя
        # (намеренное ограничение)
        my $min_max = dclone($req_schema);

        my $extra_data = $min_max->{type}->element('Reshard')->{type}->element('SomeExtraData');
        $extra_data->{min} = 2;
        $extra_data->{max} = 2;
        validation_fails(ReshardRequest => $min_max, {
            Reshard => {
                JobId => 123,
                SomeExtraData => [qw/1/],
                DestinationShard => 7,
                HowSoon => 'TODAY'
            }
        }, "SomeExtraData должен содержать не менее 2 элемент");
        ok(v(ReshardRequest => $min_max, {
            Reshard => {
                JobId => 123,
                SomeExtraData => [qw/1 2/],
                DestinationShard => 7,
                HowSoon => 'TODAY'
            }
        }), 'proper facets passed');
        validation_fails(ReshardRequest => $min_max, {
            Reshard => {
                JobId => 123,
                SomeExtraData => [qw/1 2 3/],
                DestinationShard => 7,
                HowSoon => 'TODAY'
            }
        }, "SomeExtraData не может содержать более 2 элемент");
        dies_ok(sub {$schema->build("some_unknown_element")}, "dies on nonfound element");
    });
};


subtest InValidWSDL => sub {
    my $schema = WSDL::JSON::Schema->new($nss, 'invalid');

    xsd_fails($schema, '<xsd:element name="Fail" type="xsd:int" minOccurs="-1" />', 'minOccurs must be greater than 0 or equal');
    xsd_fails($schema, '<xsd:element name="Fail" type="xsd:int" minOccurs="0" maxOccurs="0" />', 'maxOccurs must be greater than 0');
    xsd_fails($schema, '<xsd:element name="Fail" type="xsd:int" minOccurs="2" maxOccurs="1" />', 'maxOccurs must be greater or equal to minOccurs');
    xsd_fails($schema, '<xsd:element name="Fail" type="xsd:int" maxOccurs="3" />', 'maxOccurs must be 1 or unbounded');
    xsd_fails($schema, '<xsd:element name="Fail" type="xsd:int" minOccurs="2" maxOccurs="unbounded"/>', 'minOccurs must be 0 or 1');
    xsd_fails($schema, '<xsd:element name="Fail" type="xsd:int" minOccurs="0" maxOccurs="unbounded" nillable="true" />', "array can't be nillable");
    xsd_fails($schema, '<xsd:element name="Fail">
            <xsd:complexType>
            </xsd:complexType>
        </xsd:element>
    ', 'complexType can have only 1 children');
    xsd_fails($schema, '<xsd:element name="Fail">
            <xsd:complexType>
                <xsd:choice>
                    <xsd:element name="one"/>
                    <xsd:element name="two"/>
                </xsd:choice>
            </xsd:complexType>
        </xsd:element>
    ', 'complexType must either consists of sequence or consists of complexContent');
    xsd_fails($schema, '<xsd:element name="Fail">
        <xsd:complexType>
            <xsd:complexContent><xsd:extension></xsd:extension></xsd:complexContent>
        </xsd:complexType>
    </xsd:element>
    ', 'extension must have base attribute');

    # more than two childs
    xsd_fails($schema, '<xsd:element name="Fail">
        <xsd:complexType>
            <xsd:complexContent>
                <xsd:element name="first" />
                <xsd:element name="two" />
            </xsd:complexContent>
        </xsd:complexType>
    </xsd:element>
    ', 'must consists of one');

    # one child but wrong
    xsd_fails($schema, '<xsd:element name="Fail">
        <xsd:complexType>
            <xsd:complexContent>
                <xsd:element name="first" />
            </xsd:complexContent>
        </xsd:complexType>
    </xsd:element>
    ', 'must consists of one');

    xsd_fails($schema, '<xsd:element name="Fail">
        <xsd:simpleType>
            <xsd:restriction>
                <xsd:element name="first" />
            </xsd:restriction>
        </xsd:simpleType>
    </xsd:element>
    ', 'wrong tag in restriction');

    xsd_fails($schema, '<xsd:element name="Fail">
        <xsd:complexType />
        <xsd:simpleType />
    </xsd:element>
    ', 'element must have complexType child or no children at all');

    xsd_fails($schema, '<xsd:element/>', 'element has no name attribute');
    xsd_fails($schema, '<xsd:element name="foo">
        <xsd:sometag></xsd:sometag>
    </xsd:element>
    ', 'element can only consist of one complexType or simpleType sub element');
    xsd_fails($schema, '<xsd:element name="foo"/>', 'must either consist of type definition or has type attribute');
    xsd_fails($schema, '<xsd:element unknown_attr="1" name="foo" type="xsd:int"/>', 'unrecognized attributes');
    xsd_fails($schema, '<xsd:element name="WrongXsdType" type="xsd:foobar"/>', 'unknown simple datatype');

    xsd_fails($schema, '<xsd:element name="Fail" type="ns:SomeUnknownType"/>', 'not found');

    xsd_fails($schema, '<xsd:element name="Fail" type="xsd:int" nillable="true"/>', " can't be nillable");

    xsd_fails($schema, '<xsd:element name="Fail" type="xsd:int" maxOccurs="unbounded"/>', " can't have maxOccur other than 1");

    xsd_fails($schema, '<xsd:element name="Fail" type="xsd:int" minOccurs="0"/>', " can't have minOccur other than 1");

    my $builded_schema = $schema->build_by_node(
        '{invalid}some_element' => $parser->load_xml(
            string => wsdl_wrap('<xsd:element name="root">
                <xsd:complexType>
                    <xsd:sequence>
                        <xsd:element name="foo" type="xsd:string" />
                        <invalid:element name="wrong" type="xsd:string" />
                        <xsd:element name="bar" type="xsd:string" />
                    </xsd:sequence>
                </xsd:complexType>
            </xsd:element>')
        )->documentElement->firstChild
    );
    is_deeply([ $builded_schema->{type}->element_names ], [qw/foo bar/], 'wrong namespace element ignored');
};

sub validation_fails {
    my ($name, $schema, $data, $msg, $debug) = @_;
    _happens(sub {
        v($name => $schema, $data)
    }, $msg, $debug);
}

sub xsd_fails {
    my ($schema, $xml, $msg, $debug) = @_;
    my $node = $parser->load_xml(string => wsdl_wrap($xml))->documentElement->firstChild;
    _happens(sub {
        $schema->build_by_node('{invalid}'.$node->localName => $node);
    }, $msg, $debug);
}

sub _happens {
    my ($do, $expected, $debug) = @_;

    my $error = '';
    try {
        &$do();
    } catch {
        $error = ref $_ ? $_->description : $_;
        warn $error if $debug;
    };

    ok($error =~ /$expected/, "fails by $expected");
}

sub wsdl_wrap {
    '<wsdl:definitions
        xmlns:invalid="http://api.direct.yandex.com/v5/invalid"
        xmlns:wsdl="http://schemas.xmlsoap.org/wsdl/"
        xmlns:xsd="http://www.w3.org/2001/XMLSchema">'
    . $_[0]
    . '</wsdl:definitions>'
}

sub v { WSDL::JSON::Validate::validate(@_) }

done_testing;
