#!/usr/bin/perl

use Direct::Modern;

use HashingTools qw/hmac_sha256_hex/;
use TextTools qw/normalize_login/;

use Test::More tests => 7;

BEGIN {
    use_ok('API::Authorization::CheckToken');
}

my $login = "qwe-qwe";
my $blackbox_info = {
    client_id => "544e969cfc6d4e38827f957fafa02d7f",
    dbfield => {
        'accounts.login.uid' => $login
    },
    scope => {
        'direct:api' => 1
    },
    status => "VALID",
    uid => 42513537
};

my $not_valid_blackbox_info = {
    client_id => "544e969cfc6d4e38827f957fafa02d7f",
    dbfield => {
        'accounts.login.uid' => $login
    },
    scope => {
        'direct:api' => 1
    },
    status => "INVALID",
    uid => 42513537
};

my $blackbox_info_for_not_valid_scope = {
    client_id => "544e969cfc6d4e38827f957fafa02d7f",
    dbfield => {
        'accounts.login.uid' => $login
    },
    scope => {
        'strange:scope' => 1
    },
    status => "VALID",
    uid => 42513537
};

my $token = '1473d7f3c1ec4aa5ad9a58cc74c6814e';

subtest 'valid authorize' => sub {
    my $auth_info = API::Authorization::CheckToken->authorize($token, '1473d7f3c1ec4aa5ad9a58cc74c6814e', $blackbox_info);

    is( ref $auth_info, 'HASH' );
    is_deeply( $auth_info, {
        uid => 42513537,
        login => normalize_login($login),
        application_id => "544e969cfc6d4e38827f957fafa02d7f",
    });
};

subtest 'not valid authorize' => sub {
    my $auth_info = API::Authorization::CheckToken->authorize($token, '1473d7f3c1ec4aa5ad9a58cc74c6814e', $not_valid_blackbox_info);
    is( $auth_info, undef );
};

subtest 'authorize for not valid scope' => sub {
    my $auth_info = API::Authorization::CheckToken->authorize($token, '1473d7f3c1ec4aa5ad9a58cc74c6814e', $blackbox_info_for_not_valid_scope);
    is( $auth_info, undef );
};

my $ptoken     = 'token1';
my $bad_ptoken = 'token2';

{
    no warnings qw/ once redefine /;
    *API::Authorize::PersistentToken::get_uid = sub ($$) {return 1};
};

my $ptoken_key = $API::Settings::API_PERSISTENT_TOKENS_KEY = 'key';
sub ptoken_hmac { hmac_sha256_hex($_[0], $ptoken_key) }

my $ptokens = $API::Settings::API_PERSISTENT_TOKENS = {
    ptoken_hmac( $ptoken ) => {
        login => 'l1',
        application_id => 'a1',
        allow_to => ['networks:private.txt'],
    },
};

subtest 'valid authorize (by persistent token)' => sub {
    my $auth_info = API::Authorization::CheckToken->authorize_by_persistent_token($ptoken, '127.0.0.1');

    is( ref $auth_info, 'HASH' );
    is_deeply( $auth_info, {uid => 1, login => 'l1', application_id => 'a1'} );
};

subtest 'not valid authorize (by persistent token) - bad token' => sub {
    my $auth_info = API::Authorization::CheckToken->authorize_by_persistent_token($bad_ptoken, '127.0.0.1');
    is( $auth_info, undef );
};

subtest 'not valid authorize (by persistent token) - bad ip' => sub {
    my $WARN_WAS = '';
    local $SIG{__WARN__} = sub { $WARN_WAS = $_[0] };
    my $auth_info = API::Authorization::CheckToken->authorize_by_persistent_token($ptoken, '8.8.8.8');
    is( $auth_info, undef );
    ok($WARN_WAS =~ m/token for login l1 application a1 authorized, but not from allowed network\. IP: 8\.8\.8\.8 at/, "got right warning about network restriction violation");
};
