#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use Yandex::HashUtils qw/hash_cut/;

use Test::More tests => 14;
use Test::Exception;

use my_inc '../../', for => 'api/t';

BEGIN {
    use_ok('API::Converter');
    use_ok('API::Converter::ConvertMap');
    
}

my $map = {
    id => 'Id',
    adgroup_name => 'Name',
    banners => [ Banners => {
        id => 'Id',
        body => 'Text',
        title => 'Title'
    }],
    flags => sub {
        return { AgeLable => 18, Warning => 'space' }
    },
    price => [Bid => sub { $_[0] * 1000_000 }],
    field_need_create_nested => [ level1 => 'level2'],
    field_need_remove_nested => {
        nested => 'flat'
    },
    crossing_nested_field => [ level1 => 'another_level2'],
    field_need_create_nested_and_use_sub => [ level1 => level2 => sub { uc $_[0] }],
};

my $data = {
    id => 42,
    adgroup_name => 'Kepler-205',
    banners => [{
        id => 420,
        body => 'на любой случай жизни',
        title => 'сверхмассивная черная дыра'
    }, {
        id => 421,
        body => 'ваш праздник будет незабываем',
        title => 'большой взрыв на заказ'
    }
    ],
    flags => 'age:18,space',
    price => 3.12,
    strange_field => 'qwe',
    field_need_create_nested => 'qwe',
    field_need_remove_nested => {
        nested => 'qwe'
    },
    crossing_nested_field => 'qwe',
    field_need_create_nested_and_use_sub => 'qwe',
};

subtest 'creates correct converter object' => sub {
    isa_ok(API::Converter->new(map => {}), 'API::Converter');
};

subtest 'creates correct map object' => sub {
    isa_ok(API::Converter::ConvertMap->new(method => 'dummy', method => 'dummy', to_external_map => $map), 'API::Converter::ConvertMap');
};

subtest 'create correct converter with not empty map' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    lives_ok{ API::Converter->new(map => $convert_map->to_external_map) };
};

subtest 'correct convert only field names' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    my $converter = API::Converter->new(map => $convert_map->to_external_map);
    is_deeply ($converter->convert(hash_cut($data, qw/id adgroup_name/)),
        {
            Id => 42,
            Name => 'Kepler-205'
        }
    );
};

subtest 'correct convert field name and value' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    my $converter = API::Converter->new(map => $convert_map->to_external_map);
    is_deeply ($converter->convert(hash_cut($data, qw/price/)),
        {
            Bid => $data->{price} * 1000_000
        }
    );
};

subtest 'correct convert structure' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    my $converter = API::Converter->new(map => $convert_map->to_external_map);
    is_deeply ($converter->convert(hash_cut($data, qw/banners/)),
        {
            Banners => [{
                Id => 420,
                Text => 'на любой случай жизни',
                Title => 'сверхмассивная черная дыра'
            }, {
                Id => 421,
                Text => 'ваш праздник будет незабываем',
                Title => 'большой взрыв на заказ'
            }],
        }
    );
};

subtest 'correct convert one field to more then one fields' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    my $converter = API::Converter->new(map => $convert_map->to_external_map);
    is_deeply ($converter->convert(hash_cut($data, qw/flags/)),
        {
            AgeLable => 18,
            Warning => 'space'
        }
    );
};

subtest 'incorrect convert field without rule' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    my $converter = API::Converter->new(map => $convert_map->to_external_map);
    is_deeply ($converter->convert(hash_cut($data, qw/strange_field/)), {});
};


subtest 'create nesting for field' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    my $converter = API::Converter->new(map => $convert_map->to_external_map);
    is_deeply ($converter->convert(hash_cut($data, qw/field_need_create_nested/)),
               {level1 => {level2 => 'qwe'}}
    );
};

subtest 'remove nesting for field' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    my $converter = API::Converter->new(map => $convert_map->to_external_map);
    is_deeply ($converter->convert(hash_cut($data, qw/field_need_remove_nested/)),
               {flat => 'qwe'}
    );
};

subtest 'convert crossing nested fields' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    my $converter = API::Converter->new(map => $convert_map->to_external_map);
    is_deeply ($converter->convert(
            hash_cut($data, qw/field_need_create_nested crossing_nested_field/)
        ),
        { level1 => {level2 => 'qwe', another_level2 => 'qwe'} }
    );
};

subtest 'create nesting for field and convert value' => sub {
    my $convert_map = API::Converter::ConvertMap->new(method => 'dummy', to_external_map => $map);
    my $converter = API::Converter->new(map => $convert_map->to_external_map);
    is_deeply ($converter->convert(
            hash_cut($data, qw/field_need_create_nested_and_use_sub/)
        ),
        { level1 => {level2 => 'QWE'} }
    );
};









