#!/usr/bin/perl

use Direct::Modern;

use Test::More;
use Test::Exception;

use Yandex::DateTime qw/now/;

use API::Service::Statuses qw/get_field_value_depends_on_rules/;

sub split_eq {
    my $expression_string = shift;
    my @expected_result = @_;
    is_deeply(
        [ API::Service::Statuses::_split_expression($expression_string) ],
        \@expected_result,
        "$expression_string parsed"
    );
}

sub matches_ok { #matches
    my ($got, $operator, $expected) = @_;
    ok(API::Service::Statuses::_matcher($got, $operator, $expected), "$got $operator: $expected");
}

sub wont_match { #matches
    my ($got, $operator, $expected) = @_;
    ok(!API::Service::Statuses::_matcher($got, $operator, $expected), "$got $operator: $expected wont match");
}

split_eq('date__le__dont_quote', 'date', 'le', 1);
split_eq('date__dont_quote', 'date', 'eq', 1);
split_eq('date__gt', 'date', 'gt', 0);
split_eq('date__is_null', 'date', 'is_null', 0);
split_eq('date__is_not_null', 'date', 'is_not_null', 0);

ok(get_field_value_depends_on_rules({ value1 => 1, value2 => 2 }, [
    [ RULE2 => { value2 => 2 } ],
    [ RULE1 => { value1 => 1 } ],
]) eq 'RULE2', 'first matching rule fires');

ok(get_field_value_depends_on_rules({ value1 => 1, value2 => 2, value3 => 3 }, [
    [ RULE1 => { value1 => 1, value3 => 3 } ],
    [ RULE2 => { value2 => 2 } ],
]) eq 'RULE1', 'multifield condition');

ok(get_field_value_depends_on_rules({ value1 => 1, value2 => 2, value3 => 3 }, [
    [ RULE2 => { value2 => 1 } ],
    [ RULE1 => { value1 => 123 }, { value3 => 3 } ],
]) eq 'RULE1', 'multicondition matching');

ok(!defined get_field_value_depends_on_rules({ value1 => 11, value2 => 22, value3 => 33 }, [
    [ RULE2 => { value2 => 1 } ],
    [ RULE1 => { value1 => 123 }, { value3 => 3 } ],
]), 'no matching returs undef');

dies_ok { get_field_value_depends_on_rules({ value2 => 11 }, [
    [ RULE2 => { value2__dont_quoteeee => 'NOW()' } ],
]) } 'unknown operator causes exception';

dies_ok { get_field_value_depends_on_rules({ value2 => 11 }, [
    [ RULE2 => { value2__dont_quote => [qw/a b c/] } ],
]) } 'non scalar expected while dont_quote fails';

foreach my $op (qw/lt gt le ge/) {
    dies_ok { get_field_value_depends_on_rules({ value => 11 }, [
        [ RULE => { "value__$op" => [qw/a b c/] } ],
    ]) } "$op for array fails";
}

dies_ok { get_field_value_depends_on_rules({ value2 => 11 }, [
    [ RULE2 => { value2__dont_quote => 'NAU()' } ],
]) } 'unknown mysql expression fails';

ok(get_field_value_depends_on_rules({ value1 => 1, value2 => 2, date => now()->ymd }, [
    [ RULE1 => { value1 => 123 }, { value3 => 3 } ],
    [ HMEPAS_BDAY => { date__dont_quote => 'NOW()' } ],
]) eq 'HMEPAS_BDAY', 'NOW() works for dates');

ok(get_field_value_depends_on_rules({ value1 => 1, value2 => 2, date => '1970-01-01 11:58:55' }, [
    [ RULE1 => { value1 => 123 }, { value3 => 3 } ],
    [ HMEPAS_BDAY => { date__le__dont_quote => 'NOW()' } ],
]) eq 'HMEPAS_BDAY', 'NOW() works for datetimes');

ok(!defined get_field_value_depends_on_rules(
    { value1 => 1, value2 => 2, date => '1970-01-01 11:58:55' },
    [
        [ RULE1 => { value1 => 123 }, { value3 => 3 } ],
        [ HMEPAS_BDAY => { date__ge__dont_quote => 'NOW()' }
    ],
]), 'NOW() works for datetimes negative test');

ok(get_field_value_depends_on_rules({ value => 10 }, [
    [ le => { value__le => 10 } ],
]) eq 'le', 'le works for equals');

ok(get_field_value_depends_on_rules({ value => 10 }, [
    [ le => { value__le => 10 } ],
]) eq 'le', 'le works for equals');

ok(get_field_value_depends_on_rules({ value => 10 }, [
    [ ge => { value__ge => 10 } ],
]) eq 'ge', 'ge works for equals');

ok(get_field_value_depends_on_rules({ value => 10 }, [
    [ ge => { value__ge => 10 } ],
]) eq 'ge', 'ge works for equals');

ok(get_field_value_depends_on_rules({ value => 10 }, [
    [ le => { value__le => 20 } ],
]) eq 'le', 'le works');

ok(get_field_value_depends_on_rules({ value => 10 }, [
    [ ge => { value__ge => 5 } ],
]) eq 'ge', 'ge works');

ok(get_field_value_depends_on_rules({ value => 10 }, [
    [ gt => { value__gt => 5 } ],
]) eq 'gt', 'gt works');

ok(get_field_value_depends_on_rules({ value => 10 }, [
    [ lt => { value__lt => 20 } ],
]) eq 'lt', 'le works');

ok(get_field_value_depends_on_rules({ value => 3 }, [
    [ eq => { value__eq => [1, 2, 3, 5] } ],
]) eq 'eq', 'eq works for arrays');

ok(get_field_value_depends_on_rules({ value => "b" }, [
    [ eq => { value__eq => ["a", "b", "c"] } ],
]) eq 'eq', 'eq works for arrays of strings');

ok(get_field_value_depends_on_rules({ value => 6 }, [
    [ ne => { value__ne => [qw/1 2 3 5/] } ],
]) eq 'ne', 'ne works for arrays');

ok(get_field_value_depends_on_rules({ value => 10.001 }, [
    [ ne => { value__ne => 10.002 } ],
]) eq 'ne', 'ne works for numbers');

ok(get_field_value_depends_on_rules({ value => 10.001 }, [
    [ eq => { value__eq => 10.001 } ],
]) eq 'eq', 'eq works for numbers');

ok(get_field_value_depends_on_rules({ value => undef }, [
    [ is_null => { value__is_null => 1 } ],
]) eq 'is_null', 'is_null works');

ok(!defined get_field_value_depends_on_rules({ value => "asdf" }, [
    [ is_null => { value__is_null => 1 } ],
]), 'is_null negative works');

ok(get_field_value_depends_on_rules({ value => "asdf" }, [
    [ is_not_null => { value__is_not_null => 1 } ],
]) eq 'is_not_null', 'is_not_null works');

ok(!defined get_field_value_depends_on_rules({ value => undef }, [
    [ is_not_null => { value__is_not_null => 1 } ],
]), 'is_not_null negative works');


matches_ok("", 'eq', "");
matches_ok("1", 'eq', "1");
matches_ok(0, 'ne', "");
matches_ok("", 'ne', 0);
matches_ok(10, 'le', 10);
matches_ok(10, 'ge', 10);

wont_match(20, 'le', 10);
wont_match(20, 'lt', 10);
wont_match("zz", 'le', "aa");
wont_match("aa", 'ge', "zz");
wont_match(10, 'ge', 20);
wont_match(10, 'gt', 20);

wont_match(10, 'eq', 11);
wont_match(10, 'ne', 10);
wont_match(10, 'ne', 10);

matches_ok("a", 'eq', [qw/b a/]); # any
matches_ok("a", 'ne', [qw/b c/]); # none

wont_match("a", 'eq', [qw/b c/]); # !any
wont_match("a", 'ne', [qw/b a c/]); # !none


done_testing();

1;
