#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use Test::More tests => 8;

sub prepare_test_data {
    my ( %params ) = @_;


    my $client_id = $params{client_id};
    my $spent     = $params{spent} // {};

    # units, used in past intervals
    my @spent_units;
    my $current_interval = int( time / 3600 ) * 3600;
    foreach my $idx ( keys %$spent ) {
        next unless $spent->{ $idx };
        my $interval = $current_interval - ( 23 - $idx ) * 3600;

        Units::Storage::Test->get_storage()->{ sprintf( "${client_id}-API-%d-spent" => $interval )   } = $spent->{ $idx };
    }
}

sub create_units {
    my ($ClientID, $limit) = @_;
    return Units->new({
        storage   => Units::Storage::Test->new(),
        key_format => "%d-%s-%d-spent",        
        client_id => $ClientID,
        limit => $limit
    });
}

use_ok('Units');

subtest 'there are withdraws by 1000 units in past 23 intervals' => sub {
    plan tests => 22;

    prepare_test_data( client_id => 1, spent => { map { $_ => 1000 } 0 .. 22 } );

    my $units = create_units(1, 24000);

    is( $units->client_id, 1, 'client_id' );
    is( $units->basket_name, 'API', 'basket_name' );
    is( $units->interval_size, 60 * 60, 'interval_size' );
    is( $units->intervals, 24, 'intervals' );
    is( $units->limit, 24000, 'limit is 24000 units' );
    is( $units->ttl, 24 * 60 * 60, 'ttl' );
    is( $units->current_interval, int( time / 3600 ) * 3600, 'current interval' );

    is( $units->spent, 23000, '23000 units spent' );
    is( $units->balance, 1000, '1000 units remain' );
    ok( ! $units->is_available( 2000 ), '2000 units unavailable' );
    ok( $units->is_available( 500 ), '500 units available' );
    ok( $units->withdraw( 500 ), 'withdraw 500 units' );
    is( $units->balance, 500, '500 units remain' );
    is( $units->spent, 23500, '23500 units spent' );
    ok( $units->withdraw( 250 ), 'withdraw 250 units' );
    is( $units->balance, 250, '250 units remain' );
    is( $units->spent, 23750, '23750 units spent' );
    ok( ! $units->is_available( 251 ), '251 units unavailable' );
    ok( $units->is_available( 249 ), '249 units available' );

    ok( $units->_restore_balance(), 'restore balance' );
    is( $units->spent, 0, '0 units spent' );
    is( $units->balance, 24000, '24000 units remain' );
};


subtest 'there are no withdraws in past 23 intervals' => sub {
    plan tests => 9;

    prepare_test_data(client_id => 2);

    my $units = create_units(2, 24000);

    is( $units->limit, 24000, 'limit is 24000 units' );
    is( $units->spent, 0, '0 units spent' );
    is( $units->balance, 24000, '24000 units remain' );
    ok( ! $units->is_available( 25000 ), '25000 units unavailable' );
    ok( $units->is_available( 24000 ), '24000 units available' );
    ok( $units->withdraw( 25000 ), 'withdraw 25000 units' );
    is( $units->balance, 0, '0 units remain' );
    is( $units->spent, 25000, '25000 units spent' );
    ok( ! $units->is_available( 1 ), '1 unit unavailable' );

    $units->_restore_balance();
};

subtest 'there are withdraw by 24k units in past 12 interval' => sub {
    plan tests => 9;

    prepare_test_data( client_id => 3, spent => { 11 => 24000 } ); # 'cause idx of intervals start from 0

    my $units = create_units(3, 24000);

    is( $units->limit, 24000, 'limit is 24000 units' );
    is( $units->spent, 24000, '24k units spent' );
    is( $units->balance, 12000, '12000 units remain' );
    ok( ! $units->is_available( 15000 ), '15000 units unavailable' );
    ok( $units->is_available( 12000 ), '12000 units available' );
    ok( $units->withdraw( 15000 ), 'withdraw 15000 units' );
    is( $units->balance, 0, '0 units remain' );
    is( $units->spent, 39000, '39000 units spent' );
    ok( ! $units->is_available( 1 ), '1 unit unavailable' );

    $units->_restore_balance();
};

subtest 'there are withdraw by 24k units in past 12 interval' => sub {
    plan tests => 9;

    prepare_test_data( client_id => 4, spent => { map { $_ => 1500 } 0 .. 22 } );

    my $units = create_units(4, 24000);

    is( $units->limit, 24000, 'limit is 24000 units' );
    is( $units->spent, 34500, '34.5k units spent' );
    is( $units->balance, 1000, '1k units remain' );
    ok( ! $units->is_available( 1500 ), '1.5k units unavailable' );
    ok( $units->is_available( 1000 ), '1k units available' );
    ok( $units->withdraw( 2000 ), 'withdraw 2k units' );
    is( $units->balance, 0, '0 units remain' );
    is( $units->spent, 36500, '36500 units spent' );
    ok( ! $units->is_available( 1 ), '1 unit unavailable' );

    $units->_restore_balance();
};

subtest 'there are withdraw by 24k units in past 12 interval' => sub {
    plan tests => 9;

    prepare_test_data( client_id => 5, spent => { ( map { $_ => 500 } 0 .. 11 ), ( 12 => 24000 ) } );

    my $units = create_units(5, 24000);

    is( $units->limit, 24000, 'limit is 24000 units' );
    is( $units->spent, 30000, '30k units spent' );
    is( $units->balance, 11000, '11k units remain' );
    ok( ! $units->is_available( 15000 ), '15k units unavailable' );
    ok( $units->is_available( 11000 ), '11k units available' );
    ok( $units->withdraw( 11000 ), 'withdraw 11k units' );
    is( $units->balance, 0, '0 units remain' );
    is( $units->spent, 41000, '41000 units spent' );
    ok( ! $units->is_available( 1 ), '1 unit unavailable' );

    $units->_restore_balance();
};

subtest 'fractional balance' => sub {
    plan tests => 9;

    prepare_test_data( client_id => 6, spent => { 22 => 31000 } ); # 0 .. 22

    my $units = create_units(6, 32000);

    is( $units->limit, 32000, 'limit is 32000 units' );
    is( $units->spent, 31000, '31000 units spent' );
    is( $units->balance, 1334, '1334 units remain' );
    ok( ! $units->is_available( 1335 ), '1335 units unavailable' );
    ok( $units->is_available( 1334 ), '1334 units available' );
    ok( $units->withdraw( 1334 ), 'withdraw 1334 units' );
    is( $units->balance, 0, '0 units remain' );
    is( $units->spent, 32334, '32334 units spent' );
    ok( ! $units->is_available( 1 ), '1 unit unavailable' );

    $units->_restore_balance();
};

subtest 'default limit if limit unspecified' => sub {
    my $units = Units->new({
        storage   => Units::Storage::Test->new(),
        key_format => "%d-%s-%d-spent",        
        client_id => 6,
    });

    is( $units->limit, $Units::DEFAULT_LIMIT, "limit $Units::DEFAULT_LIMIT is units");
};

package Units::Storage::Test;

use strict;
use warnings;
use utf8;

use base qw/Units::Storage/;

my $storage;

BEGIN {
    $storage = {};
}


sub get_storage {
    my ($class) = @_;
    return $storage;
}

sub new {
    my ($class) = @_;

    return bless {storage => $storage }, $class;
}

sub get {
    my ( $self, $key ) = @_;
    
    if (exists $self->{storage}{$key}) {
        return $storage->{$key};
    }

    return undef;
}

sub get_multi {
    my ( $self, $keys ) = @_;
    
    return { map { $_ => $self->get($_) } grep { exists $self->{storage}{$_} } @$keys  };
}

sub set {
    my ( $self, $key, $val, $expiration ) = @_;

    $self->{storage}{$key} = $val;
    return 1;
}

sub incr {
    my ( $self, $key, $val ) = @_;
    
    #if (exists $self->{storage}{$key}) {
    $self->{storage}{$key} += ($val // 1);
    return $self->{storage}{$key};
    #}

    return undef;
}

sub incr_expire {
    my ( $self, $key, $val, $expire ) = @_;
    return $self->incr($key, $val);
}

sub delete_multi {
    my ( $self, $keys ) = @_;

    return ( map { delete $self->{storage}{$_} } grep { exists $self->{storage}{$_} } @$keys  ) ? 1 : 0;
}

1;
