#!/usr/bin/perl
use Direct::Modern;
no warnings 'redefine';

use Carp 'confess';
use MIME::Base64 'decode_base64';
use Test::Deep qw( cmp_details deep_diag :v1 );
use Test::More tests => 11;

use my_inc '../../../..', for => 'unit_tests';

use API::Authorization;
use API::Authorization::User;
use API::Service::AdImages;
use Settings;

# внутри чёрный gif 1x1
my $IMAGE_DATA = decode_base64('R0lGODlhAQABAPAAAAAAAAAAACH5BAAAAAAALAAAAAABAAEAAAICRAEAOw');

# внутри md5_base64ya от пустой строки
my $IMAGE_HASH = '1B2M2Y8AsgTpgAmY7PhCfg';

my $VALID_IMAGE_REQUEST_ITEM = { Name => 'black.gif', ImageData => $IMAGE_DATA };

my $USER = API::Authorization::User->new(
    uid => 101,
    login => 'John',
    ClientID => 1,
);

my $SERVICE = API::Service::AdImages->new('AdImages');

*API::Service::Base::units_reset_or_withdraw = sub { return 1 };

$SERVICE->set_protocol('soap');
$SERVICE->set_subclient($USER);
$SERVICE->set_authorization(
    API::Authorization->fake_new(101, 'John', { chief_rep_user => $USER, operator_user => $USER, karma => 0 })
);
$SERVICE->set_current_operation('add');

sub override_functions {
    my %opts = @_;

    my $guard = OverrideGuard->new;

    if ( $opts{prevent_saving} ) {
        confess 'prevent_saving and simulate_saving_error are mutually exclusive'
            if exists $opts{simulate_saving_error};

        *API::Service::AdImages::banner_prepare_save_image =
        *BannerImages::Pool::add_items =
        *Direct::Model::ImageFormat::Manager::save =
            sub { confess q(this test isn't supposed to upload images) };
    } else {
        if ( $opts{simulate_saving_error} ) {
            *API::Service::AdImages::banner_prepare_save_image = sub {
                return { error => 'error saving image' };
            };
            *Direct::Model::ImageFormat::Manager::save = sub {
                return { $IMAGE_HASH => 'error saving image' };
            };
        } else {
            *API::Service::AdImages::banner_prepare_save_image = sub {
                return { md5 => $IMAGE_HASH };
            };
            *Direct::Model::ImageFormat::Manager::save = sub {
                my ($self) = @_;
                my $items = $self->items;
                for my $item (@$items) {
                    $item->width(728);
                    $item->height(90);
                    $item->mds_group_id(4517);
                }
                return {};
            };
        }

        *BannerImages::Pool::add_items = sub {
            my ($items) = @_;
            for my $item (@$items) {
                $item->{imp_id} = 10000;
            }
            return $items;
        };
    }

    *BannerImages::Pool::get_users_pool_limits = sub {
        return [ {
            ClientID => 1,
            total => $opts{upload_image_cap} // 10000,
            cnt => $opts{previously_uploaded_images} // 1,
        } ];
    };

    if ( $opts{simulate_invalid_format} ) {
        *Direct::Validation::Image::check_image = sub {
            return { error => 'image format is invalid' };
        };
    } else {
        if ( $opts{simulate_image_ad_image} ) {
            *Direct::Validation::Image::check_image = sub {
                return { size => 43, width => 728, height => 90, md5 => $IMAGE_HASH };
            };
            *Direct::Model::ImageFormat::BUILD = sub {
                my ($self) = @_;
                $self->image_type('image_ad');
            }
        } else {
            *Direct::Validation::Image::check_image = sub {
                return { size => 43, width => 1, height => 1, md5 => $IMAGE_HASH };
            };
            *Direct::Model::ImageFormat::BUILD = sub {
                my ($self) = @_;
                $self->image_type('regular');
            }
        }
    }

    if ( $opts{simulate_no_rights} ) {
        *API::Service::Base::can_write_client_objects = sub { 0 };
    } else {
        *API::Service::Base::can_write_client_objects = sub { 1 };
    }

    *BannerImages::banner_image_check_size = sub {
        return exists $opts{simulate_image_size} ? $opts{simulate_image_size} : 'large';
    };

    if ( exists $opts{add_limit} ) {
        $API::Service::AdImages::ADIMAGES_ADD_LIMIT = $opts{add_limit};
    }

    if ( exists $opts{max_image_file_size} ) {
        $BannerImages::MAX_IMAGE_FILE_SIZE = $opts{max_image_file_size};
    }

    *API::Service::AdImages::get_shard = sub ($$;$) { return 1 };

    return $guard;
}

sub request_error_matcher {
    my ( $code, $name, $suffix ) = @_;
    return all(
        obj_isa('Direct::Defect'),
        code( sub {
            my ($error) = @_;
            my ( $ok, $stack ) = cmp_details( { %$error }, {
                code => $code,
                description => ignore(),
                name => $name,
                text => ignore(),
                type => 'error',
                suffix => $suffix // '',
            } );

            return 1 if $ok;
            return ( 0, deep_diag($stack) );
        } ),
    );
}

sub item_error_matcher {
    my @codes = @_;

    ## когда ошибка одна, такое соответствие лучше, чем set, потому что, когда set не совпадает,
    ## Test::Deep не говорит, какие поля в структуре не совпали
    if ( @codes == 1 ) {
        return { Errors => [ { Code => $codes[0], Message => ignore(), Details => ignore() } ] };
    }

    return { Errors => set( map { { Code => $_, Message => ignore(), Details => ignore() } } @codes ) };
}

do {
    my $guard = override_functions();

    cmp_deeply(
        $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
        { AddResults => [ { AdImageHash => '1B2M2Y8AsgTpgAmY7PhCfg' } ] },
        'valid request',
    );
};

do {
    my $guard = override_functions( simulate_image_ad_image => 1 );

    cmp_deeply(
        $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
        { AddResults => [ { AdImageHash => '1B2M2Y8AsgTpgAmY7PhCfg' } ] },
        'valid request: image ad',
    );
};

do {
    my $guard = override_functions( simulate_no_rights => 1, prevent_saving => 1 );

    cmp_deeply(
        $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
        request_error_matcher( 54, 'NoRights', 'CantWrite' ),
        'no rights',
    );
};

do {
    my $guard = override_functions( add_limit => 1, prevent_saving => 1 );

    cmp_deeply(
        $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM, $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
        request_error_matcher( 9300, 'RequestLimitExceeded' ),
        'too many images',
    );
};

subtest 'image limit reached' => sub {
    plan tests => 2;

    do {
        my $guard = override_functions( previously_uploaded_images => 2, upload_image_cap => 2, prevent_saving => 1 );
        cmp_deeply(
            $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
            { AddResults => [ item_error_matcher(7001) ] },
            'adding 1 image on top of 2 previously uploaded with an overall limit of 2',
        );
    };

    do {
        my $guard = override_functions( previously_uploaded_images => 2, upload_image_cap => 3 );
        cmp_deeply(
            $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM, $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
            { AddResults => [ { AdImageHash => '1B2M2Y8AsgTpgAmY7PhCfg' }, item_error_matcher(7001) ] },
            'adding 2 images on top of 2 previously uploaded with an overall limit of 3',
        );
    };
};

do {
    my $guard = override_functions( prevent_saving => 1 );

    cmp_deeply(
        $SERVICE->add( { AdImages => [ { %$VALID_IMAGE_REQUEST_ITEM, Name => ( 'a' x 256 ) . '.gif' } ] } ) || undef,
        { AddResults => [ item_error_matcher(5001) ] },
        'name too long',
    );
};

do {
    my $guard = override_functions( simulate_invalid_format => 1, prevent_saving => 1 );

    cmp_deeply(
        $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
        { AddResults => [ item_error_matcher(5004) ] },
        'invalid format',
    );
};

do {
    my $guard = override_functions( max_image_file_size => 3, prevent_saving => 1 );

    cmp_deeply(
        $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
        { AddResults => [ item_error_matcher(5010) ] },
        'file length too long',
    );
};

subtest 'invalid image size' => sub {
    plan tests => 2;

    do {
        my $guard = override_functions( simulate_image_size => undef, prevent_saving => 1 );

        cmp_deeply(
            $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
            { AddResults => [ item_error_matcher(5004) ] },
            'undefined',
        );
    };

    do {
        my $guard = override_functions( simulate_image_size => 'small', prevent_saving => 1 );

        cmp_deeply(
            $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
            { AddResults => [ item_error_matcher(5004) ] },
            'small',
        );
    };

};

do {
    my $guard = override_functions( simulate_saving_error => 1 );

    cmp_deeply(
        $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
        { AddResults => [ item_error_matcher(1002) ] },
        'failed to save image',
    );
};

do {
    my $guard = override_functions( simulate_saving_error => 1, simulate_image_ad_image => 1 );

    cmp_deeply(
        $SERVICE->add( { AdImages => [ $VALID_IMAGE_REQUEST_ITEM ] } ) || undef,
        { AddResults => [ item_error_matcher(1002) ] },
        'failed to save image: image ad',
    );
};

exit 0;

package OverrideGuard;

sub new {
    my ($class) = @_;
    return bless {
        'API::Service::AdImages::banner_prepare_save_image' => *API::Service::AdImages::banner_prepare_save_image,
        'API::Service::AdImages::get_shard' => *API::Service::AdImages::get_shard,
        'API::Service::Base::can_write_client_objects' => *API::Service::Base::can_write_client_objects,
        'BannerImages::Pool::add_items' => *BannerImages::Pool::add_items,
        'BannerImages::Pool::get_users_pool_limits' => *BannerImages::Pool::get_users_pool_limits,
        'Direct::Validation::Image::check_image' => *Direct::Validation::Image::check_image,
        'BannerImages::banner_image_check_size' => *BannerImages::banner_image_check_size,
        'Direct::Model::ImageFormat::Manager::save' => *Direct::Model::ImageFormat::Manager::save,

        'API::Service::AdImages::ADIMAGES_ADD_LIMIT' => $API::Service::AdImages::ADIMAGES_ADD_LIMIT,
        'BannerImages::MAX_IMAGE_FILE_SIZE' => $BannerImages::MAX_IMAGE_FILE_SIZE,
    }, $class;
}

sub DESTROY {
    my ($self) = @_;

    *API::Service::AdImages::banner_prepare_save_image = $self->{'API::Service::AdImages::banner_prepare_save_image'};
    *API::Service::AdImages::get_shard = $self->{'API::Service::AdImages::get_shard'};
    *API::Service::Base::can_write_client_objects = $self->{'API::Service::Base::can_write_client_objects'};
    *BannerImages::Pool::add_items = $self->{'BannerImages::Pool::add_items'};
    *BannerImages::Pool::get_users_pool_limits = $self->{'BannerImages::Pool::get_users_pool_limits'};
    *Direct::Validation::Image::check_image = $self->{'Direct::Validation::Image::check_image'};
    *BannerImages::banner_image_check_size = $self->{'BannerImages::banner_image_check_size'};
    *Direct::Model::ImageFormat::Manager::save = $self->{'Direct::Model::ImageFormat::Manager::save'};

    $API::Service::AdImages::ADIMAGES_ADD_LIMIT = $self->{'API::Service::AdImages::ADIMAGES_ADD_LIMIT'};
    $BannerImages::MAX_IMAGE_FILE_SIZE = $self->{'BannerImages::MAX_IMAGE_FILE_SIZE'};
}
