#!/usr/bin/perl
use Direct::Modern;
no warnings 'redefine';

use Carp 'confess';
use List::MoreUtils 'uniq';
use Test::Deep qw( cmp_details deep_diag :v1 );
use Test::More tests => 8;

use my_inc '../../../..', for => 'unit_tests';

use API::Authorization;
use API::Authorization::User;
use API::Service::AdImages;
use Settings;

# внутри md5_base64ya от строки 'unassigned'
my $UNASSIGNED_IMAGE_HASH_1 = 'utCXSUWXpOCSwVJBAu6upw';

# внутри md5_base64ya от строки 'also unassigned'
my $UNASSIGNED_IMAGE_HASH_2 = '_hDo4x-I1EsjBFQIsXiiMg';

# внутри md5_base64ya от строки 'assigned'
my $ASSIGNED_IMAGE_HASH = 'XZj-mlLh3YoAN1GBtgsfqA';

# внутри md5_base64ya от пустой строки
my $BOGUS_IMAGE_HASH = '1B2M2Y8AsgTpgAmY7PhCfg';

my %SIMULATED_IMAGE_POOL_ITEMS = (
    $UNASSIGNED_IMAGE_HASH_1 => {
        image_hash => $UNASSIGNED_IMAGE_HASH_1,
    },
    $UNASSIGNED_IMAGE_HASH_2 => {
        image_hash => $UNASSIGNED_IMAGE_HASH_2,
    },
    $ASSIGNED_IMAGE_HASH => {
        image_hash => $ASSIGNED_IMAGE_HASH,
        assigned => 1,
    },
);

my $USER = API::Authorization::User->new(
    uid => 101,
    login => 'John',
    ClientID => 1,
);

my $SERVICE = API::Service::AdImages->new('AdImages');

*API::Service::Base::units_reset_or_withdraw = sub { return 1 };

$SERVICE->set_protocol('soap');
$SERVICE->set_subclient($USER);
$SERVICE->set_authorization(
    API::Authorization->fake_new(101, 'John', { chief_rep_user => $USER, operator_user => $USER, karma => 0 })
);
$SERVICE->set_current_operation('delete');

sub override_functions {
    my %opts = @_;

    my $guard = OverrideGuard->new;

    *BannerImages::Pool::get_items = sub {
        my ( $client_ids, $image_hashes ) = @_;
        my @result;
        for my $image_hash ( uniq @$image_hashes ) {
            if ( my $item = $SIMULATED_IMAGE_POOL_ITEMS{$image_hash} ) {
                push @result, $item;
            }
        }
        return \@result;
    };

    if ( $opts{prevent_delete} ) {
        *BannerImages::Pool::delete_items = sub { confess q(this test isn't supposed to delete images) };
    } else {
        *BannerImages::Pool::delete_items = sub {};
    }

    if ( $opts{simulate_no_rights} ) {
        *API::Service::Base::can_write_client_objects = sub { 0 };
    } else {
        *API::Service::Base::can_write_client_objects = sub { 1 };
    }

    if ( exists $opts{delete_limit} ) {
        $API::Service::AdImages::ADIMAGES_DELETE_LIMIT = $opts{delete_limit};
    }

    return $guard;
}

sub request_error_matcher {
    my ( $code, $name, $suffix ) = @_;
    return all(
        obj_isa('Direct::Defect'),
        code( sub {
            my ($error) = @_;
            my ( $ok, $stack ) = cmp_details( { %$error }, {
                code => $code,
                description => ignore(),
                name => $name,
                text => ignore(),
                type => 'error',
                suffix => $suffix // '',
            } );

            return 1 if $ok;
            return ( 0, deep_diag($stack) );
        } ),
    );
}

sub item_error_matcher {
    my @codes = @_;

    ## когда ошибка одна, такое соответствие лучше, чем set, потому что, когда set не совпадает,
    ## Test::Deep не говорит, какие поля в структуре не совпали
    if ( @codes == 1 ) {
        return { Errors => [ { Code => $codes[0], Message => ignore(), Details => ignore() } ] };
    }

    return { Errors => set( map { { Code => $_, Message => ignore(), Details => ignore() } } @codes ) };
}

sub call_delete {
    my @image_hashes = @_;
    return $SERVICE->delete( { SelectionCriteria => { AdImageHashes => \@image_hashes } } );
}

do {
    my $guard = override_functions();

    cmp_deeply(
        call_delete($UNASSIGNED_IMAGE_HASH_1) || undef,
        { DeleteResults => [ { AdImageHash => $UNASSIGNED_IMAGE_HASH_1 } ] },
        'valid request',
    );
};

do {
    my $guard = override_functions();

    cmp_deeply(
        call_delete("") || undef,
        { DeleteResults => [ item_error_matcher(4000) ] },
        'empty image hash',
    );
};

do {
    my $guard = override_functions();

    cmp_deeply(
        call_delete("foobar") || undef,
        { DeleteResults => [ item_error_matcher(8800) ] },
        'invalid image hash',
    );
};

do {
    my $guard = override_functions( simulate_no_rights => 1, prevent_delete => 1 );

    cmp_deeply(
        call_delete($UNASSIGNED_IMAGE_HASH_1) || undef,
        request_error_matcher( 54, 'NoRights', 'CantWrite' ),
        'no rights',
    );
};

do {
    my $guard = override_functions( delete_limit => 1, prevent_delete => 1 );

    cmp_deeply(
        call_delete( $UNASSIGNED_IMAGE_HASH_1, $UNASSIGNED_IMAGE_HASH_2 ) || undef,
        request_error_matcher( 4001, 'WrongSelectionCriteria', 'LimitExceeded' ),
        'too many images',
    );
};

subtest 'duplicate images' => sub {
    plan tests => 2;

    do {
        my $guard = override_functions( prevent_delete => 1 );

        cmp_deeply(
            call_delete( $UNASSIGNED_IMAGE_HASH_1, $UNASSIGNED_IMAGE_HASH_1 ) || undef,
            { DeleteResults => [ item_error_matcher(9800), item_error_matcher(9800) ] },
            'no images but duplicates',
        );
    };

    do {
        my $guard = override_functions();

        cmp_deeply(
            call_delete( $UNASSIGNED_IMAGE_HASH_1, $UNASSIGNED_IMAGE_HASH_2, $UNASSIGNED_IMAGE_HASH_1 ) || undef,
            {
                DeleteResults => [
                    item_error_matcher(9800),
                    { AdImageHash => $UNASSIGNED_IMAGE_HASH_2 },
                    item_error_matcher(9800),
                ],
            },
            'some duplicates, some non-duplicates',
        );
    };

};

do {
    my $guard = override_functions( prevent_delete => 1 );

    cmp_deeply(
        call_delete($BOGUS_IMAGE_HASH) || undef,
        { DeleteResults => [ item_error_matcher(8800) ] },
        'image not found',
    );
};

do {
    my $guard = override_functions( prevent_delete => 1 );

    cmp_deeply(
        call_delete($ASSIGNED_IMAGE_HASH) || undef,
        { DeleteResults => [ item_error_matcher(8301) ] },
        'removing assigned image',
    );
};

exit 0;

package OverrideGuard;

sub new {
    my ($class) = @_;
    return bless {
        'API::Service::Base::can_write_client_objects' => *API::Service::Base::can_write_client_objects,
        'BannerImages::Pool::get_items' => *BannerImages::Pool::get_items,
        'BannerImages::Pool::delete_items' => *BannerImages::Pool::delete_items,

        'API::Service::AdImages::ADIMAGES_DELETE_LIMIT' => $API::Service::AdImages::ADIMAGES_DELETE_LIMIT,
    }, $class;
}

sub DESTROY {
    my ($self) = @_;

    *API::Service::Base::can_write_client_objects = $self->{'API::Service::Base::can_write_client_objects'};
    *BannerImages::Pool::get_items = $self->{'BannerImages::Pool::get_items'};
    *BannerImages::Pool::delete_items = $self->{'BannerImages::Pool::delete_items'};

    $API::Service::AdImages::ADIMAGES_DELETE_LIMIT = $self->{'API::Service::AdImages::ADIMAGES_DELETE_LIMIT'};
}
