#!/usr/bin/env perl

use Direct::Modern;

use Test::More;

use Yandex::DBUnitTest qw/:all/;
use Yandex::I18n;
use JSON;

use my_inc '../../../..', for => 'api/t';
use Settings;

use API::Authorization::User;
use API::Authorization;
use API::Service::AdImages;


{ no warnings 'redefine'; *API::Service::Base::units_reset_or_withdraw = sub { return 1 } }

$Direct::Errors::Messages::NO_HASH_TAG = 1;

my $S = API::Service::AdImages->new('AdImages');

my $uid = 101;
my $login = 'John';

my $user = API::Authorization::User->new(
    uid => $uid,
    login => $login,
    ClientID => 1,
);
$S->set_protocol('soap');
$S->set_subclient($user);
$S->set_authorization(
    API::Authorization->fake_new($uid, $login, { chief_rep_user => $user, operator_user => $user, karma => 0 })
);
$S->set_current_operation('get');

init_test_dataset(db_data_init());

# 1, 2
my $request = {
    FieldNames => [qw/AdImageHash Name Associated Type Subtype OriginalUrl PreviewUrl/]
};

ok(my $response = $S->get($request), 'AdImages get operation call');
is_deeply($response, {
    'AdImages' => [
        {
            'AdImageHash' => 'adfdferrgtY-6u4Rye4hrt',
            'Associated' => 'YES',
            'Name' => 'Afrgrt.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/adfdferrgtY-6u4Rye4hrt',
            'PreviewUrl' => 'https://direct.yandex.ru/images/adfdferrgtY-6u4Rye4hrt/wx300',
            'Subtype' => 'NONE',
            'Type' => 'WIDE',
        },
        {
            'AdImageHash' => 'bfg4egtryjt6yrtt2h5fgj',
            'Associated' => 'NO',
            'Name' => 'Ber4wt.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/bfg4egtryjt6yrtt2h5fgj',
            'PreviewUrl' => 'https://direct.yandex.ru/images/bfg4egtryjt6yrtt2h5fgj/x90',
            'Subtype' => 'NONE',
            'Type' => 'SMALL',
        },
        {
            'AdImageHash' => 'csdfgdfbdfgewgtrrtet43',
            'Associated' => 'NO',
            'Name' => 'Cregre.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/csdfgdfbdfgewgtrrtet43',
            'PreviewUrl' => 'https://direct.yandex.ru/images/csdfgdfbdfgewgtrrtet43/x90',
            'Subtype' => 'NONE',
            'Type' => 'REGULAR',
        },
        {
            'AdImageHash' => 'ddggmyydgfwgedt34etwfd',
            'Associated' => 'NO',
            'Name' => 'Dewrt4.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/ddggmyydgfwgedt34etwfd',
            'PreviewUrl' => 'https://direct.yandex.ru/images/ddggmyydgfwgedt34etwfd/wx300',
            'Subtype' => 'NONE',
            'Type' => 'WIDE',
        },
        {
            'AdImageHash' => 'grgkjh43tr943toiu43tkl',
            'Associated' => 'NO',
            'Name' => 'Gefwwe.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/grgkjh43tr943toiu43tkl',
            'PreviewUrl' => 'https://direct.yandex.ru/images/grgkjh43tr943toiu43tkl/x90',
            'Subtype' => 'NONE',
            'Type' => 'SMALL',
        },
        {
            'AdImageHash' => '25h9l3pwbetnwr22plfuvz',
            'Associated' => 'NO',
            'Name' => '25h9l3.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/25h9l3pwbetnwr22plfuvz',
            'PreviewUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/25h9l3pwbetnwr22plfuvz',
            'Subtype' => 'IMG_240_400',
            'Type' => 'FIXED_IMAGE'
        },
        {
            'AdImageHash' => 'd8z3g2zcx1w9fg74nnwpos',
            'Associated' => 'YES',
            'Name' => 'd8z3g2.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/d8z3g2zcx1w9fg74nnwpos',
            'PreviewUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/d8z3g2zcx1w9fg74nnwpos',
            'Subtype' => 'IMG_300_500',
            'Type' => 'FIXED_IMAGE'
        },
        {
            'AdImageHash' => 'luvlw84sngp6k4eifrgitt',
            'Associated' => 'YES',
            'Name' => 'luvlw8.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/luvlw84sngp6k4eifrgitt',
            'PreviewUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/luvlw84sngp6k4eifrgitt',
            'Subtype' => 'IMG_336_280',
            'Type' => 'FIXED_IMAGE'
        },
        {
            'AdImageHash' => 'mbswh7ju6gdzdhhy26ap33',
            'Associated' => 'NO',
            'Name' => 'mbswh7.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/mbswh7ju6gdzdhhy26ap33',
            'PreviewUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/mbswh7ju6gdzdhhy26ap33',
            'Subtype' => 'IMG_640_200',
            'Type' => 'FIXED_IMAGE'
        },
        {
            'AdImageHash' => 'sa1mwbja2xuysrwymvww2u',
            'Associated' => 'NO',
            'Name' => 'sa1mwb.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/sa1mwbja2xuysrwymvww2u',
            'PreviewUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/sa1mwbja2xuysrwymvww2u',
            'Subtype' => 'IMG_728_90',
            'Type' => 'FIXED_IMAGE'
        },
        {
            'AdImageHash' => 'b4953td44ft6pgd5jpkhmv',
            'Associated' => 'NO',
            'Name' => 'b4953t.png',
            'OriginalUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/b4953td44ft6pgd5jpkhmv',
            'PreviewUrl' => 'https://direct.yandex.ru/images/direct-picture/4517/b4953td44ft6pgd5jpkhmv',
            'Subtype' => 'IMG_970_250',
            'Type' => 'FIXED_IMAGE'
        }
    ]
}, 'AdImages get response. All client\'s images. Full fields list');
# 3
$request = {
    SelectionCriteria => {
        AdImageHashes => ['edfgehfyhhtry4wdoi4wbj']
    },
    FieldNames => [qw/AdImageHash/]
};
$response = $S->get($request);
is_deeply($response, {
}, 'No images found by hash');
# 4
$request = {
    FieldNames => [qw/AdImageHash/],
    SelectionCriteria => {
        AdImageHashes => ['adfdferrgtY-6u4Rye4hrt']
    }
};
$response = $S->get($request);
is_deeply($response, {
    'AdImages' => [
        {
            AdImageHash => 'adfdferrgtY-6u4Rye4hrt',
        },
    ]
}, 'One image by hash');
# 5
$request = {
    FieldNames => [qw/AdImageHash/],
    SelectionCriteria => {
        Associated => 'NO'
    }
};
$response = $S->get($request);
is_deeply($response, {
    'AdImages' => [
        {
            'AdImageHash' => 'bfg4egtryjt6yrtt2h5fgj',
        },
        {
            'AdImageHash' => 'csdfgdfbdfgewgtrrtet43',
        },
        {
            'AdImageHash' => 'ddggmyydgfwgedt34etwfd',
        },
        {
            'AdImageHash' => 'grgkjh43tr943toiu43tkl',
        },
        {
            'AdImageHash' => '25h9l3pwbetnwr22plfuvz'
        },
        {
            'AdImageHash' => 'mbswh7ju6gdzdhhy26ap33'
        },
        {
            'AdImageHash' => 'sa1mwbja2xuysrwymvww2u'
        },
        {
            'AdImageHash' => 'b4953td44ft6pgd5jpkhmv'
        },
    ]
}, 'All disassociated images');
# 6
$request = {
    FieldNames => [qw/AdImageHash/],
    Page => {
        Limit => 1,
        Offset => 1
    }
};
$response = $S->get($request);
is_deeply($response, {
    'AdImages' => [
        {
            'AdImageHash' => 'bfg4egtryjt6yrtt2h5fgj',
        },
    ],
    LimitedBy => 2,
}, 'All images, limit 1, offset 1');
# 7
$request = {
    FieldNames => [qw/AdImageHash/],
    SelectionCriteria => {
        AdImageHashes => ['bfg4egtryjt6yrtt2h5fgj', 'fghw4fduigh4w4idfh4nfj']
    }
};
$response = $S->get($request);
is_deeply($response, {
    'AdImages' => [
        {
            'AdImageHash' => 'bfg4egtryjt6yrtt2h5fgj',
        },
    ],
}, 'Two images by hash, one out of scope');
# 8
$request = {
    FieldNames => [qw/AdImageHash Type Subtype/],
    SelectionCriteria => {
        AdImageHashes => [
            'adfdferrgtY-6u4Rye4hrt',
            'bfg4egtryjt6yrtt2h5fgj',
            'csdfgdfbdfgewgtrrtet43',
            '25h9l3pwbetnwr22plfuvz',
            'luvlw84sngp6k4eifrgitt'
        ],
        Associated => 'YES'
    }
};
$response = $S->get($request);
is_deeply($response, {
    'AdImages' => [
        {
            AdImageHash => 'adfdferrgtY-6u4Rye4hrt',
            Type => 'WIDE',
            Subtype => 'NONE'
        },
        {
            AdImageHash => 'luvlw84sngp6k4eifrgitt',
            Type => 'FIXED_IMAGE',
            Subtype => 'IMG_336_280'
        },
    ]
}, 'Associated images by hash, get type');
done_testing;

sub db_data_init {
    return {
        banner_images_formats => {
            original_db => PPC(shard => 'all'),
            like => 'banner_images_formats',
            rows => {
                1 => [
                    image_row(wide => 'adfdferrgtY-6u4Rye4hrt'),
                    image_row(small => 'bfg4egtryjt6yrtt2h5fgj'),
                    image_row(regular => 'csdfgdfbdfgewgtrrtet43'),
                    image_row(wide => 'ddggmyydgfwgedt34etwfd'),
                    image_row(small => 'edfgehfyhhtry4wdoi4wbj'),
                    image_row(regular => 'fghw4fduigh4w4idfh4nfj'),
                    image_row(small => 'grgkjh43tr943toiu43tkl'),
                    image_row(image_ad => '25h9l3pwbetnwr22plfuvz', 240, 400),
                    image_row(image_ad => 'bnrr4gxbwhtm7y8fkt8yd9', 300, 250),
                    image_row(image_ad => 'd8z3g2zcx1w9fg74nnwpos', 300, 500),
                    image_row(image_ad => 'tj2yt4cixxgzx12izlr45j', 300, 600),
                    image_row(image_ad => 'luvlw84sngp6k4eifrgitt', 336, 280),
                    image_row(image_ad => 'ju8dv6re44frx1eqxx86cm', 640, 100),
                    image_row(image_ad => 'mbswh7ju6gdzdhhy26ap33', 640, 200),
                    image_row(image_ad => '67hd3rtf4j1y5c9zz3ugcn', 640, 960),
                    image_row(image_ad => 'sa1mwbja2xuysrwymvww2u', 728, 90),
                    image_row(image_ad => 'lwgzq8ep8f87pk87xmsuzq', 960, 640),
                    image_row(image_ad => 'b4953td44ft6pgd5jpkhmv', 970, 250),
                ]
            }
        },
        banner_images_pool => {
            original_db => PPC(shard => 'all'),
            like => 'banner_images_pool',
            rows => {
                1 => [
                    {image_hash => 'adfdferrgtY-6u4Rye4hrt', ClientID => 1, imp_id => 1,  name => 'Afrgrt.png'},
                    {image_hash => 'bfg4egtryjt6yrtt2h5fgj', ClientID => 1, imp_id => 2,  name => 'Ber4wt.png'},
                    {image_hash => 'csdfgdfbdfgewgtrrtet43', ClientID => 1, imp_id => 3,  name => 'Cregre.png'},
                    {image_hash => 'ddggmyydgfwgedt34etwfd', ClientID => 1, imp_id => 4,  name => 'Dewrt4.png'},
                    {image_hash => 'edfgehfyhhtry4wdoi4wbj', ClientID => 2, imp_id => 5,  name => 'Edfeed.png'},
                    {image_hash => 'fghw4fduigh4w4idfh4nfj', ClientID => 2, imp_id => 6,  name => 'Fefefe.png'},
                    {image_hash => 'grgkjh43tr943toiu43tkl', ClientID => 1, imp_id => 7,  name => 'Gefwwe.png'},
                    {image_hash => 'adfdferrgtY-6u4Rye4hrt', ClientID => 2, imp_id => 8,  name => 'Hdgfrr.png'},
                    {image_hash => '25h9l3pwbetnwr22plfuvz', ClientID => 1, imp_id => 9,  name => '25h9l3.png'},
                    {image_hash => 'bnrr4gxbwhtm7y8fkt8yd9', ClientID => 2, imp_id => 10, name => 'bnrr4g.png'},
                    {image_hash => 'd8z3g2zcx1w9fg74nnwpos', ClientID => 1, imp_id => 11, name => 'd8z3g2.png'},
                    {image_hash => 'tj2yt4cixxgzx12izlr45j', ClientID => 2, imp_id => 12, name => 'tj2yt4.png'},
                    {image_hash => 'luvlw84sngp6k4eifrgitt', ClientID => 1, imp_id => 13, name => 'luvlw8.png'},
                    {image_hash => 'ju8dv6re44frx1eqxx86cm', ClientID => 2, imp_id => 14, name => 'ju8dv6.png'},
                    {image_hash => 'mbswh7ju6gdzdhhy26ap33', ClientID => 1, imp_id => 15, name => 'mbswh7.png'},
                    {image_hash => '67hd3rtf4j1y5c9zz3ugcn', ClientID => 2, imp_id => 16, name => '67hd3r.png'},
                    {image_hash => 'sa1mwbja2xuysrwymvww2u', ClientID => 1, imp_id => 17, name => 'sa1mwb.png'},
                    {image_hash => 'lwgzq8ep8f87pk87xmsuzq', ClientID => 2, imp_id => 18, name => 'lwgzq8.png'},
                    {image_hash => 'b4953td44ft6pgd5jpkhmv', ClientID => 1, imp_id => 19, name => 'b4953t.png'},
                ]
            },
        },
        banner_images => {
            original_db => PPC(shard => 'all'),
            like => 'banner_images',
            rows => {
                1 => [
                    {image_hash => 'adfdferrgtY-6u4Rye4hrt', bid => 1, image_id => 149604769},
                    {image_hash => 'edfgehfyhhtry4wdoi4wbj', bid => 5, image_id => 149604765},
                    {image_hash => 'fghw4fduigh4w4idfh4nfj', bid => 6, image_id => 149604764},
                    {image_hash => 'adfdferrgtY-6u4Rye4hrt', bid => 8, image_id => 149604762},
                ]
            },
        },
        banners => {
            original_db => PPC(shard => 'all'),
            rows => [
                {
                    bid => 1,
                    cid => 263,
                },
                {
                    bid => 5,
                    cid => 263,
                },
                {
                    bid => 6,
                    cid => 263,
                },
                {
                    bid => 8,
                    cid => 100500,
                },
                {
                    bid => 11,
                    cid => 100011,
                },
                {
                    bid => 13,
                    cid => 100013,
                },
                {
                    bid => 16,
                    cid => 100016,
                },
                {
                    bid => 19,
                    cid => 100019,
                },
            ],
        },
        'campaigns' => {
            'original_db' => PPC(shard => 'all'),
            'rows' => {
                1 => [
                    {
                        cid => 263,
                        uid => 101,
                        statusempty => 'No',
                    },
                    {
                        cid => 100500,
                        uid => 201,
                        statusempty => 'No',
                    },
                    {
                        cid => 100011,
                        uid => 201,
                        statusempty => 'No',
                    },
                    {
                        cid => 100013,
                        uid => 101,
                        statusempty => 'No',
                    },
                    {
                        cid => 100016,
                        uid => 201,
                        statusempty => 'No',
                    },
                    {
                        cid => 100019,
                        uid => 101,
                        statusempty => 'No',
                    },
                ]
            }
        },
        images => {
            original_db => PPC(shard => 'all'),
            rows => {
                1 => [
                    {image_hash => 'bnrr4gxbwhtm7y8fkt8yd9', bid => 11, image_id => 159604769},
                    {image_hash => 'd8z3g2zcx1w9fg74nnwpos', bid => 13, image_id => 159604765},
                    {image_hash => 'tj2yt4cixxgzx12izlr45j', bid => 16, image_id => 159604764},
                    {image_hash => 'luvlw84sngp6k4eifrgitt', bid => 19, image_id => 159604762},
                ],
            },
        },
        'shard_inc_cid' => {
            'original_db' => PPCDICT(),
            'rows' => [
                {
                  'cid' => 263,
                  'clientid' => 1
                },
                {
                  'cid' => 1499,
                  'clientid' => 1
                },
                {
                  'cid' => 100500,
                  'clientid' => 2
                },
            ],
        },
        shard_client_id => {
            original_db => PPCDICT,
            rows => [
                {ClientID => 1, shard => 1},
                {ClientID => 2, shard => 1},
            ],
        },
        shard_uid => {
            original_db => PPCDICT,
            rows => [
                {uid => 101, ClientID => 1},
                {uid => 102, ClientID => 2},
            ],
        },
        ppc_properties => {
            original_db => PPCDICT,
        },
        users => {
            original_db => PPC(shard => 'all'),
            like => 'users',
            rows => {
                1 => [
                    {uid => 101, ClientID => 1},
                    {uid => 102, ClientID => 2},
                ]
            },
        },
        api_special_user_options => {
            original_db => PPC(shard => 'all'),
            like => 'api_special_user_options',
            rows => {
                1 => []
            },
        },
        users_options => {
            original_db => PPC(shard => 'all'),
            like => 'users_options',
            rows => {
                1 => []
            },
        },
        users_api_options => {
            original_db => PPC(shard => 'all'),
            like => 'users_api_options',
            rows => {
                1 => []
            },
        },
        clients_api_options => {
            original_db => PPC( shard => 'all' ),
        },
        banner_logos => {
            original_db => PPC( shard => 'all' ),
        },
    };
};

sub formats {
    my ($type, $width, $height) = @_;
    return to_json({
        small => {
           'x80' => {
                      'height' => 86,
                      'width' => 80
                    },
           'x90' => {
                      'height' => 97,
                      'width' => 90
                    },
           'y110' => {
                       'height' => 110,
                       'width' => 102
                     },
           'y129' => {
                       'height' => 129,
                       'width' => 120
                     },
           'y150' => {
                       'height' => 150,
                       'width' => 140
                     },
           'y65' => {
                      'height' => 65,
                      'width' => 61
                    },
           'y80' => {
                      'height' => 80,
                      'width' => 75
                    },
           'y90' => {
                      'height' => 90,
                      'width' => 84
                    }
         },
         regular => {
           'x160' => {
                       'height' => 213,
                       'width' => 160
                     },
           'x180' => {
                       'height' => 240,
                       'width' => 180
                     },
           'x450' => {
                       'height' => 600,
                       'width' => 450
                     },
           'x80' => {
                      'height' => 107,
                      'width' => 80
                    },
           'x90' => {
                      'height' => 120,
                      'width' => 90
                    },
           'y110' => {
                       'height' => 110,
                       'width' => 83
                     },
           'y129' => {
                       'height' => 129,
                       'width' => 97
                     },
           'y150' => {
                       'height' => 150,
                       'width' => 113
                     },
           'y160' => {
                       'height' => 160,
                       'width' => 120
                     },
           'y180' => {
                       'height' => 180,
                       'width' => 135
                     },
           'y300' => {
                       'height' => 300,
                       'width' => 225
                     },
           'y450' => {
                       'height' => 450,
                       'width' => 338
                     },
           'y65' => {
                      'height' => 65,
                      'width' => 49
                    },
           'y80' => {
                      'height' => 80,
                      'width' => 60
                    },
           'y90' => {
                      'height' => 90,
                      'width' => 68
                    }
        },
        wide => {
           'wx1080' => {
                         'height' => 607,
                         'width' => 1080
                       },
           'wx300' => {
                        'height' => 169,
                        'width' => 300
                      },
           'wx600' => {
                        'height' => 337,
                        'width' => 600
                      },
           'wy150' => {
                        'height' => 150,
                        'width' => 267
                      },
           'wy300' => {
                        'height' => 300,
                        'width' => 533
                      }
        },
        image_ad => {
           'orig' => {
                       'height' => $height,
                       'width' => $width
                     }
        }
    }->{$_[0]});
}

sub image_row {
    my ($type, $hash, $width, $height) = @_;
    return {
        image_hash => $hash,
        image_type => $type,
        formats => formats($type, $width, $height),
        ( $type eq 'image_ad' ? ( mds_group_id => 4517, namespace => 'direct-picture' ) : (namespace => 'direct') ),
    };
}
