#!/usr/bin/perl

use Direct::Modern;

use Test::More;# tests => 10;

use API::Service::Campaigns;
use API::Service::ResultSet::Item;

{
    no strict 'refs';
    no warnings 'redefine';
    *API::Service::Base::protocol = sub { 'json' };
}

tt_ok({}, "no errors for empty TimeTargeting");

subtest 'Schedule' => sub {

    ok(my $item = item(), 'item ok');
    check_errors({
        Schedule => {
            Items => []
        }
    }, [
    ], 'empty Schedule items');

    check_errors({
        Schedule => {
            Items => [
"1,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"2,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"3,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"4,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"5,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"6,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"7,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
    ], 'valid Schedule Items full week');

    check_errors({
        Schedule => {
            Items => [
"1,0,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"2,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
    ], 'valid Schedule Items 2 days');

    check_errors({
        Schedule => {
            Items => [
"1,200,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"2,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
    ], 'valid Schedule Items 2 days, 200 coef');

    check_errors({
        Schedule => {
            Items => [
"8,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100"
            ]
        }
    }, [
        [InvalidField => 'Временной таргетинг должен содержать номера дней недели от 1 до 7']
    ], 'wrong 8th day in schedule fails');

    check_errors({
        Schedule => {
            Items => [
"1,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"2,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"0,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
        [InvalidField => 'Временной таргетинг должен содержать номера дней недели от 1 до 7']
    ], 'wrong null day in schedule fails');

    check_errors({
        Schedule => {
            Items => [
"a,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
        [InvalidField => 'Временной таргетинг должен содержать номера дней недели от 1 до 7']
    ], 'wrong "a" day in schedule fails');

    check_errors({
        Schedule => {
            Items => [
"-1,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
        [InvalidField => 'Временной таргетинг должен содержать номера дней недели от 1 до 7']
    ], 'wrong "-1" day in schedule fails');

    check_errors({
        Schedule => {
            Items => [
"1,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"2,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"3,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"4,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"5,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"6,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"7,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"7,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
        [DuplicatedArrayItem => 'Временной таргетинг содержит повторные дни недели']
    ], 'week days has dups fails');

    check_errors({
        Schedule => {
            Items => [
"1,210,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"2,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
        [InvalidField => 'Коэффициенты ставок временного таргетинга должны содержать значения от 0 до 200 включительно' ]
    ], '210 coef fails');

    check_errors({
        Schedule => {
            Items => [
"1,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100",
"1,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100"
            ]
        }
    }, [
        [DuplicatedArrayItem => 'Временной таргетинг содержит повторные дни недели']
    ], 'week days has dups fails');

    check_errors({
        Schedule => {
            Items => [
"1,100,100,92,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100",
            ]
        }
    }, [
        [InvalidField => 'Временной таргетинг должен содержать коэффициенты ставок кратные 10']
    ], '91 coef fails as 91 mod 10 != 0');

    check_errors({
        Schedule => {
            Items => [
"1,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"2,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,0,0,0,0,0,0,0,0,0",
"3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0",
"4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0",
"5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0",
"6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0",
"7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0"
            ]
        }
    }, [
        [InvalidField => 'Временной таргетинг должен содержать коэффициенты ставок для каждого из 24 часов']
    ], '23 coefs fails');

    check_errors({
        Schedule => {
            Items => [
"1,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"2,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
        [InvalidField => 'Временной таргетинг должен содержать коэффициенты ставок для каждого из 24 часов']
    ], '23 coefs, short second day fails');

    check_errors({
        Schedule => {
            Items => [
"1,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50",
"2,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50"
            ]
        }
    }, [
        [InvalidField => 'Временной таргетинг должен содержать коэффициенты ставок для каждого из 24 часов']
    ], '23 coefs, short first day fails');

    check_coef('asdf', 'string coef fails');
    check_coef(99.9, 'float coef fails');
    check_errors({
        Schedule => {
            Items => [
"1,-1,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100",
            ]
        }
    }, [
        [ InvalidField => 'Временной таргетинг не может содержать отрицательные коэффициенты ставок' ]
    ], 'negative coef fails');

};

subtest 'HolidaysSchedule' => sub {
    tt_ok({ TimeTargeting => {
        HolidaysSchedule => {
            SuspendOnHolidays => "YES",
        }
    }},"HolidaysSchedule with suspend on holidays = YES");

    tt_ok({ TimeTargeting => {
        HolidaysSchedule => {
            StartHour => 1,
            EndHour => 24
        }
    }},"HolidaysSchedule with full hours range schedule");

    tt_ok({ TimeTargeting => {
        HolidaysSchedule => {
            SuspendOnHolidays => "NO",
            StartHour => 1,
            EndHour => 24
        }
    }},"HolidaysSchedule with full hours range schedule with SuspendOnHolidays =  'NO'");

    check_errors({
        HolidaysSchedule => {
            SuspendOnHolidays => "NO",
        }
    }, [
        [ ReqField => 'StartHour и EndHour должны быть заданы, если не задано SuspendOnHolidays = "YES"' ],
    ], 'SuspendOnHolidays = NO and no range');

    check_errors({
        HolidaysSchedule => {
            StartHour => -1,
            EndHour => 0,
        }
    }, [
        [ InvalidField => 'EndHour должен содержать значения от 1 до 24 включительно' ],
        [ InvalidField => 'StartHour должен содержать значения от 0 до 23 включительно' ],
    ], 'EndHour = 0, StartHour -1 fails');

    check_errors({
        HolidaysSchedule => {
            StartHour => 24,
            EndHour => 25
        }
    }, [
        [ InvalidField => 'EndHour должен содержать значения от 1 до 24 включительно' ],
        [ InvalidField => 'StartHour должен содержать значения от 0 до 23 включительно' ],
    ], 'EndHour = 25, StartHour 24 fails');


    check_errors({
        HolidaysSchedule => {
            SuspendOnHolidays => "YES",
            StartHour => 1,
            EndHour => 24
        }
    }, [
        [ BadParams => 'HolidaysSchedule не может содержать StartHour при SuspendOnHolidays в значении YES' ],
        [ BadParams => 'HolidaysSchedule не может содержать EndHour при SuspendOnHolidays в значении YES' ]
    ], 'SuspendOnHolidays and hours range specifies fails');

    check_errors({
        HolidaysSchedule => {
            SuspendOnHolidays => "YES",
            StartHour => 1,
        }
    }, [
        [ BadParams => 'HolidaysSchedule не может содержать StartHour при SuspendOnHolidays в значении YES' ],
    ], 'SuspendOnHolidays and StartHour range specifies fails');

    check_errors({
        HolidaysSchedule => {
            SuspendOnHolidays => "YES",
            EndHour => 24
        }
    }, [
        [ BadParams => 'HolidaysSchedule не может содержать EndHour при SuspendOnHolidays в значении YES' ]
    ], 'SuspendOnHolidays and hours range specifies fails');

};

my $called;
{
    no strict 'refs';
    no warnings 'redefine';
    *API::Service::Campaigns::_validate_hour_range = sub {
        $called = 1;
    };
}

validate_tt(item({
    TimeTargeting => {
        HolidaysSchedule => {
            StartHour => 1,
            EndHour => 24
        }
}}));

ok($called, "_validate_hours_range was called");

done_testing();

sub tt_ok {
    my ($req, $msg) = @_;
    my $item = item($req);
    validate_tt($item);
    ok(!$item->has_errors, $msg);
}
sub check_coef {
    my ($coef, $msg) = @_;
    check_errors({
        Schedule => {
            Items => [
"1,100,100,100,$coef,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100",
            ]
        }
    }, [
        [ InvalidField => 'Временной таргетинг содержит не целочисленные коэффициенты ставок' ]
    ], $msg);
}

sub validate_tt { API::Service::Campaigns->_validate_time_targeting(shift) }

sub check_errors {
    my ($request, $errors, $message) = @_; # { %request }, [ $error_details_text_re ], $test_ok_message
    my $item = item({ TimeTargeting => $request });
    validate_tt($item);
    my @failed;
    foreach my $e (@$errors) {
        push @failed, $e unless has_error($item, $e);
    }
    if(@failed) {
        warn "Got errors:\n" . join("\n", list_errors($item));
        warn "Missed errors:\n" . join("\n",  map { ref $_ ? qq~$_->[0]: $_->[1]~ : $_ } @failed);
        fail($message);
    } else {
        pass( $message );
    }
}

sub has_error {
    my ($item, $error) = @_;
    foreach my $e ($item->list_errors) {
        if (ref $error) {
            return 1 if $error->[0] eq $e->name && $e->description =~ /$error->[1]/;
        }
        return 1 if $e->description =~ /$error/;
    }
    return 0;
}

sub item {
    my $data = {};
    return API::Service::ResultSet::Item->new(0, $_[0]);
}

sub list_errors {
    map { $_->name . ': ' . $_->description } shift->list_errors
}

