#!/usr/bin/perl

use Direct::Modern;

use Test::More tests => 9;

#use my_inc '../../../../', for => 'api/t';

use API::Authorization;
use API::Authorization::User;
use API::Service::Campaigns;
use Mediaplan qw//;
use Settings;
use Units::TestDummy;
use Yandex::DBUnitTest qw/ :all /;
use Yandex::Test::UTF8Builder;

{
    no strict qw/ refs /;
    no warnings qw/ once redefine /;

    *API::Authorization::User::get_info_by_uid_passport = sub ($)   { +{} };
    *API::Authorization::User::create_update_user       = sub ($;$) { };

    *API::Service::Base::rbac                         = sub { bless {} => 'RBAC2::Extended' };
    *API::Service::Base::operator_role                = sub { 'client' };
    *API::Service::Base::subclient_role               = sub { 'client' };
    *API::Service::Base::not_owned_campaign_ids_hash  = sub { () };
    *API::Service::Base::check_campaigns_write_detail = sub { () };
    *API::Service::Base::units = sub { state $dummy_units = Units::TestDummy->new() };

    *API::Service::Base::send_alert                   = sub {};
    *Mediaplan::close_request_first_aid               = sub {};

    *Yandex::Log::new                                 = sub { return bless {}, $_[0] };
    *Yandex::Log::out                                 = sub {};
}

my $TEST_DATA = {
    api_special_user_options => {
        original_db => PPC( shard => 'all' ),
    },
    banners => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { bid => 1, pid => 1, statusActive => 'Yes', LastChange => '2015-08-03 13:00:00', },
                { bid => 2, pid => 2, statusActive => 'Yes', LastChange => '2015-08-03 13:00:00', },
            ],
        },
    },
    bids => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { id => 1, cid => 3, pid => 1, },
                { id => 2, cid => 4, pid => 2, },
            ],
        },
    },
    bids_arc => {
        original_db => PPC( shard => 'all' ),
    },
    campaigns => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { cid => 1,   uid => 1, type => 'wallet', statusEmpty => 'No', },
                { cid => 2,   uid => 2, type => 'geo',    statusEmpty => 'No', },
                { cid => 3,   uid => 1, type => 'text',   statusEmpty => 'No', archived => 'No', wallet_cid => 0,   OrderID => 0, currency => "YND_FIXED", sum => 5,  sum_spent => 5,  statusShow => 'No', lastShowTime => '0000-00-00 00:00:00', statusBsSynced => 'Yes', start_time => '2015-08-02', LastChange => '2015-08-03 13:00:00', },
                { cid => 4,   uid => 1, type => 'text',   statusEmpty => 'No', archived => 'No', wallet_cid => 104, OrderID => 0, currency => "YND_FIXED", sum => 5,  sum_spent => 10, statusShow => 'No', lastShowTime => '0000-00-00 00:00:00', statusBsSynced => 'Yes', start_time => '2015-08-02', LastChange => '2015-08-03 13:00:00', },
                { cid => 104, uid => 1, type => 'wallet', statusEmpty => 'No', archived => 'No', wallet_cid => 0,   OrderID => 0, currency => "YND_FIXED", sum => 50, sum_spent => 5,  },
                { cid => 200, uid => 1, type => 'mcbanner', statusEmpty => 'No', },
                { cid => 201, uid => 3, type => 'text', statusEmpty => 'No', },
            ],
        },
    },
    campaigns_performance => {
        original_db => PPC( shard => 'all' ),
        rows => {},
    },
    camp_operations_queue => {
        original_db => PPC( shard => 'all' ),
    },
    camp_options => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { cid => 3, stopTime => '0000-00-00 00:00:00', },
                { cid => 4, stopTime => '0000-00-00 00:00:00', }
            ],
        },
    },
    subcampaigns => {
        original_db => PPC(shard => 'all'),
    },
    clients_api_options => {
        original_db => PPC( shard => 'all' ),
    },
    clients_options => {
        original_db => PPC( shard => 'all' ),
    },
    phrases => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { pid => 1, cid => 3, LastChange => '2015-08-03 13:00:00', statusShowsForecast => 'New', },
                { pid => 2, cid => 4, LastChange => '2015-08-03 13:00:00', statusShowsForecast => 'New', },
            ],
        },
    },
    aggr_statuses_campaigns => {
        original_db => PPC(shard => 'all'),
        rows => [],
    },
    ppc_properties => {
        original_db => PPCDICT(),
        rows => [
        ],
    },
    shard_inc_bid => {
        original_db => PPCDICT(),
        rows => [
            { bid => 1, ClientID => 1, },
            { bid => 2, ClientID => 2, },
        ],
    },
    shard_client_id => {
        original_db => PPCDICT(),
        rows => [
            { ClientID => 1, shard => 1, },
            { ClientID => 2, shard => 1, },
        ],
    },
    shard_inc_cid => {
        original_db => PPCDICT(),
        rows => [
            { cid => 1,   ClientID => 1, },
            { cid => 2,   ClientID => 2, },
            { cid => 3,   ClientID => 1, },
            { cid => 4,   ClientID => 1, },
            { cid => 104, ClientID => 1, },
        ],
    },
    shard_inc_pid => {
        original_db => PPCDICT(),
        rows => [
            { pid => 1, ClientID => 1, },
            { pid => 2, ClientID => 2, },
        ],
    },
    shard_uid => {
        original_db => PPCDICT(),
        rows => [
            { uid => 1, ClientID => 1, },
            { uid => 2, ClientID => 2, },
        ],
    },
    users => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { uid => 1, login => 'unit-tester',                  statusBlocked => 'no', ClientID => 1, },
                { uid => 2, login => 'not-allowed-to-edit-geo-camp', statusBlocked => 'no', ClientID => 2, },
            ],
        },
    },
    users_api_options => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { uid => 1, api_geo_allowed => 'Yes', },
                { uid => 2, api_geo_allowed => 'No',  },
            ],
        },
    },
    users_options => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { uid => 1, passport_karma => 0, },
                { uid => 2, passport_karma => 0, },
            ],
        },
    },
};

init_test_dataset( $TEST_DATA );


my $login = 'unit-tester';
my $uid   = 1;
my $LIMIT = $API::Service::Campaigns::ARCHIVE_IDS_LIMIT;

my $user = API::Authorization::User->new(
    uid      => $uid,
    login    => $login,
    ClientID => 1,
);

my $service = API::Service::Campaigns->new('Campaigns');
$service->set_current_operation('archive');
$service->set_subclient( $user );
$service->set_authorization( API::Authorization->fake_new( $uid, $login, { chief_rep_user => $user } ) );


my %RESPONSES = (
    invalid_id => {
        ArchiveResults => [
            {
                Errors => [
                    {
                        Code    => 5005,
                        Message => 'Поле задано неверно',
                        Details => 'Значение поля Ids должно быть целым положительным числом',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 5005,
                        Message => 'Поле задано неверно',
                        Details => 'Значение поля Ids должно быть целым положительным числом',
                    },
                ],
            },
        ],
    },
    not_owned_campaign => {
        ArchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            }
        ],
    },
    not_exists_camps => {
        ArchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    readonly_campaign => {
        ArchiveResults => [
            {
                Errors => [
                    {
                        Code    => 54,
                        Message => 'Нет прав',
                        Details => 'Нет прав на запись',
                    },
                ],
            }
        ],
    },
    bad_kind_campaign => {
        ArchiveResults => [
            {
                Errors => [
                    {
                        Code    => 3500,
                        Message => 'Не поддерживается',
                        Details => 'Тип кампании не поддерживается',
                    },
                ],
            },
        ],
    },
    bad_operator => {
        ArchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    plain_text_camp => {
        ArchiveResults => [
            { Id => 3, },
        ],
    },
    duplicates_and_already_archived => {
        ArchiveResults => [
            {
                Id       => 3,
                Warnings => [
                    {
                        Code    => 10000,
                        Message => 'Объект присутствует в запросе более одного раза',
                        Details => 'Id кампании присутствует в запросе более одного раза',
                    },
                    {
                        Code    => 10022,
                        Message => 'Объект уже заархивирован',
                        Details => 'Кампания уже заархивирована',
                    },
                ],
            },
            {
                Id       => 3,
                Warnings => [
                    {
                        Code    => 10000,
                        Message => 'Объект присутствует в запросе более одного раза',
                        Details => 'Id кампании присутствует в запросе более одного раза',
                    },
                    {
                        Code    => 10022,
                        Message => 'Объект уже заархивирован',
                        Details => 'Кампания уже заархивирована',
                    },
                ],
            },
        ],
    },
    text_camp_under_wallet => {
        ArchiveResults => [
            { Id => 4, },
        ],
    }
);

my %SAVED_DATA = (
    plain_text_camp => {
        banners => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { bid => 1, pid => 1, statusActive => 'No', },
                    { bid => 2, pid => 2, statusActive => 'Yes', },
                ],
            },
        },
        bids => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { id => 2, cid => 4, pid => 2, },
                ],
            },
        },
        bids_arc => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { id => 1, cid => 3, pid => 1, },
                ],
            },
        },
        campaigns => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { cid => 1, },
                    { cid => 2, },
                    { cid => 3,   archived => 'Yes', statusShow => 'No', statusBsSynced => 'No',  start_time => '2015-08-02', },
                    { cid => 4,   archived => 'No',  statusShow => 'No', statusBsSynced => 'Yes', start_time => '2015-08-02', },
                    { cid => 104, },
                    { cid => 200, },
                    { cid => 201, },
                ],
            },
        },
        phrases => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { pid => 1, cid => 3, LastChange => '2015-08-03 13:00:00', statusShowsForecast => 'Archived', },
                    { pid => 2, cid => 4, LastChange => '2015-08-03 13:00:00', statusShowsForecast => 'New', },
                ],
            },
        },
    },
    text_camp_under_wallet => {
        banners => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { bid => 1, pid => 1, statusActive => 'No',  },
                    { bid => 2, pid => 2, statusActive => 'No', },
                ],
            },
        },
        bids => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                ],
            },
        },
        bids_arc => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { id => 2, cid => 4, pid => 2, },
                    { id => 1, cid => 3, pid => 1, },
                ],
            },
        },
        campaigns => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { cid => 1, },
                    { cid => 2, },
                    { cid => 3, archived => 'Yes', statusShow => 'No', statusBsSynced => 'No',  start_time => '2015-08-02', },
                    { cid => 4, archived => 'Yes',  statusShow => 'No', statusBsSynced => 'No', start_time => '2015-08-02', },
                    { cid => 104, },
                    { cid => 200, },
                    { cid => 201, },
                ],
            },
        },
        camp_operations_queue => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                ],
            },
        },
        phrases => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { pid => 1, cid => 3, LastChange => '2015-08-03 13:00:00', statusShowsForecast => 'Archived', },
                    { pid => 2, cid => 4, LastChange => '2015-08-03 13:00:00', statusShowsForecast => 'Archived', },
                ],
            },
        },
    },
);

###########################
# TODO:
#   * checks for db stat
#   * succes for geo with proper operator

subtest 'Campaigns.Archive - request objects limit' => sub {
    #plan tests => 5;

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ map { $_ } 0 .. $LIMIT ], },
    });
    isa_ok($resp, 'Direct::Defect');
    is( $resp->code, 4001, 'error code' );
    is( $resp->text, 'Неверно заданы параметры фильтрации SelectionCriteria', 'error message' );
    is( $resp->description, "Массив Ids не может содержать более $LIMIT элементов", 'error description' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    done_testing();
};

subtest 'Campaigns.Archive - errors for invalid ids' => sub {
    plan tests => 2;

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ 'abc', '-1' ], },
    });
    is_deeply( $resp, $RESPONSES{invalid_id}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Archive - error for not owned campaign' => sub {
    #plan tests => 2;

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ 201 ], },
    });
    is_deeply( $resp, $RESPONSES{not_owned_campaign}, 'expected response' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    done_testing();
};

subtest 'Campaigns.Archive - errors for not exist or hidden campaigns' => sub {
    #plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin = \&API::Service::Base::check_campaigns_write_detail;
    *API::Service::Base::check_campaigns_write_detail = sub { shift; { map { $_ => 'n' } @_ } };

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ 1000 ], },
    });
    is_deeply( $resp, $RESPONSES{not_exists_camps}, 'expected response' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::check_campaigns_write_detail = $origin;

    done_testing();
};

subtest 'Campaigns.Archive - error for read-only campaign' => sub {
    #plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin = \&API::Service::Base::check_campaigns_write_detail;
    *API::Service::Base::check_campaigns_write_detail = sub { shift; { map { $_ => 'r' } @_ } };

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ 3 ], },
    });
    is_deeply( $resp, $RESPONSES{readonly_campaign}, 'expected response' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::check_campaigns_write_detail = $origin;

    done_testing();
};

subtest 'Campaigns.Archive - error for campaign with not "api_edit" kind' => sub {
    #plan tests => 2;

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ 200, ], },
    });
    is_deeply( $resp, $RESPONSES{bad_kind_campaign}, 'expected response' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    done_testing();
};

=for Включить после того как разрешим работу с гео кампаниями в API5
subtest 'Campaigns.Archive - error for geo campaign and operator, not allowed to edit geo camp with api' => sub {
    #plan tests => 2;

    my $srv = API::Service::Campaigns->new('Campaigns');
    $srv->set_current_operation('archive');
    my $user = API::Authorization::User->new( uid => 2, login => 'not-allowed-to-edit-geo-camp', ClientID => 2 );
    $srv->set_subclient( $user );
    $srv->set_authorization( API::Authorization->fake_new( 2, 'not-allowed-to-edit-geo-camp', { chief_rep_user => $user } ) );

    my $resp = $srv->archive({
        SelectionCriteria => { Ids => [ 2, ], },
    });
    is_deeply( $resp, $RESPONSES{bad_operator}, 'expected response' );

    # TODO: check for opposite case

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    done_testing();
};
=cut

subtest 'Campaigns.Archive - plain text camp' => sub {
    plan tests => 2;

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ 3, ], },
    });
    is_deeply( $resp, $RESPONSES{plain_text_camp}, 'expected response' );

    check_test_dataset( $SAVED_DATA{plain_text_camp}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Archive - warnings for already archived and duplicates' => sub {
    plan tests => 2;

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ 3, 3 ], },
    });
    is_deeply( $resp, $RESPONSES{duplicates_and_already_archived}, 'expected response' );

    check_test_dataset( $SAVED_DATA{duplicates_and_already_archived}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Archive - text camp under wallet' => sub {
    plan tests => 2;

    no warnings qw/ once redefine /;

    my $orig_get_client_NDS = \&Wallet::get_client_NDS;
    my $orig_remove_nds     = \&Wallet::remove_nds;
    my $orig_prepare_transfer_money = \&Wallet::prepare_transfer_money;
    my $orig_process_transfer_money = \&Wallet::process_transfer_money;

    *Wallet::get_client_NDS         = sub { 0.18 };
    *Wallet::remove_nds             = sub { $_[0] * $_[1] };
    *Wallet::prepare_transfer_money = sub { +{} };
    *Wallet::process_transfer_money = sub { undef };

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ 4, ], },
    });
    is_deeply( $resp, $RESPONSES{text_camp_under_wallet}, 'expected response' );

    check_test_dataset( $SAVED_DATA{text_camp_under_wallet}, 'expected db state' );

    *Wallet::get_client_NDS = $orig_get_client_NDS;
    *Wallet::remove_nds     = $orig_remove_nds;
    *Wallet::prepare_transfer_money = $orig_prepare_transfer_money;
    *Wallet::process_transfer_money = $orig_process_transfer_money;

    #done_testing();
};

=for TODO: дописать после того как разрешим работу с гео кампаниями в API5
subtest 'Campaigns.Resume - archive geo campaign' => sub {
    plan tests => 2;

    my $resp = $service->archive({
        SelectionCriteria => { Ids => [ 12, ], },
    });
    is_deeply( $resp, $RESPONSES{archive_geo}, 'expected response' );

    check_test_dataset( $SAVED_DATA{archive_geo}, 'expected db state' );

    #done_testing();
};
=cut

#done_testing();
